# AGENTS.md

This file provides guidance to Codex when working with code in this repository.

## Git Rules

- NEVER commit without explicit user permission.
- Other agents may be working on the codebase concurrently.

## Build & Dev Commands

All commands run from repo root.

IMPORTANT: Always use workspace-specific commands. Never use `npm run build` or `npm run test` for all workspaces unless explicitly required.

### Workspace-Specific Commands (preferred)

```bash
# Dashboard
npm run dashboard:check   # Typecheck + lint in parallel (fast validation)
npm run dashboard:build   # Build dashboard for production
npm run dashboard:test    # Run dashboard tests (vitest)
npm run dashboard:dev     # Start dashboard dev server (port 5173)

# Relay
npm run relay:dev         # Start relay dev server (port 8787)
npm run relay:build       # Build relay for production
npm run relay:test        # Run relay tests (jest)

# Guides
npm run guides:dev        # Start guides dev server (port 5174)
npm run guides:build      # Build guides for production
```

After making changes, run `npm run dashboard:check` when relevant to quickly validate.

### Global Commands (avoid unless needed)

```bash
npm run build             # Build all workspaces
npm run lint              # Lint all workspaces
npm run lint:fix          # Lint and auto-fix all workspaces
npm run test              # Test all workspaces
```

### CLI Tool

```bash
npm run cli                             # Show all commands
npm run cli auth login                  # Store Kalshi credentials
npm run cli auth status                 # Check auth status
npm run cli discover series             # List Kalshi series
npm run cli discover markets <ticker>   # List markets for a series

# Polymarket exploration (alias: poly)
npm run cli polymarket search "Lakers"  # Search markets by text
npm run cli poly series                 # List sports series
npm run cli poly events nba-2026        # List events in a series
npm run cli poly slug <slug>            # Look up specific market
npm run cli poly analyze                # Analyze slug patterns
```

## Package Management

This project uses npm (not pnpm).

```bash
npm install
npm install <package> -w @galactus/dashboard
npm install <package> -w @galactus/relay
```

Use workspace package names, not directory paths.

## Architecture

Monorepo with npm workspaces:

- `apps/dashboard` - React + Vite frontend SPA
- `apps/guides` - React + Vite educational content
- `apps/relay` - Express + WebSocket relay server
- `packages/shared` - Shared TypeScript types

Frontend-backend split:

- Frontend holds API keys and signs requests
- Backend relay is transport-only (CORS + WS forwarding)
- RSA-PSS signing via WebCrypto + node-forge PEM conversion

## Key Files

- `apps/dashboard/src/App.tsx`
- `apps/dashboard/src/lib/kalshiApi.ts`
- `apps/dashboard/src/lib/kalshiAuth.ts`
- `apps/dashboard/src/lib/relayHttp.ts`
- `apps/dashboard/src/lib/relayWs.ts`
- `apps/dashboard/src/lib/marketStream.ts`
- `apps/dashboard/src/lib/positionTracker.ts`
- `apps/dashboard/src/lib/chartManager.ts`
- `apps/dashboard/src/lib/polymarket/`

## UI Component Rules

- NEVER use native `<select>` directly. Use `Dropdown` from `@/components/atoms/Dropdown`.
- NEVER use native checkbox inputs directly. Use `Toggle` from `@/components/atoms/Toggle`.
- For modals, use `Modal` from `@/components/atoms/Modal`.

## Number Formatting Rules

- Use `Money` from `@/components/atoms/Money` for monetary values.
- Do not use `.toFixed(2)` for user-facing monetary formatting.
- For non-money integers, use `.toLocaleString()` or `formatNumber()`.

## Code Organization Rules

- Avoid barrel files (`index.ts`) unless they significantly reduce import clutter.
- Use kebab-case for multi-word folder names.

## File Search Rules

- Always exclude `node_modules` from searches.
- Prefer scoped globs like `apps/**/*.ts` and `packages/**/*.ts`.

## Skills

When writing lessons, guides, tutorials, or educational content, use the `lesson-planning` skill.

### Available local skills

- `lesson-planning`: `/Users/brennan/code/galactus-one/.claude/skills/lesson-planning/SKILL.md`

Usage notes:

- Open the skill file and follow it before drafting lesson content.
- Resolve relative references from the skill directory first.

## Documentation Conventions

- Docs live in `docs/` (except root `README.md` and this file).
- Use lowercase kebab-case markdown filenames.

## Documentation Index

- `README.md` - Quick start
- `docs/deployment.md` - Deployment config/setup
- `docs/devops.md` - DevOps workflows and CI/CD
- `docs/merge-notes.md` - Notes from monorepo merge
