# DOMINO.md

> StreamRift → Domino onboarding contract. This file is for Domino's agents (Spark/Cursor) and StreamRift's agents (Forge/Claude Code). Domino can override anything here.

---

## What This Project Is

Galactus is an internal trading terminal for 10-20 high-frequency sports bettors with $50-100K bankrolls. It connects to Kalshi and Polymarket as equally first-class venues. Desktop-only, dense single-screen layout. **Real money is being traded on production — wrong prices = real losses.**

The stack: React 18 + Vite frontend, Express + WebSocket relay, npm workspaces monorepo.

---

## Your Role

**Domino** = quant + builder. You mold the clay.
**StreamRift** = master engineer. StreamRift polishes what you build into production-grade code.

You have **full autonomy** over the `domino-0` branch. Build freely, experiment, break things — your sandbox (`domino.dev.g1.newbx.com`) is yours to crash and rebuild. StreamRift can restart anything.

Your code doesn't need to be perfect. It needs to **communicate intent**. StreamRift's agents (Forge) will read your commits, your notes, your TODOs, and turn them into shippable code. Think of it as: you sketch the blueprint, StreamRift builds the house.

---

## The Pipeline

```
You push to domino-0
        │
        ▼
domino.dev.g1.newbx.com          ← your sandbox (auto-deploy)
        │
   [StreamRift reviews]
        │
        ▼
streamrift.dev.g1.newbx.com      ← StreamRift polishes
        │
   [push to dev branch]
        │
        ▼
dev.g1.newbx.com                 ← shared staging
        │
   [manual promote]
        │
        ▼
g1.newbx.com → kt.gbuff.com     ← client review → production
```

---

## Git Workflow

### Setup

```bash
# Clone the repo (one-time)
git clone g1:git/g1.1.git galactus
cd galactus
git checkout domino-0

# Install dependencies
npm install
```

### Daily Workflow

```bash
# Start working
git checkout domino-0

# Make changes, test locally
npm run dashboard:dev        # Dev server on localhost:5173
npm run dashboard:check      # Quick typecheck + lint

# When ready to deploy to your sandbox
git add <files>
git commit -m "feat: description of what you did"
git push origin domino-0     # Auto-deploys to domino.dev.g1.newbx.com
```

### Branch Rules

- **Your branch:** `domino-0` — push freely, it's yours
- **Read-only:** `streamrift-0`, `dev`, `main` — you can read/checkout these for context
- The server enforces this — pushing to other branches will be rejected

### Checking Out Stable Versions

StreamRift tags stable versions. To start from a known-good state:

```bash
git tag -l                       # List all tags
git checkout v0.5.0              # Checkout specific version
git checkout stable-post-refactor # Checkout by human-readable name
git checkout domino-0            # Back to your branch
```

---

## Session Commands

### Starting a Session

Type `/domino-run-it` in your editor's AI assistant (Cursor/Claude Code) to boot up. This:
- Loads project context
- Checks git status and branch
- Runs a quick health check
- Says "Ready. What are we building?"

Or just start working — the command is optional.

### Ending a Session

Type `/domino-wrap-it` when you're done. This is your **save button**. It:
1. Summarizes what you changed
2. Scans for TODO(domino) markers you left
3. Writes a session summary to `docs/domino/`
4. Writes a handoff manifest for StreamRift's agents
5. Commits everything
6. Pushes to domino-0
7. Reports: "Push complete. Check domino.dev.g1.newbx.com/status.html"

### First Time Setup

Type `/get-started` on your first session. This interviews you about your preferences and configures Spark (your agent) to work the way you like.

---

## Commit Conventions

These are **suggestions, not rules**. Descriptive commits help StreamRift's agents understand what you were doing.

### Prefix Guide

| Prefix | When to use |
|--------|-------------|
| `feat:` | New feature or capability |
| `fix:` | Bug fix |
| `refactor:` | Code restructure (no behavior change) |
| `test:` | Adding or updating tests |
| `docs:` | Documentation changes |
| `wip:` | Work in progress (incomplete, pushing to save state) |

### Examples

```
feat: add MLB market discovery
fix: odds display showing NaN for tennis markets
wip: experimenting with batch order execution (not working yet)
refactor: extract pricing logic into separate module
```

**The commit message body is valuable.** If the change isn't obvious, add a line or two explaining what you were trying to do and why. StreamRift's agents will read this.

---

## Scribe Protocol

Your AI agents (Spark) should document your work as you go. This is how StreamRift's agents (Forge) understand your intent later.

### In-Code Markers

Leave breadcrumbs in the code:

```typescript
// TODO(domino): finish the error handling for timeout case
// TODO(domino): this pricing formula needs review — using simplified model
// TODO(domino): wire up to the real API, currently using mock data
// NOTE(domino): I chose this approach because [reason]
```

These are grepable — StreamRift's agents search for `TODO(domino)` to find your incomplete work.

### Session Summaries

After each session, Spark should write to `docs/domino/session-YYYY-MM-DD.md`:

```markdown
# Session: 2026-02-20

## What I worked on
- Started MLB market discovery
- Fixed odds display bug for tennis

## What's incomplete
- MLB discovery finds markets but grouping is wrong
- See TODO(domino) in sportsDiscovery/mlb.ts:45

## Notes for StreamRift
- I wasn't sure about the market key format for MLB — used same pattern as NBA
- The pricing formula in line 89 is a rough draft, needs quant review
```

### Overhead Target

Scribe work should be **10-30% of your context window**. Don't let documentation eat your working memory. Quick notes > thorough essays.

---

## Handoff Format

When `/domino-wrap-it` pushes, it generates `docs/domino/handoff-YYYY-MM-DD.md`:

```markdown
# Handoff: 2026-02-20 — MLB discovery + tennis odds fix

## Changes (scan fast)
- feat: MLB market discovery — files: sportsDiscovery/mlb.ts, stream.ts
- fix: tennis odds NaN — files: components/OddsCell.tsx

## Intent (understand why)
Building MLB support per F-016 from backlog. Tennis fix was blocking
my testing — odds showed NaN when spread was undefined.

## Incomplete (know where to dig)
- [ ] MLB grouping logic — see TODO(domino) in mlb.ts:45
- [ ] MLB needs rotation number mapping — didn't find the data source

## Notes for StreamRift
Market key format might conflict with existing NBA pattern. I used
KMLB-{date}-{away}v{home} but check if that matches Kalshi's actual tickers.
```

**Optimized for scanning:** StreamRift's agents need to scan wide within limited context windows, then know where to go deep.

---

## Soft Guardrails

These help your agents avoid footguns. **You can override any of these.**

### SHOULD avoid modifying
- `CLAUDE.md` — StreamRift's agent config
- `.hand/` — StreamRift's operational layer
- `scripts/deploy-*.sh` — deploy infrastructure
- Production config files (`.env`, `.staging-env`)

### SHOULD avoid deleting
- Existing tests (they protect real money flows)
- Files in `docs/` that aren't yours

### SHOULD do
- Write descriptive commit messages
- Leave `TODO(domino)` markers on incomplete work
- Run `/domino-wrap-it` before ending a session
- Test locally before pushing (`npm run dashboard:check`)

### CAN always do
- Override any guideline above if you have a reason
- Touch any code file — there are no hard restrictions
- Experiment freely on domino-0
- Push incomplete work (use `wip:` prefix)

---

## Build & Dev Commands

Run from repo root. **Always use workspace-specific commands.**

```bash
# Development
npm run dashboard:dev       # Start dev server (port 5173)
npm run dashboard:check     # Typecheck + lint (FAST validation)
npm run dashboard:test      # Run tests
npm run dashboard:build     # Production build

# Relay (if you need it)
npm run relay:dev           # Start relay (port 8787)

# Full stack
npm run dev                 # Dashboard + relay together
```

### Before pushing

```bash
npm run dashboard:check     # Should pass clean
```

This isn't enforced — your push will go through either way. But broken typechecks make StreamRift's integration harder.

---

## Architecture Quick Reference

### Key Files

| File | What it does |
|------|-------------|
| `apps/dashboard/src/App.tsx` | Root component, orchestrates everything |
| `apps/dashboard/src/lib/kalshiApi.ts` | Kalshi REST API client |
| `apps/dashboard/src/lib/marketStream.ts` | WebSocket market data |
| `apps/dashboard/src/lib/positionTracker.ts` | P&L calculation |
| `apps/dashboard/src/lib/polymarket/` | Polymarket integration |
| `apps/dashboard/src/components/atoms/` | Reusable UI components |

### Data Flow

```
sportsDiscovery/ → discovers markets from APIs
        ↓
sportsStream/ → subscribes to real-time price updates via WebSocket
        ↓
Table components → display odds, volumes, liquidity
        ↓
GameDetailPanel → order books, order entry (right side panel)
```

### Critical Rules

- **Never use native `<select>`** → use `<Dropdown>` from atoms
- **Never use `.toFixed(2)` for money** → use `<Money>` component
- **Frontend holds API keys** — they never cross the wire to the relay
- **Relay is transport-only** — CORS proxy, no business logic

---

## Escalation — Phone a Friend

When you're stuck:

1. **Push what you have** — even if broken, with a `wip:` commit describing the problem
2. **Leave notes** — `TODO(domino): STUCK — [describe what's wrong]`
3. **Message StreamRift** — via Telegram, text, or the code exchange
4. **StreamRift or Forge will investigate** from your branch

You're never stuck alone. Push the state, describe the problem, move on or wait.

---

## Deploy Status

After pushing, check your deploy at:
- **Your sandbox:** `https://domino.dev.g1.newbx.com/`
- **Deploy status:** `https://domino.dev.g1.newbx.com/status.html`

The status page shows build status, last commit, deploy time, and pipeline position.

---

## The Agent Ecosystem

```
┌─────────────────────────────────────────────────────┐
│  Domino's World                                     │
│  ┌──────────┐    ┌──────────┐    ┌───────────────┐  │
│  │  Domino   │───▶│  Spark   │───▶│  domino-0     │  │
│  │  (human)  │    │ (Cursor) │    │  (git branch) │  │
│  └──────────┘    └──────────┘    └──────┬────────┘  │
└─────────────────────────────────────────┼───────────┘
                                          │ push
                                          ▼
                              domino.dev.g1.newbx.com
                                          │
                                     [review]
                                          │
┌─────────────────────────────────────────┼───────────┐
│  StreamRift's World                     │           │
│  ┌──────────┐    ┌──────────┐    ┌──────▼────────┐  │
│  │StreamRift │───▶│  Forge   │───▶│ streamrift-0  │  │
│  │  (human)  │    │ (Claude) │    │  (git branch) │  │
│  └──────────┘    └──────────┘    └───────────────┘  │
└─────────────────────────────────────────────────────┘
```

**DominoBots write → StreamRiftBots read → Forge integrates → StreamRift ships.**
