/**
 * Main App Component
 *
 * Two states:
 * - Non-authenticated: Shows login screen
 * - Authenticated: Shows app shell with sidebar navigation
 */

import { useState, useRef, useCallback, useMemo, lazy, Suspense } from 'react';
import { useNavigate, useLocation, useParams } from 'react-router-dom';
import type { NBAMarketRow } from './types';
import type { AppView } from './components/layout';

// Hooks
import { useDisplayPreferences } from './hooks/useDisplayPreferences';
import { useKalshiConnection } from './hooks/useKalshiConnection';
import { usePositionTracking } from './hooks/usePositionTracking';
import { useStreamManagement } from './hooks/useStreamManagement';
import { useOrderManagement } from './hooks/useOrderManagement';
import { usePolymarketIntegration } from './hooks/usePolymarketIntegration';

// Layout components
import { AppShell, AppSidebar, AppHeader, AppStatusBar } from './components/layout';
import type { ApiStatus } from './components/layout/AppHeader';
import { getOddsApiKey } from './lib/oddsApi';

// Critical-path page components (loaded eagerly)
import {
  LoginView,
  ExploreView,
  NBAView,
  ValueDashboardView,
  PositionsView,
  OrdersView,
  GateView,
} from './components/pages';

// Non-critical page components (lazy-loaded, code-split into separate chunks)
const DevExplorerView = lazy(() =>
  import('./components/pages/DevExplorerView').then((m) => ({ default: m.DevExplorerView }))
);
const DiagnosticsView = lazy(() =>
  import('./components/pages/DiagnosticsView').then((m) => ({ default: m.DiagnosticsView }))
);
const SettingsView = lazy(() =>
  import('./components/pages/SettingsView').then((m) => ({ default: m.SettingsView }))
);
const MarketDetailView = lazy(() =>
  import('./components/pages/MarketDetailView').then((m) => ({ default: m.MarketDetailView }))
);

// Expected hash from environment variable for dashboard gate
const EXPECTED_HASH = import.meta.env.VITE_DASHBOARD_PASSWORD_HASH || '';

// Legacy components (still needed)
import OrderTicket from './components/OrderTicket';
import FillNotificationComponent from './components/FillNotification';
import { ToastProvider } from './components/organisms/Toast';
import { ConfirmModal } from './components/atoms/Modal';
import { useHotkeys } from './lib/hotkeys';

// View titles for header
const viewTitles: Record<AppView | 'market', string> = {
  explore: 'Explore Markets',
  nba: 'NBA Markets',
  'value-dashboard': 'Value Dashboard',
  positions: 'Positions',
  orders: 'Orders',
  'dev-explorer': 'Sports Inventory',
  diagnostics: 'Diagnostics',
  settings: 'Settings',
  market: 'Market Detail',
};

// Valid views for URL routing
const validViews: AppView[] = [
  'explore',
  'nba',
  'value-dashboard',
  'positions',
  'orders',
  'dev-explorer',
  'diagnostics',
  'settings',
];

function App() {
  // Routing
  const navigate = useNavigate();
  const location = useLocation();
  const params = useParams();

  // Gate state
  const [isGateUnlocked, setIsGateUnlocked] = useState(() => {
    const unlocked = sessionStorage.getItem('gateUnlocked') === 'true';
    const storedHash = sessionStorage.getItem('gateHash');
    return unlocked && storedHash === EXPECTED_HASH && EXPECTED_HASH !== '';
  });

  const handleGateUnlock = useCallback((verifiedHash: string) => {
    sessionStorage.setItem('gateUnlocked', 'true');
    sessionStorage.setItem('gateHash', verifiedHash);
    setIsGateUnlocked(true);
  }, []);

  // Shared ref (needed by both streams and positions)
  const marketsDataRef = useRef<Map<string, NBAMarketRow>>(new Map());

  // === HOOKS ===

  const display = useDisplayPreferences();

  const connection = useKalshiConnection({ navigate, location });

  const positionTracking = usePositionTracking({
    isConnected: connection.isConnected,
    apiClientRef: connection.apiClientRef,
    credentials: connection.credentials,
    marketsDataRef,
    setErrorMessage: connection.setErrorMessage,
    portfolioPollingEnabled: connection.portfolioPollingEnabled,
    setPortfolioPollingEnabled: connection.setPortfolioPollingEnabled,
  });

  const streams = useStreamManagement({
    isConnected: connection.isConnected,
    apiClientRef: connection.apiClientRef,
    credentials: connection.credentials,
    dataMode: display.dataMode,
    chartPeriod: display.chartPeriod,
    positionTrackerRef: positionTracking.positionTrackerRef,
    refreshPositions: positionTracking.refreshPositions,
    marketsDataRef,
    setIsConnected: connection.setIsConnected,
    setConnectionStatus: connection.setConnectionStatus,
  });

  const orderMgmt = useOrderManagement({
    isConnected: connection.isConnected,
    apiClientRef: connection.apiClientRef,
    credentials: connection.credentials,
    positionTrackerRef: positionTracking.positionTrackerRef,
    marketsDataRef,
    refreshPositions: positionTracking.refreshPositions,
    loadPositions: positionTracking.loadPositions,
    setErrorMessage: connection.setErrorMessage,
    portfolioPollingEnabled: connection.portfolioPollingEnabled,
    setPortfolioPollingEnabled: connection.setPortfolioPollingEnabled,
  });

  const poly = usePolymarketIntegration({
    isConnected: connection.isConnected,
    consolidatedGames: streams.consolidatedGames,
    setErrorMessage: connection.setErrorMessage,
  });

  // Composed disconnect: clean up all hooks then disconnect
  const handleDisconnect = useCallback(() => {
    streams.cleanupStreams();
    positionTracking.cleanupPositions();
    orderMgmt.cleanupOrders();
    poly.cleanupPoly();
    connection.disconnect();
  }, [streams, positionTracking, orderMgmt, poly, connection]);

  // Composed reconnect: full disconnect then reconnect
  const handleKalshiReconnect = useCallback(
    async (args: {
      id?: string;
      name?: string;
      accessKey: string;
      privateKey: string;
      remember: boolean;
    }) => {
      handleDisconnect();
      await connection.handleKalshiReconnect(args);
    },
    [handleDisconnect, connection]
  );

  // Keyboard shortcuts
  useHotkeys({
    'new-order': orderMgmt.openOrderTicket,
    'kill-all': () => orderMgmt.setShowKillAllConfirm(true),
  });

  // Derive current view from URL
  const isMarketRoute = location.pathname.startsWith('/app/market/');
  const isExploreRoute =
    location.pathname.startsWith('/app/explore') || location.pathname === '/app';

  const currentView: AppView | 'market' = isMarketRoute
    ? 'market'
    : isExploreRoute
      ? 'explore'
      : validViews.includes(params.view as AppView)
        ? (params.view as AppView)
        : 'explore';

  const exploreParams = {
    category: params.category,
    series: params.series,
    event: params.event,
  };

  const marketTicker = params.ticker;

  // API connection status for header indicators
  const apiStatus = useMemo<ApiStatus>(
    () => ({
      kalshi: connection.isConnected ? 'connected' : 'disconnected',
      feed: streams.feedStatus,
      poly:
        poly.polyStatus === 'connected'
          ? 'connected'
          : poly.polyStatus === 'error'
            ? 'error'
            : 'disconnected',
      odds: getOddsApiKey() ? 'configured' : 'none',
    }),
    [connection.isConnected, streams.feedStatus, poly.polyStatus]
  );

  // Render current view
  const renderView = () => {
    if (!connection.apiClientRef.current) return null;

    if (currentView === 'market' && marketTicker) {
      return (
        <MarketDetailView
          marketTicker={marketTicker}
          apiClient={connection.apiClientRef.current}
          credentials={connection.credentials ?? undefined}
        />
      );
    }

    if (currentView === 'explore') {
      return (
        <ExploreView
          apiClient={connection.apiClientRef.current}
          credentials={connection.credentials ?? undefined}
          params={exploreParams}
        />
      );
    }

    const viewMap: Record<Exclude<AppView, 'explore'>, React.ReactNode> = {
      nba: (
        <NBAView
          markets={streams.markets}
          chartPeriod={display.chartPeriod}
          onMarketClick={orderMgmt.openOrderTicket}
          onChartPeriodChange={display.handleChartPeriodChange}
        />
      ),
      'value-dashboard': (
        <ValueDashboardView
          markets={streams.markets}
          orders={orderMgmt.orders}
          positions={positionTracking.positions}
          consolidatedGames={streams.consolidatedGames}
          consolidatedLoading={streams.consolidatedLoading}
          feedStatus={streams.feedStatus}
          lastUpdateTime={streams.lastUpdateTime}
          polyOpenOrders={poly.polyOpenOrders}
          polyStatus={poly.polyStatus}
          positionTitleByTicker={positionTracking.positionTitleByTicker}
          positionGameDateByTicker={positionTracking.positionGameDateByTicker}
          positionOutcomeByTicker={positionTracking.positionOutcomeByTicker}
          onPlaceKalshiOrder={orderMgmt.handlePlaceKalshiDashboardOrder}
          onPlacePolymarketOrder={poly.handlePlacePolymarketDashboardOrder}
          onCancelOrder={orderMgmt.handleCancelOrder}
          onDecreaseOrder={orderMgmt.handleDecreaseOrder}
          onKillAll={orderMgmt.handleKillAll}
          onCancelPolyOrder={poly.handleCancelPolyOrder}
          onKillAllPolyOrders={poly.handleKillAllPolyOrders}
          sportsGames={streams.sportsGames}
          sportsLoading={streams.sportsLoading}
          sportsLoadingStates={streams.sportsLoadingStates}
          onSportMarketChange={streams.handleSportMarketChange}
        />
      ),
      positions: (
        <PositionsView
          positions={positionTracking.positions}
          totalPnl={positionTracking.totalPnl}
          markets={streams.markets}
          onClosePosition={positionTracking.handleClosePosition}
          onNewOrder={orderMgmt.openOrderTicket}
        />
      ),
      orders: (
        <OrdersView
          orders={orderMgmt.orders}
          onNewOrder={orderMgmt.openOrderTicket}
          onKillAll={orderMgmt.handleKillAll}
          onCancelOrder={orderMgmt.handleCancelOrder}
        />
      ),
      'dev-explorer': (
        <DevExplorerView
          apiClientRef={connection.apiClientRef}
          credentials={connection.credentials}
          onPlaceOrder={orderMgmt.openOrderTicket}
        />
      ),
      diagnostics: (
        <DiagnosticsView
          apiClientRef={connection.apiClientRef}
          credentials={connection.credentials}
          consolidatedGames={streams.consolidatedGames}
          sportsGames={streams.sportsGames}
          feedStatus={streams.feedStatus}
          lastUpdateTime={streams.lastUpdateTime}
        />
      ),
      settings: (
        <SettingsView
          pov={display.pov}
          oddsFormat={display.oddsFormat}
          dataMode={display.dataMode}
          onPOVToggle={display.handlePOVToggle}
          onOddsFormatToggle={display.handleOddsFormatToggle}
          onDataModeToggle={display.handleDataModeToggle}
          connectionStatus={connection.connectionStatus}
          environment={connection.environment}
          kalshiProfiles={connection.kalshiStore.profiles}
          activeKalshiProfileId={connection.kalshiStore.activeProfileId}
          onKalshiSetActive={connection.handleKalshiSetActive}
          kalshiCredentials={connection.credentials}
          onKalshiReconnect={handleKalshiReconnect}
          onKalshiDeleteProfile={connection.handleKalshiDeleteProfile}
          onKalshiSaveProfile={connection.handleKalshiSaveProfile}
          polyConfig={poly.polyConfig}
          polyStatus={poly.polyStatus}
          onPolyConnect={poly.handlePolyConnect}
          onPolyDisconnect={poly.handlePolyDisconnect}
        />
      ),
    };

    return viewMap[currentView as Exclude<AppView, 'explore'>];
  };

  // Auto-connecting: Show loading state
  if (connection.isAutoConnecting) {
    return (
      <div className="bg-background flex min-h-screen flex-col items-center justify-center p-4">
        <div className="text-center">
          <div className="border-primary mx-auto mb-4 h-8 w-8 animate-spin rounded-full border-b-2" />
          <h2 className="text-foreground text-lg font-medium">Reconnecting...</h2>
          <p className="text-muted-foreground mt-1 text-sm">Using saved credentials</p>
        </div>
      </div>
    );
  }

  // Gate not unlocked: Show password gate
  if (!isGateUnlocked) {
    return <GateView onUnlock={handleGateUnlock} />;
  }

  // Non-authenticated: Show login
  if (!connection.isConnected) {
    return (
      <LoginView
        onConnect={connection.handleConnect}
        error={connection.errorMessage}
      />
    );
  }

  // For sidebar, map 'market' to 'explore'
  const sidebarView: AppView = currentView === 'market' ? 'explore' : currentView;

  // Authenticated: Show app shell
  return (
    <ToastProvider>
      <AppShell
        sidebar={
          <AppSidebar
            currentView={sidebarView}
            onViewChange={(view) => navigate(`/app/${view}`)}
            onDisconnect={handleDisconnect}
          />
        }
        header={
          <AppHeader
            title={viewTitles[currentView]}
            feedStatus={currentView === 'value-dashboard' ? streams.feedStatus : undefined}
            lastUpdate={currentView === 'value-dashboard' ? streams.lastUpdateTime : undefined}
            apiStatus={apiStatus}
          />
        }
        statusBar={
          <AppStatusBar
            status={connection.connectionStatus}
            error={connection.errorMessage}
          />
        }
      >
        <Suspense
          fallback={
            <div className="flex h-full items-center justify-center">
              <div className="border-primary h-6 w-6 animate-spin rounded-full border-b-2" />
            </div>
          }
        >
          {renderView()}
        </Suspense>
      </AppShell>

      {/* Order Ticket Modal */}
      {orderMgmt.showOrderTicket && (
        <OrderTicket
          markets={streams.markets}
          marketsData={marketsDataRef.current}
          defaultMarket={orderMgmt.orderTicketMarket}
          defaultPrice={orderMgmt.orderTicketPrice}
          apiClient={connection.apiClientRef.current}
          onClose={orderMgmt.closeOrderTicket}
          onOrderPlaced={orderMgmt.handleOrderPlaced}
        />
      )}

      {/* Kill All Orders Confirm Modal */}
      <ConfirmModal
        isOpen={orderMgmt.showKillAllConfirm}
        onClose={() => orderMgmt.setShowKillAllConfirm(false)}
        onConfirm={orderMgmt.handleKillAll}
        title="Kill All Orders"
        message="Are you sure you want to kill all open orders? This action cannot be undone."
        confirmLabel="Kill All"
        cancelLabel="Keep Orders"
        variant="danger"
      />

      {/* Fill Notifications */}
      {orderMgmt.fillNotifications.map((fill) => (
        <FillNotificationComponent
          key={fill.orderId}
          fill={fill}
        />
      ))}
    </ToastProvider>
  );
}

export default App;
