/**
 * Credentials Panel Component
 */

import React, { useState, useEffect } from 'react';
import { ensureKalshiProfilesStore, getActiveKalshiProfile } from '@/lib/kalshiProfiles';

interface CredentialsPanelProps {
  onConnect: (
    accessKey: string,
    privateKey: string,
    environment: 'prod' | 'demo',
    useRelay: boolean,
    remember: boolean
  ) => Promise<void>;
  onDisconnect: () => void;
  isConnected: boolean;
}

const CredentialsPanel: React.FC<CredentialsPanelProps> = ({
  onConnect,
  onDisconnect,
  isConnected,
}) => {
  const [accessKey, setAccessKey] = useState('');
  const [privateKey, setPrivateKey] = useState('');
  const [environment, setEnvironment] = useState<'prod' | 'demo'>('prod');
  const [useRelay, setUseRelay] = useState(true);
  const [remember, setRemember] = useState(true);
  const [isConnecting, setIsConnecting] = useState(false);

  // Load saved credentials
  useEffect(() => {
    try {
      const store = ensureKalshiProfilesStore();
      const active = getActiveKalshiProfile(store);
      if (active) {
        setAccessKey(active.accessKey || '');
        setPrivateKey(active.privateKey || '');
        // Locked: always use relay and prod regardless of saved value
        setUseRelay(true);
        setEnvironment('prod');
        // Default "remember" to true if credentials exist (keeps them persisted)
        setRemember(true);
      }
    } catch (error) {
      console.error('Failed to load saved credentials:', error);
    }
  }, []);

  const handleConnect = async () => {
    console.log('CredentialsPanel: handleConnect called', {
      hasAccessKey: !!accessKey,
      hasPrivateKey: !!privateKey,
      useRelay,
    });

    if (!accessKey || !privateKey) {
      console.warn('CredentialsPanel: Missing credentials');
      alert('Please enter both Access Key ID and Private Key');
      return;
    }

    /* // Direct browser calls to Kalshi are typically blocked by CORS.
    // Nudge users toward the relay unless they explicitly want to try direct.
    let effectiveUseRelay = useRelay;
    if (!useRelay) {
      const switchToRelay = window.confirm(
        '“Connect Directly” is usually blocked by browser CORS for the Kalshi API.\n\nSwitch to “Connect Through Server” (recommended)?'
      );
      if (switchToRelay) {
        effectiveUseRelay = true;
        setUseRelay(true);
      }
    } */
    // Locked to relay (useRelay=true) and prod — no user choice needed.

    setIsConnecting(true);
    try {
      console.log('CredentialsPanel: Calling onConnect...');
      await onConnect(accessKey, privateKey, environment, useRelay, remember);
      console.log('CredentialsPanel: onConnect completed successfully');
    } catch (error) {
      console.error('Connection failed in CredentialsPanel:', error);
      // Error is already handled by App.tsx and displayed in StatusBar
      // Don't re-throw - App.tsx handles errors by setting state, not throwing
    } finally {
      setIsConnecting(false);
    }
  };

  return (
    <div
      id="credentials-panel"
      className="panel"
    >
      <h2>Credentials</h2>
      <div className="form-group">
        <label htmlFor="access-key">Access Key ID:</label>
        <input
          type="text"
          id="access-key"
          value={accessKey}
          onChange={(e) => setAccessKey(e.target.value)}
          placeholder="Enter access key ID"
          disabled={isConnected}
        />
      </div>
      <div className="form-group">
        <label htmlFor="private-key">Private Key PEM:</label>
        <textarea
          id="private-key"
          rows={6}
          value={privateKey}
          onChange={(e) => setPrivateKey(e.target.value)}
          placeholder="-----BEGIN PRIVATE KEY-----&#10;...&#10;-----END PRIVATE KEY-----"
          disabled={isConnected}
        />
      </div>
      {/* Environment selection — locked to prod
      <div className="form-group">
        <label htmlFor="environment">Environment:</label>
        <div style={{ display: 'flex', gap: '10px', alignItems: 'center', marginTop: '5px' }}>
          <label style={{ display: 'flex', alignItems: 'center', cursor: 'pointer' }}>
            <input
              type="radio"
              name="environment"
              checked={environment === 'demo'}
              onChange={() => setEnvironment('demo')}
              disabled={isConnected}
              style={{ marginRight: '5px' }}
            />
            Demo (recommended for dev)
          </label>
          <label style={{ display: 'flex', alignItems: 'center', cursor: 'pointer' }}>
            <input
              type="radio"
              name="environment"
              checked={environment === 'prod'}
              onChange={() => setEnvironment('prod')}
              disabled={isConnected}
              style={{ marginRight: '5px' }}
            />
            Prod
          </label>
        </div>
        <small style={{ display: 'block', marginTop: '5px', color: '#666' }}>
          {environment === 'demo'
            ? 'Uses demo-api.kalshi.co (demo keys only)'
            : 'Uses api.elections.kalshi.com (prod keys only)'}
        </small>
      </div>
      */}

      {/* Connection mode selection — locked to relay (Connect Through Server)
      <div className="form-group">
        <label htmlFor="connection-mode">Connection Mode:</label>
        <div style={{ display: 'flex', gap: '10px', alignItems: 'center', marginTop: '5px' }}>
          <label style={{ display: 'flex', alignItems: 'center', cursor: 'pointer' }}>
            <input
              type="radio"
              name="connection-mode"
              checked={!useRelay}
              onChange={() => setUseRelay(false)}
              disabled={isConnected}
              style={{ marginRight: '5px' }}
            />
            Connect Directly
          </label>
          <label style={{ display: 'flex', alignItems: 'center', cursor: 'pointer' }}>
            <input
              type="radio"
              name="connection-mode"
              checked={useRelay}
              onChange={() => setUseRelay(true)}
              disabled={isConnected}
              style={{ marginRight: '5px' }}
            />
            Connect Through Server
          </label>
        </div>
        <small style={{ display: 'block', marginTop: '5px', color: '#666' }}>
          {useRelay
            ? 'Requests forwarded through relay server (localhost:8787)'
            : 'Direct connection to Kalshi API (may be blocked by CORS)'}
        </small>
      </div>
      */}
      <div className="form-group">
        <label>
          <input
            type="checkbox"
            id="remember-creds"
            checked={remember}
            onChange={(e) => setRemember(e.target.checked)}
            disabled={isConnected}
          />
          Remember credentials
        </label>
      </div>
      <div className="form-group">
        <button
          id="connect-btn"
          className="btn-primary"
          onClick={(e) => {
            console.log('Button clicked!', e);
            e.preventDefault();
            handleConnect().catch((err) => {
              console.error('Unhandled error in handleConnect:', err);
            });
          }}
          disabled={isConnected || isConnecting || !accessKey || !privateKey}
        >
          {isConnecting ? 'Connecting...' : 'Connect'}
        </button>
        <button
          id="disconnect-btn"
          className="btn-secondary"
          onClick={onDisconnect}
          disabled={!isConnected}
        >
          Disconnect
        </button>
      </div>
    </div>
  );
};

export default CredentialsPanel;
