/**
 * Order Ticket Component
 */

import React, { useState, useEffect, useMemo, useRef } from 'react';
import type { NBAMarketRow } from '../types';
import type { KalshiApiClient } from '../lib/kalshiApi';
import { getPOV, getBuyPrice, getSellPrice } from '../lib/pricingPOV';

interface OrderTicketProps {
  markets: NBAMarketRow[];
  marketsData: Map<string, NBAMarketRow>;
  defaultMarket?: string;
  defaultPrice?: number;
  apiClient: KalshiApiClient | null;
  onClose: () => void;
  onOrderPlaced: () => void;
}

const OrderTicket: React.FC<OrderTicketProps> = ({
  markets,
  marketsData,
  defaultMarket,
  defaultPrice,
  apiClient,
  onClose,
  onOrderPlaced,
}) => {
  const [marketTicker, setMarketTicker] = useState<string>(defaultMarket || '');
  const [side, setSide] = useState<'yes' | 'no'>('yes');
  const [action, setAction] = useState<'buy' | 'sell'>('buy');
  const [type, setType] = useState<'limit' | 'market'>('limit');
  const [price, setPrice] = useState<string>(defaultPrice?.toFixed(4) || '');
  const [quantity, setQuantity] = useState<string>('100');
  const [isPlacing, setIsPlacing] = useState(false);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const suggestionsRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);

  // Filter markets matching the typed ticker
  const filteredSuggestions = useMemo(() => {
    if (!marketTicker) return markets;
    const query = marketTicker.toUpperCase();
    return markets.filter(
      (m) =>
        m.market_ticker.toUpperCase().includes(query) ||
        `${m.awayTeam} @ ${m.homeTeam}`.toUpperCase().includes(query)
    );
  }, [marketTicker, markets]);

  // Close suggestions on outside click
  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      if (
        suggestionsRef.current &&
        !suggestionsRef.current.contains(e.target as Node) &&
        inputRef.current &&
        !inputRef.current.contains(e.target as Node)
      ) {
        setShowSuggestions(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  // Update price when market changes
  useEffect(() => {
    if (marketTicker && !price) {
      const market = marketsData.get(marketTicker);
      if (market) {
        const midPrice = market.awayProb || market.homeProb;
        if (midPrice) {
          setPrice(midPrice.toFixed(4));
        }
      }
    }
  }, [marketTicker, marketsData, price]);

  // Calculate preview
  const preview = useMemo(() => {
    if (!marketTicker) {
      return { text: 'Select a market', cost: '' };
    }

    const market = marketsData.get(marketTicker);
    const marketName = market ? `${market.awayTeam} @ ${market.homeTeam}` : marketTicker;

    const sideText = side.toUpperCase();
    const actionText = action.toUpperCase();
    const typeText = type === 'market' ? 'MARKET' : 'LIMIT';
    const priceText = type === 'market' ? 'MARKET' : price || 'N/A';

    const text = `${actionText} ${quantity} ${sideText} @ ${priceText} (${typeText})\n${marketName}`;
    const cost =
      type === 'limit' && price
        ? `Est. Cost: $${(parseInt(quantity) * parseFloat(price)).toFixed(2)}`
        : 'Market order - price TBD';

    return { text, cost };
  }, [marketTicker, side, action, type, price, quantity, marketsData]);

  // Handle quick price buttons
  const handleQuickPrice = (priceType: 'bid' | 'ask' | 'mid') => {
    if (!marketTicker) return;

    const market = marketsData.get(marketTicker);
    if (!market) return;

    const pov = getPOV();
    let newPrice: number | null = null;

    if (side === 'yes') {
      if (priceType === 'bid') {
        newPrice =
          pov === 'maker'
            ? getBuyPrice(market.awayYesBid ?? null, market.awayYesAsk ?? null)
            : (market.awayYesBid ?? null);
      } else if (priceType === 'ask') {
        newPrice =
          pov === 'maker'
            ? getSellPrice(market.awayYesBid ?? null, market.awayYesAsk ?? null)
            : (market.awayYesAsk ?? null);
      } else if (priceType === 'mid') {
        if (market.awayYesBid != null && market.awayYesAsk != null) {
          newPrice = (market.awayYesBid + market.awayYesAsk) / 2;
        } else if (market.awayProb !== null) {
          newPrice = market.awayProb;
        }
      }
    } else {
      // NO side
      if (market.awayYesBid != null && market.awayYesAsk != null) {
        if (priceType === 'bid') {
          newPrice = 1 - market.awayYesAsk;
        } else if (priceType === 'ask') {
          newPrice = 1 - market.awayYesBid;
        } else if (priceType === 'mid') {
          newPrice = 1 - (market.awayYesBid + market.awayYesAsk) / 2;
        }
      }
    }

    if (newPrice !== null) {
      setPrice(newPrice.toFixed(4));
    }
  };

  // Handle place order
  const handlePlaceOrder = async () => {
    if (!apiClient) {
      alert('Not connected to API');
      return;
    }

    if (!marketTicker) {
      alert('Please select a market');
      return;
    }

    if (type === 'limit' && !price) {
      alert('Please enter a price for limit orders');
      return;
    }

    const qty = parseInt(quantity, 10);
    if (isNaN(qty) || qty <= 0) {
      alert('Quantity must be greater than 0');
      return;
    }

    setIsPlacing(true);
    try {
      await apiClient.placeOrder({
        ticker: marketTicker,
        side,
        action,
        type,
        count: qty,
        price: type === 'limit' ? parseFloat(price) : undefined,
      });

      onOrderPlaced();
      onClose();
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Failed to place order';
      alert(`Error: ${message}`);
    } finally {
      setIsPlacing(false);
    }
  };

  // Handle keyboard shortcuts
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.key === 'Escape') {
        onClose();
      } else if (e.key === 'Enter' && !e.shiftKey && !e.ctrlKey) {
        e.preventDefault();
        handlePlaceOrder();
      }
    };

    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, [marketTicker, side, action, type, price, quantity, apiClient]);

  return (
    <div
      className="modal"
      style={{ display: 'flex' }}
      onClick={(e) => e.target === e.currentTarget && onClose()}
    >
      <div
        className="modal-content"
        onClick={(e) => e.stopPropagation()}
      >
        <div className="modal-header">
          <h2>PLACE ORDER</h2>
          <button
            className="modal-close"
            onClick={onClose}
          >
            &times;
          </button>
        </div>
        <div className="order-ticket-body">
          <div
            className="form-group"
            style={{ position: 'relative' }}
          >
            <label htmlFor="order-market">Market:</label>
            <input
              ref={inputRef}
              type="text"
              id="order-market"
              className="order-input"
              value={marketTicker}
              onChange={(e) => {
                setMarketTicker(e.target.value);
                setShowSuggestions(true);
              }}
              onFocus={() => setShowSuggestions(true)}
              placeholder="Type ticker or search markets..."
              autoComplete="off"
            />
            {showSuggestions && filteredSuggestions.length > 0 && (
              <div
                ref={suggestionsRef}
                style={{
                  position: 'absolute',
                  top: '100%',
                  left: 0,
                  right: 0,
                  maxHeight: '160px',
                  overflowY: 'auto',
                  zIndex: 50,
                  border: '1px solid var(--border)',
                  borderRadius: '4px',
                  background: 'var(--card)',
                }}
              >
                {filteredSuggestions.map((market) => (
                  <button
                    key={market.market_ticker}
                    type="button"
                    style={{
                      display: 'block',
                      width: '100%',
                      textAlign: 'left',
                      padding: '6px 10px',
                      border: 'none',
                      background: 'transparent',
                      color: 'var(--foreground)',
                      cursor: 'pointer',
                      fontSize: '12px',
                    }}
                    onMouseEnter={(e) => (e.currentTarget.style.background = 'var(--muted)')}
                    onMouseLeave={(e) => (e.currentTarget.style.background = 'transparent')}
                    onClick={() => {
                      setMarketTicker(market.market_ticker);
                      setShowSuggestions(false);
                    }}
                  >
                    <span style={{ fontFamily: 'monospace', fontSize: '10px' }}>
                      {market.market_ticker}
                    </span>
                    <span style={{ marginLeft: '8px', opacity: 0.7 }}>
                      {market.awayTeam} @ {market.homeTeam}
                    </span>
                  </button>
                ))}
              </div>
            )}
          </div>

          <div className="form-group">
            <label>Side:</label>
            <div className="toggle-group">
              <button
                className={`toggle-btn ${side === 'yes' ? 'active' : ''}`}
                onClick={() => setSide('yes')}
              >
                YES
              </button>
              <button
                className={`toggle-btn ${side === 'no' ? 'active' : ''}`}
                onClick={() => setSide('no')}
              >
                NO
              </button>
            </div>
          </div>

          <div className="form-group">
            <label>Action:</label>
            <div className="toggle-group">
              <button
                className={`toggle-btn ${action === 'buy' ? 'active' : ''}`}
                onClick={() => setAction('buy')}
              >
                BUY
              </button>
              <button
                className={`toggle-btn ${action === 'sell' ? 'active' : ''}`}
                onClick={() => setAction('sell')}
              >
                SELL
              </button>
            </div>
          </div>

          <div className="form-group">
            <label>Type:</label>
            <div className="toggle-group">
              <button
                className={`toggle-btn ${type === 'limit' ? 'active' : ''}`}
                onClick={() => {
                  setType('limit');
                }}
              >
                LIMIT
              </button>
              <button
                className={`toggle-btn ${type === 'market' ? 'active' : ''}`}
                onClick={() => {
                  setType('market');
                  setPrice('');
                }}
              >
                MARKET
              </button>
            </div>
          </div>

          <div className="form-group">
            <label htmlFor="order-price">Price:</label>
            <div className="price-input-group">
              <input
                type="number"
                id="order-price"
                className="order-input"
                step="0.0001"
                min="0"
                max="1"
                placeholder="0.5000"
                value={price}
                onChange={(e) => setPrice(e.target.value)}
                disabled={type === 'market'}
              />
              <div className="quick-price-buttons">
                <button
                  className="btn-quick-price"
                  onClick={() => handleQuickPrice('bid')}
                >
                  BID
                </button>
                <button
                  className="btn-quick-price"
                  onClick={() => handleQuickPrice('ask')}
                >
                  ASK
                </button>
                <button
                  className="btn-quick-price"
                  onClick={() => handleQuickPrice('mid')}
                >
                  MID
                </button>
              </div>
            </div>
          </div>

          <div className="form-group">
            <label htmlFor="order-quantity">Quantity:</label>
            <div className="quantity-input-group">
              <input
                type="number"
                id="order-quantity"
                className="order-input"
                min="1"
                value={quantity}
                onChange={(e) => setQuantity(e.target.value)}
                placeholder="100"
              />
              <div className="quick-qty-buttons">
                {[10, 50, 100, 500, 1000].map((qty) => (
                  <button
                    key={qty}
                    className="btn-quick-qty"
                    onClick={() => setQuantity(qty.toString())}
                  >
                    {qty >= 1000 ? '1K' : qty.toString()}
                  </button>
                ))}
              </div>
            </div>
          </div>

          <div className="order-preview">
            <div className="preview-label">Preview:</div>
            <div
              className="preview-text"
              style={{ whiteSpace: 'pre-line' }}
            >
              {preview.text}
            </div>
            <div className="preview-cost">{preview.cost}</div>
          </div>

          <div className="order-ticket-actions">
            <button
              className="btn-secondary"
              onClick={onClose}
            >
              CANCEL
            </button>
            <button
              className="btn-primary"
              onClick={handlePlaceOrder}
              disabled={isPlacing || !marketTicker || (type === 'limit' && !price)}
            >
              {isPlacing ? 'PLACING...' : 'PLACE ORDER'}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default OrderTicket;
