/**
 * ChangeIndicator - Show price change with color coding
 */

import { cn } from '@/lib/utils';
import { TrendingUp, TrendingDown, Minus } from 'lucide-react';
import { TEXT_SIZE_CLASSES } from '@/lib/sizeClasses';

type ChangeIndicatorSize = 'xs' | 'sm' | 'md' | 'lg';

interface ChangeIndicatorProps {
  value: number;
  showIcon?: boolean;
  showPercent?: boolean;
  /** Show "24h" label after the value */
  showLabel?: boolean;
  /** Number of decimal places (default: 1) */
  decimals?: number;
  size?: ChangeIndicatorSize;
  className?: string;
}

// Smaller icon sizes for inline display
const INLINE_ICON_SIZES: Record<ChangeIndicatorSize, number> = {
  xs: 10,
  sm: 12,
  md: 14,
  lg: 16,
};

export function ChangeIndicator({
  value,
  showIcon = true,
  showPercent = true,
  showLabel = false,
  decimals = 1,
  size = 'sm',
  className,
}: ChangeIndicatorProps) {
  const isPositive = value > 0;
  const isNegative = value < 0;

  const colorClass = isPositive
    ? 'text-green-500'
    : isNegative
      ? 'text-red-500'
      : 'text-muted-foreground';

  const Icon = isPositive ? TrendingUp : isNegative ? TrendingDown : Minus;

  // Format value with specified decimal places
  const formattedValue = value.toFixed(decimals);
  const displayValue = isPositive ? `+${formattedValue}` : formattedValue;

  return (
    <span
      className={cn(
        'inline-flex items-center gap-0.5 font-mono tabular-nums',
        TEXT_SIZE_CLASSES[size],
        colorClass,
        className
      )}
    >
      {showIcon && <Icon size={INLINE_ICON_SIZES[size]} />}
      <span>
        {displayValue}
        {showPercent && '%'}
        {showLabel && <span className="text-muted-foreground ml-1 font-sans">24h</span>}
      </span>
    </span>
  );
}

export default ChangeIndicator;
