/**
 * Dropdown - Dropdown menu component
 */

import { useState, useRef, useEffect } from 'react';
import { createPortal } from 'react-dom';
import { ChevronDown, Check } from 'lucide-react';
import { cn } from '@/lib/utils';

interface DropdownOption {
  value: string;
  label: string;
  icon?: React.ReactNode;
  description?: string;
  disabled?: boolean;
}

interface DropdownProps {
  options: DropdownOption[];
  value?: string;
  onChange?: (value: string) => void;
  placeholder?: string;
  label?: string;
  size?: 'sm' | 'md' | 'lg';
  variant?: 'default' | 'ghost' | 'outline';
  disabled?: boolean;
  className?: string;
}

export function Dropdown({
  options,
  value,
  onChange,
  placeholder = 'Select...',
  label,
  size = 'md',
  variant = 'default',
  disabled = false,
  className,
}: DropdownProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [menuPosition, setMenuPosition] = useState({ top: 0, left: 0, width: 0 });
  const containerRef = useRef<HTMLDivElement>(null);
  const triggerRef = useRef<HTMLButtonElement>(null);
  const menuRef = useRef<HTMLDivElement>(null);

  const selectedOption = options.find((opt) => opt.value === value);

  // Update menu position when opening
  useEffect(() => {
    if (isOpen && triggerRef.current) {
      const rect = triggerRef.current.getBoundingClientRect();
      setMenuPosition({
        top: rect.bottom + 4,
        left: rect.left,
        width: rect.width,
      });
    }
  }, [isOpen]);

  // Close on outside click
  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      const target = e.target as Node;
      if (
        containerRef.current &&
        !containerRef.current.contains(target) &&
        menuRef.current &&
        !menuRef.current.contains(target)
      ) {
        setIsOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  // Close on ESC
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.key === 'Escape') setIsOpen(false);
    };
    if (isOpen) {
      window.addEventListener('keydown', handleKeyDown);
      return () => window.removeEventListener('keydown', handleKeyDown);
    }
  }, [isOpen]);

  const sizeClasses = {
    sm: 'h-8 text-xs px-2',
    md: 'h-10 text-sm px-3',
    lg: 'h-12 text-base px-4',
  };

  const variantClasses = {
    default: 'bg-muted border border-border',
    ghost: 'bg-transparent hover:bg-muted',
    outline: 'bg-transparent border border-border',
  };

  const menu =
    isOpen &&
    createPortal(
      <div
        ref={menuRef}
        className="bg-card border-border fixed z-[100] max-h-60 overflow-y-auto rounded-lg border py-1 shadow-lg"
        style={{
          top: menuPosition.top,
          left: menuPosition.left,
          width: menuPosition.width,
        }}
        onMouseDown={(e) => e.stopPropagation()}
      >
        {options.map((option) => (
          <button
            key={option.value}
            onClick={() => {
              if (!option.disabled) {
                onChange?.(option.value);
                setIsOpen(false);
              }
            }}
            disabled={option.disabled}
            className={cn(
              'flex w-full items-center gap-2 px-3 py-2 text-left text-sm transition-colors',
              option.value === value
                ? 'bg-primary/10 text-foreground'
                : 'text-muted-foreground hover:bg-muted hover:text-foreground',
              option.disabled && 'cursor-not-allowed opacity-50'
            )}
          >
            {option.icon && <span className="shrink-0">{option.icon}</span>}
            <div className="min-w-0 flex-1">
              <div className="truncate">{option.label}</div>
              {option.description && (
                <div className="text-muted-foreground truncate text-xs">{option.description}</div>
              )}
            </div>
            {option.value === value && (
              <Check
                size={14}
                className="text-primary shrink-0"
              />
            )}
          </button>
        ))}
      </div>,
      document.body
    );

  return (
    <div
      ref={containerRef}
      className={cn('relative', className)}
    >
      {label && (
        <label className="text-muted-foreground mb-1 block text-xs font-medium">{label}</label>
      )}

      {/* Trigger */}
      <button
        ref={triggerRef}
        onClick={() => !disabled && setIsOpen(!isOpen)}
        disabled={disabled}
        className={cn(
          'flex w-full items-center justify-between gap-2 rounded-lg transition-colors',
          sizeClasses[size],
          variantClasses[variant],
          disabled && 'cursor-not-allowed opacity-50',
          isOpen && 'ring-primary/50 ring-2'
        )}
      >
        <span className={cn('truncate', !selectedOption && 'text-muted-foreground')}>
          {selectedOption?.label || placeholder}
        </span>
        <ChevronDown
          size={16}
          className={cn('text-muted-foreground transition-transform', isOpen && 'rotate-180')}
        />
      </button>

      {/* Menu rendered via portal */}
      {menu}
    </div>
  );
}

// Multi-select dropdown
interface MultiSelectProps {
  options: DropdownOption[];
  values: string[];
  onChange: (values: string[]) => void;
  placeholder?: string;
  label?: string;
  maxDisplay?: number;
  className?: string;
}

export function MultiSelect({
  options,
  values,
  onChange,
  placeholder = 'Select...',
  label,
  maxDisplay = 2,
  className,
}: MultiSelectProps) {
  const [isOpen, setIsOpen] = useState(false);
  const containerRef = useRef<HTMLDivElement>(null);

  const selectedOptions = options.filter((opt) => values.includes(opt.value));

  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      if (containerRef.current && !containerRef.current.contains(e.target as Node)) {
        setIsOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const toggleValue = (value: string) => {
    if (values.includes(value)) {
      onChange(values.filter((v) => v !== value));
    } else {
      onChange([...values, value]);
    }
  };

  const displayText = () => {
    if (selectedOptions.length === 0) return placeholder;
    if (selectedOptions.length <= maxDisplay) {
      return selectedOptions.map((o) => o.label).join(', ');
    }
    return `${selectedOptions
      .slice(0, maxDisplay)
      .map((o) => o.label)
      .join(', ')} +${selectedOptions.length - maxDisplay}`;
  };

  return (
    <div
      ref={containerRef}
      className={cn('relative', className)}
    >
      {label && (
        <label className="text-muted-foreground mb-1 block text-xs font-medium">{label}</label>
      )}

      <button
        onClick={() => setIsOpen(!isOpen)}
        className={cn(
          'bg-muted border-border flex h-10 w-full items-center justify-between gap-2 rounded-lg border px-3 text-sm transition-colors',
          isOpen && 'ring-primary/50 ring-2'
        )}
      >
        <span className={cn('truncate', selectedOptions.length === 0 && 'text-muted-foreground')}>
          {displayText()}
        </span>
        <ChevronDown
          size={16}
          className={cn('text-muted-foreground transition-transform', isOpen && 'rotate-180')}
        />
      </button>

      {isOpen && (
        <div className="bg-card border-border absolute left-0 right-0 top-full z-20 mt-1 max-h-60 overflow-y-auto rounded-lg border py-1 shadow-lg">
          {options.map((option) => (
            <button
              key={option.value}
              onClick={() => toggleValue(option.value)}
              className="hover:bg-muted flex w-full items-center gap-2 px-3 py-2 text-left text-sm transition-colors"
            >
              <div
                className={cn(
                  'flex h-4 w-4 items-center justify-center rounded border',
                  values.includes(option.value) ? 'bg-primary border-primary' : 'border-border'
                )}
              >
                {values.includes(option.value) && (
                  <Check
                    size={12}
                    className="text-primary-foreground"
                  />
                )}
              </div>
              <span className="truncate">{option.label}</span>
            </button>
          ))}
        </div>
      )}
    </div>
  );
}
