/**
 * Odds - Consistent odds value display
 *
 * Displays odds in cents (46¢), American (+115), or dual (46¢ | +115) format.
 * Color coding is based on cents threshold (<=50 green, >=51 red),
 * regardless of output format.
 */

import { memo } from 'react';
import { cn } from '@/lib/utils';

interface OddsProps {
  /** The price in cents (0-100) */
  cents: number | null | undefined;

  /** Display format */
  format: 'cents' | 'american' | 'dual';

  /**
   * Color code using cents threshold (<=50 green, >=51 red)
   * Default: true
   */
  colorize?: boolean;

  /** Text to show when value is null/undefined (default: '—') */
  nullText?: string;

  /** Additional CSS classes */
  className?: string;
}

/**
 * Convert cents (probability) to American odds
 */
function centsToAmericanOdds(cents: number): { value: number; display: string } | null {
  const prob = cents / 100;
  if (!Number.isFinite(prob) || prob <= 0 || prob >= 1) return null;

  const oneOverProb = 1 / prob;
  let odds: number;
  if (oneOverProb > 2) {
    odds = (oneOverProb - 1) * 100;
  } else {
    odds = -100 / (oneOverProb - 1);
  }

  const rounded = Math.round(odds);
  const display = rounded > 0 ? `+${rounded}` : String(rounded);
  return { value: rounded, display };
}

function getCentsThresholdColorClass(cents: number): string {
  return cents <= 50 ? 'text-positive' : 'text-negative';
}

/**
 * Renders odds with proper formatting and styling
 */
export const Odds = memo(function Odds({
  cents,
  format,
  colorize,
  nullText = '—',
  className,
}: OddsProps) {
  // Handle null/undefined
  if (cents === null || cents === undefined || !Number.isFinite(cents)) {
    return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
  }

  const shouldColorize = colorize ?? true;
  const thresholdColorClass = shouldColorize ? getCentsThresholdColorClass(cents) : '';

  // Cents format - threshold-colored display
  if (format === 'cents') {
    return <span className={cn(thresholdColorClass, className)}>{Math.round(cents)}¢</span>;
  }

  if (format === 'dual') {
    const american = centsToAmericanOdds(cents);
    if (!american) {
      return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
    }
    return (
      <span className={cn(thresholdColorClass, className)}>
        {Math.round(cents)}¢ | {american.display}
      </span>
    );
  }

  // American format
  const american = centsToAmericanOdds(cents);
  if (!american) {
    return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
  }

  return <span className={cn(thresholdColorClass, className)}>{american.display}</span>;
});

export default Odds;
