/**
 * ProbabilityBar - Visual representation of binary probability split
 *
 * A slim horizontal bar showing the probability split between two sides.
 * Labels are placed externally (above/below) for reliability at all probability levels.
 * Includes a dotted midpoint indicator at 50%.
 */

import { cn } from '@/lib/utils';

interface ProbabilityBarProps {
  /** Probability for left/YES side (0-100) */
  probability: number;
  /** Height in pixels (default: 6) */
  height?: number;
  /** Additional class names */
  className?: string;
}

export function ProbabilityBar({ probability, height = 6, className }: ProbabilityBarProps) {
  const leftPct = Math.max(1, Math.min(99, probability));

  return (
    <div
      className={cn('relative w-full', className)}
      style={{ height }}
    >
      {/* Bar container with overflow hidden */}
      <div className="absolute inset-0 flex overflow-hidden rounded-full">
        {/* Left (YES) side */}
        <div
          className="bg-green-500 transition-all duration-300"
          style={{ width: `${leftPct}%` }}
        />
        {/* Right (NO) side */}
        <div
          className="bg-red-500 transition-all duration-300"
          style={{ width: `${100 - leftPct}%` }}
        />
      </div>
      {/* Midpoint indicator - outside overflow container so it extends past */}
      <div
        className="absolute border-l border-dotted border-white"
        style={{ left: '50%', top: '-4px', bottom: '-4px', transform: 'translateX(-50%)' }}
      />
    </div>
  );
}

export default ProbabilityBar;
