import { useState } from 'react';
import { Eye, EyeOff } from 'lucide-react';
import { cn } from '@/lib/utils';

interface SecretCredentialFieldProps {
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
  disabled?: boolean;
  multiline?: boolean;
  className?: string;
}

export function SecretCredentialField({
  value,
  onChange,
  placeholder,
  disabled = false,
  multiline = false,
  className,
}: SecretCredentialFieldProps) {
  const [visible, setVisible] = useState(false);

  if (multiline) {
    return (
      <div className={cn('relative', className)}>
        <textarea
          value={value}
          onChange={(e) => onChange(e.target.value)}
          placeholder={placeholder}
          disabled={disabled}
          className={cn(
            'border-border bg-background h-32 w-full resize-none rounded-lg border py-2 pl-3 pr-10 font-mono text-sm',
            visible
              ? ''
              : '[-webkit-text-security:disc] [font-family:monospace] [text-security:disc]'
          )}
        />
        <button
          type="button"
          onClick={() => setVisible((v) => !v)}
          disabled={disabled}
          className="text-muted-foreground hover:bg-muted absolute right-1 top-1 rounded p-2 transition-colors disabled:opacity-50"
          aria-label={visible ? 'Hide secret' : 'Show secret'}
          title={visible ? 'Hide' : 'Show'}
        >
          {visible ? <EyeOff size={16} /> : <Eye size={16} />}
        </button>
      </div>
    );
  }

  return (
    <div className={cn('relative', className)}>
      <input
        type={visible ? 'text' : 'password'}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        placeholder={placeholder}
        disabled={disabled}
        className="border-border bg-background h-10 w-full rounded-lg border px-3 pr-10 font-mono text-sm"
      />
      <button
        type="button"
        onClick={() => setVisible((v) => !v)}
        disabled={disabled}
        className="text-muted-foreground hover:bg-muted absolute right-1 top-1/2 -translate-y-1/2 rounded p-2 transition-colors disabled:opacity-50"
        aria-label={visible ? 'Hide secret' : 'Show secret'}
        title={visible ? 'Hide' : 'Show'}
      >
        {visible ? <EyeOff size={16} /> : <Eye size={16} />}
      </button>
    </div>
  );
}
