/**
 * TickerDisplay - Renders parsed Kalshi ticker information
 *
 * Supports multiple display variants for different contexts.
 */

import { useMemo } from 'react';
import { cn } from '@/lib/utils';
import { parseKalshiTicker, formatDateShort } from '@/lib/kalshiTicker';
import type { ParsedTicker } from '@/lib/kalshiTicker';

export type TickerDisplayVariant = 'compact' | 'full' | 'inline';

interface TickerDisplayProps {
  /** The raw ticker string to parse and display */
  ticker: string;

  /** Optional pre-parsed ticker (to avoid re-parsing) */
  parsed?: ParsedTicker;

  /** Display variant */
  variant?: TickerDisplayVariant;

  /** Optional market title to show as fallback/supplement */
  title?: string;

  /** Additional CSS classes */
  className?: string;

  /** Show the raw ticker below the parsed info */
  showRaw?: boolean;
}

/**
 * Renders a parsed Kalshi ticker with smart formatting
 */
export function TickerDisplay({
  ticker,
  parsed: parsedProp,
  variant = 'compact',
  title,
  className,
  showRaw = true,
}: TickerDisplayProps) {
  const parsed = useMemo(() => parsedProp ?? parseKalshiTicker(ticker), [ticker, parsedProp]);

  // For unknown tickers, prefer showing the title if available
  const isUnknown = parsed.category === 'unknown';
  const mainLabel = isUnknown && title ? title : parsed.shortLabel;

  if (variant === 'inline') {
    return <span className={cn('font-mono text-sm', className)}>{mainLabel}</span>;
  }

  if (variant === 'full') {
    return (
      <div className={cn('flex flex-col gap-0.5', className)}>
        <TickerFullDisplay
          parsed={parsed}
          title={title}
        />
        {showRaw && <span className="text-muted-foreground font-mono text-xs">{parsed.raw}</span>}
      </div>
    );
  }

  // Default: compact variant
  return (
    <div className={cn('flex flex-col leading-tight', className)}>
      <span className="text-foreground text-sm font-medium">{mainLabel}</span>
      {showRaw && <span className="text-muted-foreground font-mono text-xs">{parsed.raw}</span>}
    </div>
  );
}

/**
 * Full variant with all parsed details
 */
function TickerFullDisplay({ parsed, title }: { parsed: ParsedTicker; title?: string }) {
  const { category, date, game, player } = parsed;

  // Category badge
  const categoryLabel = getCategoryLabel(category);
  const categoryColor = getCategoryColor(category);

  return (
    <div className="flex flex-col gap-1">
      {/* Category badge + date */}
      <div className="flex items-center gap-2">
        <span
          className={cn(
            'rounded px-1.5 py-0.5 text-[10px] font-semibold uppercase tracking-wide',
            categoryColor
          )}
        >
          {categoryLabel}
        </span>
        {date && (
          <span className="text-muted-foreground font-mono text-xs">{formatDateShort(date)}</span>
        )}
      </div>

      {/* Game info */}
      {game && (
        <div className="text-sm font-medium">
          {game.awayName ?? game.awayCode} @ {game.homeName ?? game.homeCode}
        </div>
      )}

      {/* Player info */}
      {player && (
        <div className="text-sm">
          <span className="text-muted-foreground">{player.teamCode}</span>{' '}
          <span className="font-medium">{player.displayName ?? player.playerId}</span>
        </div>
      )}

      {/* Fallback to title if no structured data */}
      {!game && !player && title && <div className="text-sm font-medium">{title}</div>}
    </div>
  );
}

function getCategoryLabel(category: ParsedTicker['category']): string {
  const labels: Record<ParsedTicker['category'], string> = {
    // NBA
    'nba-game': 'NBA',
    'nba-spread': 'NBA SPR',
    'nba-total': 'NBA TOT',
    'nba-season': 'NBA',
    // NFL
    'nfl-game': 'NFL',
    'nfl-spread': 'NFL SPR',
    'nfl-total': 'NFL TOT',
    'nfl-2td': '2+ TD',
    'nfl-first-td': '1st TD',
    'nfl-superbowl': 'SB',
    // NHL
    'nhl-game': 'NHL',
    'nhl-spread': 'NHL SPR',
    'nhl-total': 'NHL TOT',
    // NCAA
    'ncaa-game': 'NCAA',
    'ncaa-spread': 'NCAA SPR',
    'ncaa-total': 'NCAA TOT',
    // Tennis
    'tennis-match': 'TENNIS',
    // Other
    politics: 'POL',
    crypto: 'CRYPTO',
    unknown: '?',
  };
  return labels[category];
}

function getCategoryColor(category: ParsedTicker['category']): string {
  const colors: Record<ParsedTicker['category'], string> = {
    // NBA - orange
    'nba-game': 'bg-orange-500/20 text-orange-400',
    'nba-spread': 'bg-orange-500/20 text-orange-400',
    'nba-total': 'bg-orange-500/20 text-orange-400',
    'nba-season': 'bg-orange-500/20 text-orange-400',
    // NFL - blue
    'nfl-game': 'bg-blue-500/20 text-blue-400',
    'nfl-spread': 'bg-blue-500/20 text-blue-400',
    'nfl-total': 'bg-blue-500/20 text-blue-400',
    'nfl-2td': 'bg-blue-500/20 text-blue-400',
    'nfl-first-td': 'bg-blue-500/20 text-blue-400',
    'nfl-superbowl': 'bg-blue-500/20 text-blue-400',
    // NHL - cyan
    'nhl-game': 'bg-cyan-500/20 text-cyan-400',
    'nhl-spread': 'bg-cyan-500/20 text-cyan-400',
    'nhl-total': 'bg-cyan-500/20 text-cyan-400',
    // NCAA - purple
    'ncaa-game': 'bg-purple-500/20 text-purple-400',
    'ncaa-spread': 'bg-purple-500/20 text-purple-400',
    'ncaa-total': 'bg-purple-500/20 text-purple-400',
    // Tennis - green
    'tennis-match': 'bg-green-500/20 text-green-400',
    // Other
    politics: 'bg-red-500/20 text-red-400',
    crypto: 'bg-yellow-500/20 text-yellow-400',
    unknown: 'bg-muted text-muted-foreground',
  };
  return colors[category];
}

export default TickerDisplay;
