/**
 * Toggle/Switch - On/off toggle component
 */

import { cn } from '@/lib/utils';

interface ToggleProps {
  checked: boolean;
  onChange: (checked: boolean) => void;
  label?: string;
  description?: string;
  size?: 'sm' | 'md' | 'lg';
  disabled?: boolean;
  className?: string;
}

export function Toggle({
  checked,
  onChange,
  label,
  description,
  size = 'md',
  disabled = false,
  className,
}: ToggleProps) {
  const sizeClasses = {
    sm: { track: 'w-8 h-4', thumb: 'w-3 h-3', translate: 'translate-x-4' },
    md: { track: 'w-11 h-6', thumb: 'w-5 h-5', translate: 'translate-x-5' },
    lg: { track: 'w-14 h-7', thumb: 'w-6 h-6', translate: 'translate-x-7' },
  };

  const sizes = sizeClasses[size];

  return (
    <label
      className={cn(
        'flex cursor-pointer items-center gap-3',
        disabled && 'cursor-not-allowed opacity-50',
        className
      )}
    >
      <button
        type="button"
        role="switch"
        aria-checked={checked}
        disabled={disabled}
        onClick={() => !disabled && onChange(!checked)}
        className={cn(
          'focus:ring-primary/50 focus:ring-offset-background relative inline-flex shrink-0 rounded-full transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-offset-2',
          sizes.track,
          checked ? 'bg-primary' : 'bg-muted'
        )}
      >
        <span
          className={cn(
            'pointer-events-none inline-block rounded-full bg-white shadow-lg ring-0 transition duration-200 ease-in-out',
            sizes.thumb,
            'my-auto translate-x-0.5',
            checked && sizes.translate
          )}
        />
      </button>
      {(label || description) && (
        <div className="flex flex-col">
          {label && (
            <span className={cn('font-medium', size === 'sm' ? 'text-xs' : 'text-sm')}>
              {label}
            </span>
          )}
          {description && <span className="text-muted-foreground text-xs">{description}</span>}
        </div>
      )}
    </label>
  );
}

// Labeled toggle with on/off text
interface LabeledToggleProps extends Omit<ToggleProps, 'label' | 'description'> {
  onLabel?: string;
  offLabel?: string;
}

export function LabeledToggle({
  checked,
  onChange,
  onLabel = 'On',
  offLabel = 'Off',
  size = 'md',
  disabled = false,
  className,
}: LabeledToggleProps) {
  return (
    <div className={cn('flex items-center gap-2', className)}>
      <span
        className={cn(
          'text-sm transition-colors',
          !checked ? 'text-foreground font-medium' : 'text-muted-foreground'
        )}
      >
        {offLabel}
      </span>
      <Toggle
        checked={checked}
        onChange={onChange}
        size={size}
        disabled={disabled}
      />
      <span
        className={cn(
          'text-sm transition-colors',
          checked ? 'text-foreground font-medium' : 'text-muted-foreground'
        )}
      >
        {onLabel}
      </span>
    </div>
  );
}
