/**
 * HotkeyConfigModal
 *
 * Modal for customizing keyboard shortcuts.
 * Users can click on a hotkey to record a new binding.
 */

import { useState, useEffect, useCallback, useRef } from 'react';
import { X, RotateCcw } from 'lucide-react';
import { cn } from '@/lib/utils';
import { Modal } from '@/components/atoms/Modal';
import {
  useHotkeyContext,
  type HotkeyId,
  type HotkeyConfig,
  formatHotkey,
  eventToHotkeyConfig,
  hotkeyConfigsEqual,
} from '@/lib/hotkeys';

interface HotkeyConfigModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export function HotkeyConfigModal({ isOpen, onClose }: HotkeyConfigModalProps) {
  const { mappings, updateHotkey, resetHotkey, resetAll } = useHotkeyContext();
  const [recordingId, setRecordingId] = useState<HotkeyId | null>(null);
  const [pendingConfig, setPendingConfig] = useState<HotkeyConfig | null>(null);
  const recordingRef = useRef<HTMLButtonElement>(null);

  // Group mappings by category
  const groupedMappings = Array.from(mappings.values()).reduce(
    (acc, mapping) => {
      if (!acc[mapping.category]) {
        acc[mapping.category] = [];
      }
      acc[mapping.category].push(mapping);
      return acc;
    },
    {} as Record<string, typeof mappings extends Map<unknown, infer V> ? V[] : never>
  );

  // Handle key recording
  const handleKeyDown = useCallback(
    (e: KeyboardEvent) => {
      if (!recordingId) return;

      // Ignore modifier-only presses
      if (['Control', 'Shift', 'Alt', 'Meta'].includes(e.key)) return;

      // Escape cancels recording
      if (e.key === 'Escape') {
        setRecordingId(null);
        setPendingConfig(null);
        return;
      }

      e.preventDefault();
      e.stopPropagation();

      const config = eventToHotkeyConfig(e);
      setPendingConfig(config);
    },
    [recordingId]
  );

  // Register/unregister key listener when recording
  useEffect(() => {
    if (recordingId) {
      window.addEventListener('keydown', handleKeyDown, true);
      return () => window.removeEventListener('keydown', handleKeyDown, true);
    }
  }, [recordingId, handleKeyDown]);

  // Focus the recording button when starting to record
  useEffect(() => {
    if (recordingId && recordingRef.current) {
      recordingRef.current.focus();
    }
  }, [recordingId]);

  const handleStartRecording = (id: HotkeyId) => {
    setRecordingId(id);
    setPendingConfig(null);
  };

  const handleSaveRecording = () => {
    if (recordingId && pendingConfig) {
      updateHotkey(recordingId, pendingConfig);
    }
    setRecordingId(null);
    setPendingConfig(null);
  };

  const handleCancelRecording = () => {
    setRecordingId(null);
    setPendingConfig(null);
  };

  const handleResetHotkey = (id: HotkeyId) => {
    resetHotkey(id);
  };

  const handleResetAll = () => {
    if (confirm('Reset all hotkeys to defaults?')) {
      resetAll();
    }
  };

  // Check for conflicts
  const getConflict = (id: HotkeyId, config: HotkeyConfig): string | null => {
    for (const [otherId, mapping] of mappings) {
      if (otherId === id) continue;
      if (hotkeyConfigsEqual(mapping.currentConfig, config)) {
        return mapping.label;
      }
    }
    return null;
  };

  const conflict = recordingId && pendingConfig ? getConflict(recordingId, pendingConfig) : null;

  // Define category order for consistent layout
  const categoryOrder = ['Sport', 'Market Type', 'Odds Format', 'Trading'];
  const sortedCategories = categoryOrder.filter((cat) => groupedMappings[cat]);

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Keyboard Shortcuts"
      size="2xl"
    >
      <div className="space-y-4">
        {/* Instructions */}
        <p className="text-muted-foreground text-sm">
          Click on a shortcut to record a new key binding. Press Escape to cancel.
        </p>

        {/* Hotkey groups in 2-column grid */}
        <div className="grid grid-cols-2 gap-6">
          {sortedCategories.map((category) => {
            const items = groupedMappings[category];
            if (!items) return null;
            return (
              <div key={category}>
                <h3 className="text-muted-foreground mb-2 text-xs font-medium uppercase tracking-wider">
                  {category}
                </h3>
                <div className="space-y-1">
                  {items.map((mapping) => {
                    const isRecording = recordingId === mapping.id;
                    const isModified = !hotkeyConfigsEqual(
                      mapping.currentConfig,
                      mapping.defaultConfig
                    );
                    const displayConfig =
                      isRecording && pendingConfig ? pendingConfig : mapping.currentConfig;

                    return (
                      <div
                        key={mapping.id}
                        className={cn(
                          'flex items-center gap-3 rounded-lg px-3 py-2',
                          isRecording ? 'bg-primary/10 ring-primary ring-2' : 'hover:bg-muted/50'
                        )}
                      >
                        {/* Label - fixed width for alignment */}
                        <span className="w-32 flex-shrink-0 text-sm">{mapping.label}</span>

                        {/* Key button and controls */}
                        <div className="flex flex-1 items-center justify-end gap-2">
                          {isRecording ? (
                            <>
                              <button
                                ref={recordingRef}
                                className={cn(
                                  'bg-muted w-20 rounded px-3 py-1 text-center font-mono text-sm',
                                  pendingConfig
                                    ? 'text-foreground'
                                    : 'text-muted-foreground animate-pulse'
                                )}
                              >
                                {pendingConfig ? formatHotkey(pendingConfig) : '...'}
                              </button>
                              {pendingConfig && (
                                <>
                                  {conflict && (
                                    <span className="text-destructive text-xs">
                                      Conflicts with {conflict}
                                    </span>
                                  )}
                                  <button
                                    onClick={handleSaveRecording}
                                    disabled={!!conflict}
                                    className={cn(
                                      'rounded px-2 py-1 text-xs font-medium',
                                      conflict
                                        ? 'bg-muted text-muted-foreground cursor-not-allowed'
                                        : 'bg-primary text-primary-foreground hover:bg-primary/90'
                                    )}
                                  >
                                    Save
                                  </button>
                                </>
                              )}
                              <button
                                onClick={handleCancelRecording}
                                className="text-muted-foreground hover:text-foreground w-6 p-1"
                              >
                                <X className="h-4 w-4" />
                              </button>
                            </>
                          ) : (
                            <>
                              <button
                                onClick={() => handleStartRecording(mapping.id as HotkeyId)}
                                className={cn(
                                  'bg-muted hover:bg-muted/80 w-20 rounded px-3 py-1 text-center font-mono text-sm transition-colors',
                                  isModified && 'ring-primary/50 ring-1'
                                )}
                              >
                                {formatHotkey(displayConfig)}
                              </button>
                              {/* Always reserve space for reset button */}
                              <button
                                onClick={() => handleResetHotkey(mapping.id as HotkeyId)}
                                className={cn(
                                  'w-6 p-1',
                                  isModified
                                    ? 'text-muted-foreground hover:text-foreground'
                                    : 'invisible'
                                )}
                                title="Reset to default"
                              >
                                <RotateCcw className="h-3 w-3" />
                              </button>
                            </>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>
              </div>
            );
          })}
        </div>

        {/* Reset all button */}
        <div className="border-border flex justify-end border-t pt-4">
          <button
            onClick={handleResetAll}
            className="text-muted-foreground hover:text-foreground flex items-center gap-1 text-sm"
          >
            <RotateCcw className="h-3 w-3" />
            Reset all to defaults
          </button>
        </div>
      </div>
    </Modal>
  );
}
