/**
 * AppHeader - Top header bar
 *
 * Shows:
 * - Current view title (left)
 * - API connection status pills (right)
 * - Optional feed status and timestamp (right, below pills when on value dashboard)
 */

import { cn } from '@/lib/utils';

/** Per-service connection status */
export interface ApiStatus {
  /** Kalshi REST API */
  kalshi: 'connected' | 'disconnected';
  /** WebSocket market data feed */
  feed: 'live' | 'offline';
  /** Polymarket CLOB client */
  poly: 'connected' | 'disconnected' | 'error';
  /** The Odds API (key configured?) */
  odds: 'configured' | 'none';
}

interface AppHeaderProps {
  title?: string;
  /** Optional feed status for live views */
  feedStatus?: 'live' | 'offline';
  /** Optional last update timestamp */
  lastUpdate?: string;
  /** API connection statuses */
  apiStatus?: ApiStatus;
}

type PillColor = 'green' | 'grey' | 'red';

const PILL_STYLES: Record<PillColor, string> = {
  green: 'border-emerald-500/40 bg-emerald-500/10 text-emerald-400',
  grey: 'border-zinc-500/30 bg-zinc-500/10 text-zinc-500',
  red: 'border-red-500/40 bg-red-500/10 text-red-400',
};

const DOT_STYLES: Record<PillColor, string> = {
  green: 'bg-emerald-500',
  grey: 'bg-zinc-600',
  red: 'bg-red-500',
};

function StatusPill({ label, color, title }: { label: string; color: PillColor; title: string }) {
  return (
    <div
      className={cn('flex items-center gap-1.5 rounded border px-2 py-0.5', PILL_STYLES[color])}
      title={title}
    >
      <span className={cn('h-1.5 w-1.5 rounded-full', DOT_STYLES[color])} />
      <span className="text-sm">{label}</span>
    </div>
  );
}

export function AppHeader({ title, feedStatus, lastUpdate, apiStatus }: AppHeaderProps) {
  return (
    <div className="flex h-full items-center justify-between px-4">
      {/* Left: page title */}
      <div>{title && <h1 className="text-sm font-medium">{title}</h1>}</div>

      {/* Right: status pills + optional feed detail */}
      <div className="flex items-center gap-3">
        {/* Feed timestamp (shown on value dashboard) */}
        {feedStatus && lastUpdate && (
          <span className="text-muted-foreground mr-1 font-mono text-xs">{lastUpdate}</span>
        )}

        {/* API connection pills */}
        {apiStatus && (
          <>
            <StatusPill
              label="Kalshi"
              color={apiStatus.kalshi === 'connected' ? 'green' : 'grey'}
              title={
                apiStatus.kalshi === 'connected'
                  ? 'Kalshi API connected'
                  : 'Kalshi API disconnected'
              }
            />
            <StatusPill
              label="Feed"
              color={apiStatus.feed === 'live' ? 'green' : 'grey'}
              title={apiStatus.feed === 'live' ? 'Market data streaming' : 'Market data offline'}
            />
            <StatusPill
              label="Poly"
              color={
                apiStatus.poly === 'connected'
                  ? 'green'
                  : apiStatus.poly === 'error'
                    ? 'red'
                    : 'grey'
              }
              title={
                apiStatus.poly === 'connected'
                  ? 'Polymarket connected'
                  : apiStatus.poly === 'error'
                    ? 'Polymarket error'
                    : 'Polymarket not connected'
              }
            />
            <StatusPill
              label="Odds"
              color={apiStatus.odds === 'configured' ? 'green' : 'grey'}
              title={
                apiStatus.odds === 'configured' ? 'Odds API key configured' : 'No Odds API key'
              }
            />
          </>
        )}
      </div>
    </div>
  );
}
