/**
 * AppSidebar - Navigation sidebar
 *
 * Icon-based navigation for switching between views.
 */

import { useState } from 'react';
import {
  Compass,
  BarChart3,
  Briefcase,
  ClipboardList,
  FlaskConical,
  Bug,
  Settings,
  LogOut,
  Sun,
  Moon,
  RefreshCw,
  Megaphone,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { useTheme } from '@/contexts/ThemeContext';
import { UpdatesModal } from '@/components/layout/UpdatesModal';
import { CURRENT_VERSION } from '@/lib/changelog';

export type AppView =
  | 'explore'
  | 'nba'
  | 'value-dashboard'
  | 'positions'
  | 'orders'
  | 'dev-explorer'
  | 'diagnostics'
  | 'settings';

interface NavItem {
  id: AppView;
  icon: React.ReactNode;
  label: string;
}

const navItems: NavItem[] = [
  { id: 'explore', icon: <Compass className="h-5 w-5" />, label: 'Explore Markets' },
  { id: 'value-dashboard', icon: <BarChart3 className="h-5 w-5" />, label: 'Value Dashboard' },
  { id: 'positions', icon: <Briefcase className="h-5 w-5" />, label: 'Positions' },
  { id: 'orders', icon: <ClipboardList className="h-5 w-5" />, label: 'Orders' },
  {
    id: 'dev-explorer',
    icon: <FlaskConical className="h-5 w-5" />,
    label: 'Sports Inventory (Dev)',
  },
  { id: 'diagnostics', icon: <Bug className="h-5 w-5" />, label: 'Diagnostics' },
  { id: 'settings', icon: <Settings className="h-5 w-5" />, label: 'Settings' },
];

interface AppSidebarProps {
  currentView: AppView;
  onViewChange: (view: AppView) => void;
  onDisconnect?: () => void;
}

export function AppSidebar({ currentView, onViewChange, onDisconnect }: AppSidebarProps) {
  const [showUpdates, setShowUpdates] = useState(false);

  return (
    <div className="flex h-full flex-col">
      {/* Logo/Brand */}
      <div className="border-border flex h-12 items-center justify-center border-b">
        <span className="text-primary text-lg font-bold">K</span>
      </div>

      {/* Navigation */}
      <nav className="flex-1 py-2">
        {navItems.map((item) => (
          <NavButton
            key={item.id}
            icon={item.icon}
            label={item.label}
            isActive={currentView === item.id}
            onClick={() => onViewChange(item.id)}
          />
        ))}
      </nav>

      {/* Theme toggle, refresh, updates, and disconnect at bottom */}
      <div className="border-border border-t py-2">
        <ThemeButton />
        <RefreshCacheButton />
        <UpdatesButton onClick={() => setShowUpdates(true)} />
        {onDisconnect && (
          <NavButton
            icon={<LogOut className="h-5 w-5" />}
            label="Disconnect"
            onClick={onDisconnect}
            variant="danger"
          />
        )}
      </div>

      <UpdatesModal
        isOpen={showUpdates}
        onClose={() => setShowUpdates(false)}
      />
    </div>
  );
}

interface NavButtonProps {
  icon: React.ReactNode;
  label: string;
  isActive?: boolean;
  onClick: () => void;
  variant?: 'default' | 'danger';
}

function NavButton({ icon, label, isActive, onClick, variant = 'default' }: NavButtonProps) {
  return (
    <button
      onClick={onClick}
      title={label}
      className={cn(
        'group relative flex h-12 w-full items-center justify-center transition-colors',
        variant === 'default' &&
          !isActive &&
          'text-muted-foreground hover:text-foreground hover:bg-muted/50',
        variant === 'default' && isActive && 'text-primary bg-primary/10',
        variant === 'danger' &&
          'text-muted-foreground hover:text-destructive hover:bg-destructive/10'
      )}
    >
      {icon}

      {/* Active indicator */}
      {isActive && (
        <div className="bg-primary absolute left-0 top-1/2 h-6 w-0.5 -translate-y-1/2 rounded-r" />
      )}

      {/* Tooltip */}
      <div className="bg-popover text-popover-foreground invisible absolute left-full z-50 ml-2 whitespace-nowrap rounded px-2 py-1 text-xs opacity-0 shadow-lg transition-all group-hover:visible group-hover:opacity-100">
        {label}
      </div>
    </button>
  );
}

function ThemeButton() {
  const { theme, toggleTheme } = useTheme();
  const isDark = theme === 'dark';

  return (
    <button
      onClick={toggleTheme}
      title={isDark ? 'Switch to light mode' : 'Switch to dark mode'}
      className={cn(
        'group relative flex h-12 w-full items-center justify-center transition-colors',
        'text-muted-foreground hover:text-foreground hover:bg-muted/50'
      )}
    >
      {isDark ? <Sun className="h-5 w-5" /> : <Moon className="h-5 w-5" />}

      {/* Tooltip */}
      <div className="bg-popover text-popover-foreground invisible absolute left-full z-50 ml-2 whitespace-nowrap rounded px-2 py-1 text-xs opacity-0 shadow-lg transition-all group-hover:visible group-hover:opacity-100">
        {isDark ? 'Light mode' : 'Dark mode'}
      </div>
    </button>
  );
}

/** Keys to preserve across cache refresh (credentials + auth) */
const PRESERVE_KEYS = [
  'kalshi_profiles_v1',
  'kalshi_creds_encrypted',
  'polymarket_config_v1',
  'galactus_odds_api_key',
];

function RefreshCacheButton() {
  const handleRefresh = () => {
    // Save credential values
    const saved: [string, string | null][] = PRESERVE_KEYS.map((k) => [k, localStorage.getItem(k)]);
    // Clear everything
    localStorage.clear();
    // Restore credentials
    for (const [key, value] of saved) {
      if (value !== null) localStorage.setItem(key, value);
    }
    // Navigate to root and hard reload
    window.location.href = '/';
  };

  return (
    <button
      onClick={handleRefresh}
      title="Refresh cache"
      className={cn(
        'group relative flex h-12 w-full items-center justify-center transition-colors',
        'text-muted-foreground hover:text-foreground hover:bg-muted/50'
      )}
    >
      <RefreshCw className="h-5 w-5" />

      {/* Tooltip */}
      <div className="bg-popover text-popover-foreground invisible absolute left-full z-50 ml-2 whitespace-nowrap rounded px-2 py-1 text-xs opacity-0 shadow-lg transition-all group-hover:visible group-hover:opacity-100">
        Refresh cache
      </div>
    </button>
  );
}

function UpdatesButton({ onClick }: { onClick: () => void }) {
  return (
    <button
      onClick={onClick}
      title={`Updates (${CURRENT_VERSION})`}
      className={cn(
        'group relative flex h-12 w-full items-center justify-center transition-colors',
        'text-muted-foreground hover:text-foreground hover:bg-muted/50'
      )}
    >
      <Megaphone className="h-5 w-5" />

      {/* Tooltip */}
      <div className="bg-popover text-popover-foreground invisible absolute left-full z-50 ml-2 whitespace-nowrap rounded px-2 py-1 text-xs opacity-0 shadow-lg transition-all group-hover:visible group-hover:opacity-100">
        Updates ({CURRENT_VERSION})
      </div>
    </button>
  );
}
