import { ChevronRight, Calendar, TrendingUp } from 'lucide-react';
import type { KalshiEvent } from '@/types';

interface EventCardProps {
  event: KalshiEvent;
  onClick: () => void;
}

export function EventCard({ event, onClick }: EventCardProps) {
  const subtitle = event.subtitle || event.sub_title;
  const strikeDate = event.strike_date ? formatStrikeDate(event.strike_date) : null;
  const strikePeriod = event.strike_period;
  const marketCount = event.markets?.length;

  return (
    <button
      onClick={onClick}
      className="bg-card border-border hover:border-primary/50 hover:bg-muted/30 group flex h-full flex-col rounded-lg border p-3 text-left transition-colors"
    >
      {/* Header: title + chevron */}
      <div className="mb-1 flex items-start justify-between gap-2">
        <h3 className="group-hover:text-primary line-clamp-2 flex-1 text-sm font-medium transition-colors">
          {event.title}
        </h3>
        <ChevronRight className="text-muted-foreground group-hover:text-primary mt-0.5 h-4 w-4 flex-shrink-0 transition-colors" />
      </div>

      {/* Subtitle */}
      {subtitle && <p className="text-muted-foreground mb-2 line-clamp-1 text-xs">{subtitle}</p>}

      {/* Meta row: strike date/period + market count */}
      <div className="mt-auto flex flex-wrap items-center gap-2">
        {(strikeDate || strikePeriod) && (
          <span className="inline-flex items-center gap-1 rounded bg-amber-500/20 px-1.5 py-0.5 text-[10px] font-medium text-amber-400">
            <Calendar className="h-2.5 w-2.5" />
            {strikeDate || strikePeriod}
          </span>
        )}
        {marketCount !== undefined && marketCount > 0 && (
          <span className="text-muted-foreground inline-flex items-center gap-1 text-[10px]">
            <TrendingUp className="h-2.5 w-2.5" />
            {marketCount} market{marketCount !== 1 ? 's' : ''}
          </span>
        )}
      </div>

      {/* Ticker */}
      <div className="text-muted-foreground/60 mt-2 truncate font-mono text-[10px]">
        {event.event_ticker}
      </div>
    </button>
  );
}

function formatStrikeDate(dateStr: string): string {
  try {
    const date = new Date(dateStr);
    const now = new Date();
    const diffMs = date.getTime() - now.getTime();
    const diffDays = Math.ceil(diffMs / (1000 * 60 * 60 * 24));

    if (diffDays < 0) return 'Past';
    if (diffDays === 0) return 'Today';
    if (diffDays === 1) return 'Tomorrow';
    if (diffDays < 7) return `${diffDays}d`;

    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
  } catch {
    return dateStr;
  }
}
