import { TrendingUp, BarChart3 } from 'lucide-react';
import { Badge } from '@/components/atoms/Badge';
import { ProbabilityBar } from '@/components/atoms/ProbabilityBar';
import { formatVolume, formatAmericanOdds, type TransformedMarket } from '@/lib/marketTransform';

interface MarketCardProps {
  market: TransformedMarket;
  onClick: () => void;
}

export function MarketCard({ market, onClick }: MarketCardProps) {
  const isPositive = market.change24h >= 0;
  const { sides } = market;

  // For sports markets with two teams, show team-vs-team layout
  const hasTwoSides = sides.hasTeams && sides.teamA && sides.teamB;

  return (
    <button
      onClick={onClick}
      className="bg-card border-border/50 hover:border-border w-full cursor-pointer rounded-xl border p-4 text-left transition-all hover:shadow-lg"
    >
      {/* Header */}
      <div className="mb-3 flex items-start justify-between">
        <div className="min-w-0 flex-1">
          <div className="mb-1 flex items-center gap-2">
            <span className="text-muted-foreground font-mono text-xs">{market.ticker}</span>
            {market.status === 'live' && (
              <Badge
                variant="success"
                size="xs"
              >
                Live
              </Badge>
            )}
          </div>
          {hasTwoSides ? (
            <h3 className="text-sm font-semibold">
              {sides.teamA} <span className="text-muted-foreground font-normal">@</span>{' '}
              {sides.teamB}
            </h3>
          ) : (
            <h3 className="line-clamp-2 text-sm font-semibold">{market.title}</h3>
          )}
          {market.subtitle && (
            <p className="text-muted-foreground mt-0.5 truncate text-xs">
              {sides.hasTeams ? `YES = ${market.subtitle} wins` : market.subtitle}
            </p>
          )}
        </div>
      </div>

      {/* Probability bar */}
      <div className="mb-3">
        <ProbabilityBar
          probability={market.yesPrice}
          height={6}
          className="mb-2"
        />

        {/* Team odds row */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <span
              className="max-w-[100px] truncate text-xs font-medium text-green-400"
              title={sides.yesLabel}
            >
              {sides.yesLabel}
            </span>
            <span className="text-sm font-semibold text-green-400">
              {formatAmericanOdds(market.yesOdds)}
            </span>
            <span className="text-muted-foreground text-xs">({market.yesPrice}%)</span>
          </div>

          <div className="flex items-center gap-2">
            <span className="text-muted-foreground text-xs">({market.noPrice}%)</span>
            <span className="text-sm font-semibold text-red-400">
              {formatAmericanOdds(market.noOdds)}
            </span>
            <span
              className="max-w-[100px] truncate text-xs font-medium text-red-400"
              title={sides.noLabel}
            >
              {sides.noLabel}
            </span>
          </div>
        </div>
      </div>

      {/* Stats */}
      <div className="flex items-center justify-between text-xs">
        <div className="text-muted-foreground flex items-center gap-1">
          <span className={isPositive ? 'text-green-500' : 'text-red-500'}>
            {isPositive ? (
              <TrendingUp size={12} />
            ) : (
              <TrendingUp
                size={12}
                className="rotate-180"
              />
            )}
          </span>
          <span className="font-medium">
            {isPositive ? '+' : ''}
            {market.change24h.toFixed(1)}%
          </span>
          <span className="text-muted-foreground/60">24h</span>
        </div>

        <div className="text-muted-foreground flex items-center gap-1">
          <BarChart3 size={12} />
          <span>{formatVolume(market.volume24h)}</span>
        </div>

        <div className="text-muted-foreground">{market.expiresFormatted}</div>
      </div>
    </button>
  );
}
