import { useEffect, useRef, useState } from 'react';
import { Loader2 } from 'lucide-react';
import { cn } from '@/lib/utils';

interface DecreaseOrderQuantityEditorProps {
  currentQuantity: number;
  minQuantity?: number;
  onSubmit: (nextQuantity: number) => Promise<void> | void;
  className?: string;
}

export function DecreaseOrderQuantityEditor({
  currentQuantity,
  minQuantity = 0,
  onSubmit,
  className,
}: DecreaseOrderQuantityEditorProps) {
  const rootRef = useRef<HTMLDivElement | null>(null);
  const [isEditing, setIsEditing] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [statusMessage, setStatusMessage] = useState<string | null>(null);
  const [draftText, setDraftText] = useState(currentQuantity.toString());
  const inputRef = useRef<HTMLInputElement | null>(null);

  const parsedDraft = Number.parseInt(draftText.replace(/[^0-9]/g, ''), 10);
  const draftValue = Number.isFinite(parsedDraft) ? parsedDraft : null;
  const isValidRange =
    draftValue !== null && draftValue >= minQuantity && draftValue < currentQuantity;
  const percentOfOriginal =
    draftValue !== null && currentQuantity > 0
      ? Math.round((draftValue / currentQuantity) * 100)
      : 0;

  useEffect(() => {
    if (!isEditing) setDraftText(currentQuantity.toString());
  }, [currentQuantity, isEditing]);

  useEffect(() => {
    if (!isEditing) return;
    inputRef.current?.focus();
    inputRef.current?.select();
  }, [isEditing]);

  useEffect(() => {
    if (!isEditing) return;

    const onPointerDown = (event: PointerEvent) => {
      if (isSaving) return;
      const target = event.target as Node | null;
      if (!target) return;
      if (rootRef.current?.contains(target)) return;
      resetEditor();
    };

    document.addEventListener('pointerdown', onPointerDown, true);
    return () => document.removeEventListener('pointerdown', onPointerDown, true);
  }, [isEditing, currentQuantity, isSaving]);

  const stepForEvent = (e: { shiftKey: boolean; ctrlKey: boolean; metaKey: boolean }) => {
    if (e.ctrlKey || e.metaKey) return 1000;
    if (e.shiftKey) return 100;
    return 1;
  };

  const resetEditor = () => {
    setDraftText(currentQuantity.toString());
    setStatusMessage(null);
    setIsEditing(false);
  };

  const handleSave = async () => {
    if (isSaving) return;

    // Treat "unchanged" as a confirmed no-op and exit edit mode.
    if (draftValue === currentQuantity) {
      setStatusMessage(null);
      setIsEditing(false);
      return;
    }

    if (!isValidRange) {
      setStatusMessage('Enter a smaller quantity to decrease this order.');
      return;
    }
    const nextQuantity = draftValue;
    if (nextQuantity === null) return;
    setIsSaving(true);
    setStatusMessage(null);
    try {
      await onSubmit(nextQuantity);
      setStatusMessage(null);
      setIsEditing(false);
    } catch (err) {
      const message =
        err instanceof Error && err.message
          ? err.message
          : 'Unable to decrease order quantity (check permissions).';
      setStatusMessage(message);
    } finally {
      setIsSaving(false);
    }
  };

  const handleFormSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    void handleSave();
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    // Keep this feeling like a clean numeric input while allowing typed commas/spaces.
    const sanitized = e.target.value.replace(/[^0-9]/g, '');
    setStatusMessage(null);
    setDraftText(sanitized);
  };

  const applyPercent = (percent: number) => {
    const next = Math.max(
      minQuantity,
      Math.min(currentQuantity, Math.floor(currentQuantity * percent))
    );
    setStatusMessage(null);
    setDraftText(String(next));
    inputRef.current?.focus();
  };

  useEffect(() => {
    if (!isEditing) return;

    const onWindowKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape') {
        event.preventDefault();
        event.stopPropagation();
        resetEditor();
        return;
      }

      if (event.key === 'Enter') {
        event.preventDefault();
        event.stopPropagation();
        void handleSave();
        return;
      }

      if (event.key === 'ArrowUp' || event.key === 'ArrowDown') {
        event.preventDefault();
        event.stopPropagation();
        const delta = stepForEvent(event);
        const baseValue = draftValue ?? minQuantity;
        const next = event.key === 'ArrowUp' ? baseValue + delta : baseValue - delta;
        const clamped = Math.max(minQuantity, Math.min(currentQuantity, next));
        setDraftText(String(clamped));
      }
    };

    window.addEventListener('keydown', onWindowKeyDown, true);
    return () => window.removeEventListener('keydown', onWindowKeyDown, true);
  }, [isEditing, draftValue, minQuantity, currentQuantity, isSaving]);

  if (!isEditing) {
    return (
      <div
        className={cn('flex h-8 w-[220px] items-center justify-end', className)}
        title="Queue-safe decrease only"
      >
        <button
          type="button"
          onClick={() => {
            setDraftText(currentQuantity.toString());
            setStatusMessage(null);
            setIsEditing(true);
          }}
          className="cursor-pointer font-mono text-xs"
          title="Edit quantity"
          aria-label="Edit quantity"
        >
          {currentQuantity.toLocaleString()}
        </button>
      </div>
    );
  }

  return (
    <div
      ref={rootRef}
      className={cn('flex h-8 w-[220px] items-center justify-end', className)}
      title="Arrow Up/Down: +/-1 · Shift: 100 · Ctrl/Cmd: 1000 · Enter: Save · Esc: Cancel"
    >
      <div className="relative inline-flex items-center">
        <div className="border-border bg-card absolute -top-16 right-0 z-20 translate-x-[38%] rounded border px-2 py-1 shadow-sm">
          <div className="text-muted-foreground flex items-center gap-2 whitespace-nowrap font-mono text-[9px]">
            <span>↑/↓:1</span>
            <span>Shift:100</span>
            <span>Ctrl/Cmd:1000</span>
            <span>Enter:Save</span>
            <span>Esc:Cancel</span>
          </div>
          {isSaving && (
            <div className="mt-1 flex items-center gap-1 font-mono text-[9px] text-cyan-300">
              <Loader2 className="h-3 w-3 animate-spin" />
              <span>Submitting...</span>
            </div>
          )}
          {!isSaving && statusMessage && (
            <div
              className="mt-1 max-w-[360px] truncate font-mono text-[9px] text-yellow-300"
              role="status"
              aria-live="polite"
            >
              {statusMessage}
            </div>
          )}
        </div>
        <div className="border-border bg-card absolute -top-8 left-1/2 z-10 -translate-x-1/2 rounded border px-1.5 py-1 shadow-sm">
          <div className="inline-flex items-center gap-1">
            <button
              type="button"
              onClick={() => applyPercent(0.25)}
              className="hover:bg-muted rounded px-1.5 py-0.5 font-mono text-[10px]"
            >
              25%
            </button>
            <button
              type="button"
              onClick={() => applyPercent(0.5)}
              className="hover:bg-muted rounded px-1.5 py-0.5 font-mono text-[10px]"
            >
              50%
            </button>
            <button
              type="button"
              onClick={() => applyPercent(0.75)}
              className="hover:bg-muted rounded px-1.5 py-0.5 font-mono text-[10px]"
            >
              75%
            </button>
            <span className="text-muted-foreground border-border ml-1 border-l pl-1.5 font-mono text-[10px]">
              {percentOfOriginal}%
            </span>
          </div>
        </div>
        <form onSubmit={handleFormSubmit}>
          <div className="border-border bg-muted inline-flex h-7 items-center overflow-hidden rounded border px-2">
            <input
              ref={inputRef}
              type="text"
              inputMode="numeric"
              value={draftText}
              onChange={handleInputChange}
              className="w-[92px] border-0 bg-transparent text-right font-mono text-xs outline-none"
              aria-label="Order quantity"
            />
          </div>
        </form>
      </div>
    </div>
  );
}

export default DecreaseOrderQuantityEditor;
