/**
 * NumericStepper - Input with +/- buttons for quantity selection
 */

import { Minus, Plus } from 'lucide-react';
import { cn } from '@/lib/utils';

interface NumericStepperProps {
  value: number;
  onChange: (value: number) => void;
  min?: number;
  max?: number;
  step?: number;
  label?: string;
  suffix?: string;
  size?: 'sm' | 'md' | 'lg';
  disabled?: boolean;
  className?: string;
}

export function NumericStepper({
  value,
  onChange,
  min = 0,
  max = Infinity,
  step = 1,
  label,
  suffix,
  size = 'md',
  disabled = false,
  className,
}: NumericStepperProps) {
  const handleIncrement = () => {
    const newValue = Math.min(max, value + step);
    onChange(newValue);
  };

  const handleDecrement = () => {
    const newValue = Math.max(min, value - step);
    onChange(newValue);
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const newValue = parseFloat(e.target.value);
    if (!isNaN(newValue)) {
      onChange(Math.max(min, Math.min(max, newValue)));
    }
  };

  const sizeClasses = {
    sm: { container: 'h-8', button: 'w-7', input: 'text-sm', icon: 14 },
    md: { container: 'h-10', button: 'w-9', input: 'text-base', icon: 16 },
    lg: { container: 'h-12', button: 'w-11', input: 'text-lg', icon: 18 },
  };

  const sizes = sizeClasses[size];

  return (
    <div className={cn('space-y-1', className)}>
      {label && <label className="text-muted-foreground text-xs font-medium">{label}</label>}
      <div
        className={cn(
          'bg-muted border-border flex items-stretch overflow-hidden rounded-lg border',
          sizes.container,
          disabled && 'opacity-50'
        )}
      >
        <button
          onClick={handleDecrement}
          disabled={disabled || value <= min}
          className={cn(
            'hover:bg-muted-foreground/10 flex items-center justify-center bg-transparent transition-colors disabled:cursor-not-allowed disabled:opacity-50',
            sizes.button
          )}
        >
          <Minus size={sizes.icon} />
        </button>

        <div className="border-border flex flex-1 items-center justify-center border-x">
          <input
            type="number"
            value={value}
            onChange={handleInputChange}
            disabled={disabled}
            className={cn(
              'w-full bg-transparent text-center font-medium outline-none [appearance:textfield] [&::-webkit-inner-spin-button]:appearance-none [&::-webkit-outer-spin-button]:appearance-none',
              sizes.input
            )}
          />
          {suffix && <span className="text-muted-foreground pr-2 text-sm">{suffix}</span>}
        </div>

        <button
          onClick={handleIncrement}
          disabled={disabled || value >= max}
          className={cn(
            'hover:bg-muted-foreground/10 flex items-center justify-center bg-transparent transition-colors disabled:cursor-not-allowed disabled:opacity-50',
            sizes.button
          )}
        >
          <Plus size={sizes.icon} />
        </button>
      </div>
    </div>
  );
}

// Compact inline stepper
interface InlineStepperProps {
  value: number;
  onChange: (value: number) => void;
  min?: number;
  max?: number;
  step?: number;
  disabled?: boolean;
  className?: string;
}

export function InlineStepper({
  value,
  onChange,
  min = 0,
  max = Infinity,
  step = 1,
  disabled = false,
  className,
}: InlineStepperProps) {
  return (
    <div className={cn('inline-flex items-center gap-1', className)}>
      <button
        onClick={() => onChange(Math.max(min, value - step))}
        disabled={disabled || value <= min}
        className="bg-muted hover:bg-muted-foreground/20 flex h-6 w-6 items-center justify-center rounded transition-colors disabled:opacity-50"
      >
        <Minus size={12} />
      </button>
      <span className="w-10 text-center text-sm font-medium">{value}</span>
      <button
        onClick={() => onChange(Math.min(max, value + step))}
        disabled={disabled || value >= max}
        className="bg-muted hover:bg-muted-foreground/20 flex h-6 w-6 items-center justify-center rounded transition-colors disabled:opacity-50"
      >
        <Plus size={12} />
      </button>
    </div>
  );
}

// Contract quantity input with common presets
interface ContractQuantityInputProps {
  value: number;
  onChange: (value: number) => void;
  maxContracts?: number;
  presets?: number[];
  className?: string;
}

export function ContractQuantityInput({
  value,
  onChange,
  maxContracts = 10000,
  presets = [10, 50, 100, 500, 1000],
  className,
}: ContractQuantityInputProps) {
  return (
    <div className={cn('space-y-2', className)}>
      <NumericStepper
        value={value}
        onChange={onChange}
        min={1}
        max={maxContracts}
        step={1}
        label="Contracts"
      />
      <div className="flex gap-1">
        {presets.map((preset) => (
          <button
            key={preset}
            onClick={() => onChange(preset)}
            className={cn(
              'h-6 flex-1 rounded text-xs transition-colors',
              value === preset
                ? 'bg-primary/20 text-primary'
                : 'bg-muted text-muted-foreground hover:bg-muted-foreground/20'
            )}
          >
            {preset >= 1000 ? `${preset / 1000}K` : preset}
          </button>
        ))}
      </div>
    </div>
  );
}
