/**
 * OrderBookLevel - Single level in order book
 *
 * Clicking a bid level sets up a sell order at that price.
 * Clicking an ask level sets up a buy order at that price.
 */

import { cn } from '@/lib/utils';
import { PriceText } from '../atoms/PriceText';

interface OrderBookLevelProps {
  price: number;
  quantity: number;
  cumulative: number;
  maxCumulative: number;
  side: 'bid' | 'ask';
  isFlashing?: boolean;
  className?: string;
  onClick?: (
    price: number,
    quantity: number,
    cumulative: number,
    side: 'bid' | 'ask',
    shiftKey: boolean
  ) => void;
}

export function OrderBookLevel({
  price,
  quantity,
  cumulative,
  maxCumulative,
  side,
  isFlashing,
  className,
  onClick,
}: OrderBookLevelProps) {
  const isBid = side === 'bid';
  const depthPercent = (cumulative / maxCumulative) * 100;

  const handleClick = (e: React.MouseEvent) => {
    onClick?.(price, quantity, cumulative, side, e.shiftKey);
  };

  // Format dollar amounts compactly
  const formatDollars = (cents: number, contracts: number): string => {
    const dollars = (cents * contracts) / 100;
    if (dollars >= 1_000_000) return `$${(dollars / 1_000_000).toFixed(1)}M`;
    if (dollars >= 10_000) return `$${(dollars / 1_000).toFixed(0)}K`;
    if (dollars >= 1_000) return `$${(dollars / 1_000).toFixed(1)}K`;
    return `$${Math.round(dollars).toLocaleString()}`;
  };

  // Format large numbers compactly to prevent overflow
  const formatQuantity = (num: number): string => {
    if (num >= 1_000_000) {
      return `${(num / 1_000_000).toFixed(1)}M`;
    }
    if (num >= 100_000) {
      return `${(num / 1_000).toFixed(0)}K`;
    }
    return num.toLocaleString();
  };

  return (
    <div
      onClick={handleClick}
      className={cn(
        'relative flex items-center px-2 py-1 font-mono text-xs',
        'group cursor-pointer transition-all duration-150',
        // Strong hover effect with colored background and glow
        isBid
          ? 'hover:bg-green-500/25 hover:shadow-[inset_0_0_12px_rgba(34,197,94,0.3)]'
          : 'hover:bg-red-500/25 hover:shadow-[inset_0_0_12px_rgba(239,68,68,0.3)]',
        // Flash effect when level changes
        isFlashing && (isBid ? 'animate-flash-up' : 'animate-flash-down'),
        className
      )}
    >
      {/* Depth bar */}
      <div
        className={cn(
          'pointer-events-none absolute inset-y-0 transition-colors duration-150',
          isBid
            ? 'right-0 bg-green-500/15 group-hover:bg-green-500/30'
            : 'left-0 bg-red-500/15 group-hover:bg-red-500/30'
        )}
        style={{ width: `${depthPercent}%` }}
      />

      {/* Hover indicator bar */}
      <div
        className={cn(
          'absolute inset-y-0 w-1 opacity-0 transition-opacity duration-150 group-hover:opacity-100',
          isBid ? 'left-0 bg-green-500' : 'right-0 bg-red-500'
        )}
      />

      {/* Content - using CSS Grid for precise column alignment */}
      <div
        className="relative grid w-full gap-1 tabular-nums"
        style={{
          gridTemplateColumns: isBid
            ? 'minmax(28px, auto) 1fr minmax(42px, auto) minmax(48px, auto)' // price | qty | $cost | $depth
            : 'minmax(48px, auto) minmax(42px, auto) 1fr minmax(28px, auto)', // $depth | $cost | qty | price
        }}
      >
        {isBid ? (
          <>
            <PriceText
              value={price}
              size="sm"
              showSymbol={false}
              className="font-semibold text-green-500 transition-colors group-hover:text-green-400"
            />
            <span className="text-muted-foreground group-hover:text-foreground truncate text-right transition-colors">
              {formatQuantity(quantity)}
            </span>
            <span className="text-muted-foreground/70 group-hover:text-muted-foreground truncate text-right transition-colors">
              {formatDollars(price, quantity)}
            </span>
            <span className="text-muted-foreground/50 group-hover:text-muted-foreground truncate text-right transition-colors">
              {formatDollars(price, cumulative)}
            </span>
          </>
        ) : (
          <>
            <span className="text-muted-foreground/50 group-hover:text-muted-foreground truncate transition-colors">
              {formatDollars(price, cumulative)}
            </span>
            <span className="text-muted-foreground/70 group-hover:text-muted-foreground truncate transition-colors">
              {formatDollars(price, quantity)}
            </span>
            <span className="text-muted-foreground group-hover:text-foreground truncate transition-colors">
              {formatQuantity(quantity)}
            </span>
            <PriceText
              value={price}
              size="sm"
              showSymbol={false}
              className="text-right font-semibold text-red-500 transition-colors group-hover:text-red-400"
            />
          </>
        )}
      </div>
    </div>
  );
}

export default OrderBookLevel;
