/**
 * SearchInput - Search input with optional filters and suggestions
 */

import { useState, useRef, useEffect } from 'react';
import { Search, X, Filter, ChevronDown } from 'lucide-react';
import { cn } from '@/lib/utils';
import { Dropdown } from '@/components/atoms/Dropdown';

interface FilterOption {
  id: string;
  label: string;
  options: { value: string; label: string }[];
}

interface SearchInputProps {
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
  filters?: FilterOption[];
  activeFilters?: Record<string, string>;
  onFilterChange?: (filterId: string, value: string) => void;
  suggestions?: string[];
  onSuggestionSelect?: (suggestion: string) => void;
  onSubmit?: (value: string) => void;
  loading?: boolean;
  size?: 'sm' | 'md' | 'lg';
  className?: string;
}

export function SearchInput({
  value,
  onChange,
  placeholder = 'Search...',
  filters,
  activeFilters = {},
  onFilterChange,
  suggestions,
  onSuggestionSelect,
  onSubmit,
  loading = false,
  size = 'md',
  className,
}: SearchInputProps) {
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [showFilters, setShowFilters] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);
  const containerRef = useRef<HTMLDivElement>(null);

  const sizeClasses = {
    sm: 'h-8 text-sm',
    md: 'h-10 text-sm',
    lg: 'h-12 text-base',
  };

  const iconSizes = {
    sm: 14,
    md: 16,
    lg: 18,
  };

  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      if (containerRef.current && !containerRef.current.contains(e.target as Node)) {
        setShowSuggestions(false);
        setShowFilters(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      onSubmit?.(value);
      setShowSuggestions(false);
    }
    if (e.key === 'Escape') {
      setShowSuggestions(false);
      inputRef.current?.blur();
    }
  };

  const activeFilterCount = Object.values(activeFilters).filter(Boolean).length;

  return (
    <div
      ref={containerRef}
      className={cn('relative', className)}
    >
      <div
        className={cn(
          'bg-card border-border focus-within:border-primary flex items-center gap-2 rounded-lg border px-3 transition-colors',
          sizeClasses[size]
        )}
      >
        <Search
          size={iconSizes[size]}
          className="text-muted-foreground shrink-0"
        />

        <input
          ref={inputRef}
          type="text"
          value={value}
          onChange={(e) => onChange(e.target.value)}
          onFocus={() => suggestions && setShowSuggestions(true)}
          onKeyDown={handleKeyDown}
          placeholder={placeholder}
          className="placeholder:text-muted-foreground min-w-0 flex-1 bg-transparent outline-none"
        />

        {loading && (
          <div className="border-muted-foreground/30 border-t-primary h-4 w-4 animate-spin rounded-full border-2" />
        )}

        {value && !loading && (
          <button
            onClick={() => onChange('')}
            className="hover:bg-muted text-muted-foreground rounded p-1 transition-colors"
          >
            <X size={iconSizes[size] - 2} />
          </button>
        )}

        {filters && filters.length > 0 && (
          <button
            onClick={() => setShowFilters(!showFilters)}
            className={cn(
              'flex items-center gap-1 rounded px-2 py-1 transition-colors',
              showFilters || activeFilterCount > 0
                ? 'bg-primary/10 text-primary'
                : 'hover:bg-muted text-muted-foreground'
            )}
          >
            <Filter size={iconSizes[size] - 2} />
            {activeFilterCount > 0 && (
              <span className="text-xs font-medium">{activeFilterCount}</span>
            )}
            <ChevronDown size={12} />
          </button>
        )}
      </div>

      {/* Suggestions dropdown */}
      {showSuggestions && suggestions && suggestions.length > 0 && (
        <div className="bg-card border-border absolute left-0 right-0 top-full z-20 mt-1 overflow-hidden rounded-lg border shadow-lg">
          {suggestions.map((suggestion, i) => (
            <button
              key={i}
              onClick={() => {
                onSuggestionSelect?.(suggestion);
                onChange(suggestion);
                setShowSuggestions(false);
              }}
              className="hover:bg-muted w-full px-3 py-2 text-left text-sm transition-colors"
            >
              {suggestion}
            </button>
          ))}
        </div>
      )}

      {/* Filters dropdown */}
      {showFilters && filters && (
        <div className="bg-card border-border absolute left-0 right-0 top-full z-20 mt-1 rounded-lg border p-3 shadow-lg">
          <div className="grid gap-3">
            {filters.map((filter) => (
              <Dropdown
                key={filter.id}
                label={filter.label}
                options={[{ value: '', label: 'All' }, ...filter.options]}
                value={activeFilters[filter.id] || ''}
                onChange={(val) => onFilterChange?.(filter.id, val)}
                size="sm"
              />
            ))}
          </div>
          {activeFilterCount > 0 && (
            <button
              onClick={() => {
                filters.forEach((f) => onFilterChange?.(f.id, ''));
              }}
              className="text-primary mt-3 text-xs hover:underline"
            >
              Clear all filters
            </button>
          )}
        </div>
      )}
    </div>
  );
}
