/**
 * TabNav - Horizontal tab navigation
 */

import { useRef, useState, useEffect } from 'react';
import { cn } from '@/lib/utils';
import { ChevronLeft, ChevronRight } from 'lucide-react';

interface Tab {
  id: string;
  label: string;
  icon?: React.ReactNode;
  badge?: number | string;
  disabled?: boolean;
}

interface TabNavProps {
  tabs: Tab[];
  activeTab: string;
  onChange: (tabId: string) => void;
  variant?: 'underline' | 'pills' | 'boxed' | 'buttons';
  size?: 'sm' | 'md' | 'lg';
  fullWidth?: boolean;
  className?: string;
}

export function TabNav({
  tabs,
  activeTab,
  onChange,
  variant = 'underline',
  size = 'md',
  fullWidth = false,
  className,
}: TabNavProps) {
  const sizeClasses = {
    sm: 'text-xs h-8 px-2',
    md: 'text-sm h-10 px-3',
    lg: 'text-base h-12 px-4',
  };

  const variantStyles = {
    underline: {
      container: 'border-b border-border',
      tab: 'border-b-2 -mb-px rounded-none',
      active: 'border-primary text-foreground',
      inactive:
        'border-transparent text-muted-foreground hover:text-foreground hover:border-border',
    },
    pills: {
      container: 'gap-1',
      tab: 'rounded-full',
      active: 'bg-primary text-primary-foreground',
      inactive: 'text-muted-foreground hover:bg-muted hover:text-foreground',
    },
    boxed: {
      container: 'bg-muted rounded-lg p-1 gap-1',
      tab: 'rounded-md',
      active: 'bg-card text-foreground shadow-sm',
      inactive: 'text-muted-foreground hover:text-foreground',
    },
    buttons: {
      container: 'gap-1',
      tab: 'rounded-lg border border-border',
      active: 'bg-primary/10 border-primary text-primary',
      inactive: 'text-muted-foreground hover:bg-muted hover:text-foreground',
    },
  };

  const styles = variantStyles[variant];

  return (
    <div className={cn('flex items-center', styles.container, className)}>
      {tabs.map((tab) => (
        <button
          key={tab.id}
          onClick={() => !tab.disabled && onChange(tab.id)}
          disabled={tab.disabled}
          className={cn(
            'inline-flex items-center justify-center gap-1.5 font-medium transition-all',
            sizeClasses[size],
            styles.tab,
            tab.id === activeTab ? styles.active : styles.inactive,
            tab.disabled && 'cursor-not-allowed opacity-50',
            fullWidth && 'flex-1'
          )}
        >
          {tab.icon}
          <span>{tab.label}</span>
          {tab.badge !== undefined && (
            <span
              className={cn(
                'ml-1 rounded-full px-1.5 py-0.5 text-[10px] font-semibold',
                tab.id === activeTab
                  ? 'bg-primary-foreground/20 text-primary-foreground'
                  : 'bg-muted-foreground/20 text-muted-foreground'
              )}
            >
              {tab.badge}
            </span>
          )}
        </button>
      ))}
    </div>
  );
}

/**
 * ScrollableTabNav - Scrollable tab nav for many tabs
 */
interface ScrollableTabNavProps extends Omit<TabNavProps, 'fullWidth'> {
  showArrows?: boolean;
}

export function ScrollableTabNav({
  tabs,
  activeTab,
  onChange,
  variant = 'underline',
  size = 'md',
  showArrows = true,
  className,
}: ScrollableTabNavProps) {
  const scrollRef = useRef<HTMLDivElement>(null);
  const [canScrollLeft, setCanScrollLeft] = useState(false);
  const [canScrollRight, setCanScrollRight] = useState(false);

  const checkScroll = () => {
    if (scrollRef.current) {
      const { scrollLeft, scrollWidth, clientWidth } = scrollRef.current;
      setCanScrollLeft(scrollLeft > 0);
      setCanScrollRight(scrollLeft < scrollWidth - clientWidth - 1);
    }
  };

  useEffect(() => {
    checkScroll();
    window.addEventListener('resize', checkScroll);
    return () => window.removeEventListener('resize', checkScroll);
  }, [tabs]);

  const scroll = (direction: 'left' | 'right') => {
    if (scrollRef.current) {
      const scrollAmount = scrollRef.current.clientWidth * 0.5;
      scrollRef.current.scrollBy({
        left: direction === 'left' ? -scrollAmount : scrollAmount,
        behavior: 'smooth',
      });
    }
  };

  return (
    <div className={cn('relative flex items-center', className)}>
      {showArrows && canScrollLeft && (
        <button
          onClick={() => scroll('left')}
          className="from-background absolute left-0 z-10 flex h-full w-8 items-center justify-start bg-gradient-to-r to-transparent"
        >
          <ChevronLeft
            size={16}
            className="text-muted-foreground"
          />
        </button>
      )}

      <div
        ref={scrollRef}
        onScroll={checkScroll}
        className="scrollbar-hide flex-1 overflow-x-auto"
      >
        <TabNav
          tabs={tabs}
          activeTab={activeTab}
          onChange={onChange}
          variant={variant}
          size={size}
        />
      </div>

      {showArrows && canScrollRight && (
        <button
          onClick={() => scroll('right')}
          className="from-background absolute right-0 z-10 flex h-full w-8 items-center justify-end bg-gradient-to-l to-transparent"
        >
          <ChevronRight
            size={16}
            className="text-muted-foreground"
          />
        </button>
      )}
    </div>
  );
}
