/**
 * CombinedOrdersPositionsPanel - Unified view of orders and positions
 *
 * Prototype panel that combines open orders and filled positions into a single
 * table, grouped by (ticker, side).
 */

import { useMemo, useState } from 'react';
import { cn } from '@/lib/utils';
import { ChevronDown, ChevronUp, X } from 'lucide-react';
import type { KalshiOrder } from '@/lib/kalshiApi';
import type { Position } from '@/types';
import { formatDualPriceCents } from '@/lib/nbaConsolidated/format';
import {
  parseKalshiNbaMarketTicker,
  parseKalshiNbaEventTicker,
} from '@/lib/nbaConsolidated/kalshiNbaTicker';
import { TickerDisplay } from '@/components/atoms/TickerDisplay';
import { Money } from '@/components/atoms/Money';
import { DateDisplay } from '@/components/atoms/DateDisplay';
import { Th, Td } from '@/components/atoms/TableCells';
import { ConfirmModal } from '@/components/atoms/Modal';

/** Orderbook best bid/ask for a single market ticker */
interface TickerOrderbook {
  /** Best YES bid price in cents */
  yesBestBid: number | null;
  /** Best YES ask price in cents (derived: 100 - best NO bid) */
  yesBestAsk: number | null;
  /** Best NO bid price in cents */
  noBestBid: number | null;
  /** Best NO ask price in cents (derived: 100 - best YES bid) */
  noBestAsk: number | null;
}

interface CombinedOrdersPositionsPanelProps {
  kalshiOrders: KalshiOrder[];
  positions: Position[];
  /** Optional mapping from ticker -> market title */
  titleByTicker?: Record<string, string>;
  /** Optional mapping from ticker -> game date (ISO string) */
  gameDateByTicker?: Record<string, string>;
  /** Optional mapping from ticker -> { yes/no outcome label } */
  outcomeByTicker?: Record<string, { yes?: string; no?: string }>;
  /** Optional mapping from ticker -> orderbook best bid/ask */
  orderbookByTicker?: Record<string, TickerOrderbook>;
  /** Handler for canceling an order */
  onCancelOrder?: (orderId: string) => void;
  /** Handler for killing all orders */
  onKillAll?: () => void;
  className?: string;
}

interface UnifiedRow {
  // Grouping key
  key: string;
  ticker: string;
  side: 'yes' | 'no';

  // Display fields
  gameDate: string | null;
  gameDescription: string;
  positionLabel: string;

  // Order fields
  orderId: string | null;
  orderAction: 'buy' | 'sell' | null;
  qtyOpen: number;
  queuePosition: number | null;
  orderPriceCents: number | null;
  /** Whether order is at top of book (best price). null if no order or no orderbook data */
  topOfBook: boolean | null;

  // Position fields
  qtyClosed: number;
  avgPriceCents: number | null;
  currentPriceCents: number | null;
  pnl: number | null;
}

function deriveGameDescription(ticker: string, title?: string): string {
  // Try to parse from Kalshi ticker
  const marketInfo = parseKalshiNbaMarketTicker(ticker);
  if (marketInfo) {
    const eventInfo = parseKalshiNbaEventTicker(marketInfo.eventTicker);
    if (eventInfo) {
      return `NBA: ${eventInfo.awayCode} at ${eventInfo.homeCode}`;
    }
  }
  // Fall back to title or ticker
  return title || ticker;
}

function derivePositionLabel(
  ticker: string,
  side: 'yes' | 'no',
  outcomeByTicker?: Record<string, { yes?: string; no?: string }>
): string {
  // Check explicit mapping first
  const outcome = outcomeByTicker?.[ticker]?.[side];
  if (outcome) return outcome;

  // Try to parse from Kalshi ticker
  const marketInfo = parseKalshiNbaMarketTicker(ticker);
  if (marketInfo) {
    const eventInfo = parseKalshiNbaEventTicker(marketInfo.eventTicker);
    if (eventInfo) {
      // The market ticker contains the team code for this contract
      // YES on this market = that team wins
      // NO on this market = the other team wins
      const isAwayTeam = marketInfo.teamCode === eventInfo.awayCode;
      const teamCode =
        side === 'yes' ? marketInfo.teamCode : isAwayTeam ? eventInfo.homeCode : eventInfo.awayCode;
      return `${teamCode} Win`;
    }
  }

  return `${side.toUpperCase()}`;
}

function deriveDateFromTicker(ticker: string): string | null {
  // Try NBA ticker parsing first
  const marketInfo = parseKalshiNbaMarketTicker(ticker);
  if (marketInfo) {
    const eventInfo = parseKalshiNbaEventTicker(marketInfo.eventTicker);
    if (eventInfo) {
      return eventInfo.dateYyyyMmDd;
    }
  }

  // Try to extract date from NFL-style tickers (e.g., KXNFL2TD-26FEB08SEANE-...)
  // Format: YY + 3-letter month + DD (e.g., 26FEB08 = Feb 08, 2026)
  const nflDateMatch = ticker.match(
    /(\d{2})(JAN|FEB|MAR|APR|MAY|JUN|JUL|AUG|SEP|OCT|NOV|DEC)(\d{2})/i
  );
  if (nflDateMatch) {
    const [, yy, monthStr, dd] = nflDateMatch;
    const monthMap: Record<string, string> = {
      JAN: '01',
      FEB: '02',
      MAR: '03',
      APR: '04',
      MAY: '05',
      JUN: '06',
      JUL: '07',
      AUG: '08',
      SEP: '09',
      OCT: '10',
      NOV: '11',
      DEC: '12',
    };
    const mm = monthMap[monthStr.toUpperCase()];
    if (mm) {
      return `20${yy}-${mm}-${dd}`;
    }
  }

  return null;
}

/**
 * Determine if an order is at the top of book (best price).
 * - BUY orders: top if order price >= best bid
 * - SELL orders: top if order price <= best ask
 */
function isAtTopOfBook(args: {
  action: 'buy' | 'sell';
  side: 'yes' | 'no';
  orderPriceCents: number;
  orderbook: TickerOrderbook | undefined;
}): boolean | null {
  const { action, side, orderPriceCents, orderbook } = args;
  if (!orderbook) return null;

  if (action === 'buy') {
    // Buying: compare to best bid for this side
    const bestBid = side === 'yes' ? orderbook.yesBestBid : orderbook.noBestBid;
    if (bestBid === null) return null;
    return orderPriceCents >= bestBid;
  }

  // Selling: compare to best ask for this side
  const bestAsk = side === 'yes' ? orderbook.yesBestAsk : orderbook.noBestAsk;
  if (bestAsk === null) return null;
  return orderPriceCents <= bestAsk;
}

export function CombinedOrdersPositionsPanel({
  kalshiOrders,
  positions,
  titleByTicker,
  gameDateByTicker,
  outcomeByTicker,
  orderbookByTicker,
  onCancelOrder,
  onKillAll,
  className,
}: CombinedOrdersPositionsPanelProps) {
  const unifiedRows = useMemo(() => {
    const rowMap = new Map<string, UnifiedRow>();

    // Process orders first
    for (const order of kalshiOrders) {
      const key = `${order.ticker}:${order.side}`;
      const existing = rowMap.get(key);

      const priceCents = order.side === 'yes' ? order.yes_price : order.no_price;
      const tickerDate = deriveDateFromTicker(order.ticker);
      const gameDate = gameDateByTicker?.[order.ticker] || tickerDate || undefined;

      // Calculate top of book status
      const topOfBook =
        priceCents !== null && priceCents !== undefined
          ? isAtTopOfBook({
              action: order.action,
              side: order.side,
              orderPriceCents: priceCents,
              orderbook: orderbookByTicker?.[order.ticker],
            })
          : null;

      if (existing) {
        // Merge order into existing row
        existing.orderId = order.order_id;
        existing.orderAction = order.action;
        existing.qtyOpen = order.remaining_count;
        existing.queuePosition = order.queue_position ?? null;
        existing.orderPriceCents = priceCents ?? null;
        existing.topOfBook = topOfBook;
      } else {
        rowMap.set(key, {
          key,
          ticker: order.ticker,
          side: order.side,
          gameDate: gameDate ?? null,
          gameDescription: deriveGameDescription(order.ticker, titleByTicker?.[order.ticker]),
          positionLabel: derivePositionLabel(order.ticker, order.side, outcomeByTicker),
          orderId: order.order_id,
          orderAction: order.action,
          qtyOpen: order.remaining_count,
          queuePosition: order.queue_position ?? null,
          orderPriceCents: priceCents ?? null,
          topOfBook,
          qtyClosed: 0,
          avgPriceCents: null,
          currentPriceCents: null,
          pnl: null,
        });
      }
    }

    // Process positions
    for (const pos of positions) {
      const key = `${pos.ticker}:${pos.side}`;
      const existing = rowMap.get(key);

      const tickerDate = deriveDateFromTicker(pos.ticker);
      const gameDate = gameDateByTicker?.[pos.ticker] || tickerDate || undefined;

      // Convert position price from dollars to cents
      const avgCents = pos.averagePrice * 100;
      const currentCents = pos.currentPrice !== null ? pos.currentPrice * 100 : null;

      if (existing) {
        // Merge position into existing row
        existing.qtyClosed = pos.quantity;
        existing.avgPriceCents = avgCents;
        existing.currentPriceCents = currentCents;
        existing.pnl = pos.totalPnl;
      } else {
        rowMap.set(key, {
          key,
          ticker: pos.ticker,
          side: pos.side,
          gameDate: gameDate ?? null,
          gameDescription: deriveGameDescription(pos.ticker, titleByTicker?.[pos.ticker]),
          positionLabel: derivePositionLabel(pos.ticker, pos.side, outcomeByTicker),
          orderId: null,
          orderAction: null,
          qtyOpen: 0,
          queuePosition: null,
          orderPriceCents: null,
          topOfBook: null,
          qtyClosed: pos.quantity,
          avgPriceCents: avgCents,
          currentPriceCents: currentCents,
          pnl: pos.totalPnl,
        });
      }
    }

    // Convert to array and sort by date, then game
    const rows = Array.from(rowMap.values());
    rows.sort((a, b) => {
      const dateA = a.gameDate ?? '';
      const dateB = b.gameDate ?? '';
      const dateCompare = dateA.localeCompare(dateB);
      if (dateCompare !== 0) return dateCompare;
      return a.gameDescription.localeCompare(b.gameDescription);
    });

    return rows;
  }, [
    kalshiOrders,
    positions,
    titleByTicker,
    gameDateByTicker,
    outcomeByTicker,
    orderbookByTicker,
  ]);

  const totalPnl = useMemo(() => {
    return unifiedRows.reduce((sum, row) => sum + (row.pnl ?? 0), 0);
  }, [unifiedRows]);

  const [showAllRows, setShowAllRows] = useState(false);
  const [showKillAllConfirm, setShowKillAllConfirm] = useState(false);
  const maxRows = 10;

  const visibleRows = useMemo(() => {
    if (showAllRows) return unifiedRows;
    return unifiedRows.slice(0, maxRows);
  }, [unifiedRows, showAllRows]);

  const hasOpenOrders = kalshiOrders.length > 0;

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header - clean section title */}
      <div className="border-border flex items-center justify-between border-b px-4 py-4">
        <div className="flex items-center gap-3">
          <h2 className="text-foreground text-lg font-semibold">Orders & Positions</h2>
          <span className="text-muted-foreground text-xs">{unifiedRows.length} rows</span>
        </div>
        <div className="flex items-center gap-4">
          <div className="font-mono text-sm font-semibold">
            Total P&L:{' '}
            <Money
              value={totalPnl}
              variant="pnl"
            />
          </div>
          {hasOpenOrders && onKillAll && (
            <button
              onClick={() => setShowKillAllConfirm(true)}
              className="bg-destructive hover:bg-destructive/90 text-destructive-foreground rounded-md px-4 py-2 text-xs font-medium uppercase tracking-wider shadow-sm transition-colors"
            >
              Kill All
            </button>
          )}

          {/* Kill All Confirmation Modal */}
          <ConfirmModal
            isOpen={showKillAllConfirm}
            onClose={() => setShowKillAllConfirm(false)}
            onConfirm={() => onKillAll?.()}
            title="Cancel All Orders"
            message={`Are you sure you want to cancel all ${kalshiOrders.length} open order${kalshiOrders.length === 1 ? '' : 's'}? This action cannot be undone.`}
            confirmLabel="Kill All"
            cancelLabel="Keep Orders"
            variant="danger"
          />
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <Th>Date</Th>
              <Th>Market</Th>
              <Th>Position</Th>
              <Th className="text-right">Qty Open</Th>
              <Th className="text-right">Qty Closed</Th>
              <Th className="text-right">Queue</Th>
              <Th className="text-center">Top</Th>
              <Th className="text-right">Avg</Th>
              <Th className="text-right">Current</Th>
              <Th className="text-right">P&L</Th>
              <Th className="w-10 text-center"> </Th>
            </tr>
          </thead>
          <tbody className="divide-border divide-y">
            {unifiedRows.length === 0 ? (
              <tr>
                <td
                  colSpan={11}
                  className="text-muted-foreground px-3 py-6 text-center"
                >
                  No orders or positions
                </td>
              </tr>
            ) : (
              visibleRows.map((row, rowIndex) => {
                // Determine which price to show in Avg column
                // If we have a position, show avg fill price; otherwise show order price
                const avgToShow = row.avgPriceCents ?? row.orderPriceCents;
                const zebraStyle =
                  rowIndex % 2 === 1
                    ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
                    : undefined;

                return (
                  <tr
                    key={row.key}
                    className="hover:bg-accent/50 transition-colors"
                    style={zebraStyle}
                  >
                    <Td className="font-mono text-xs">
                      <DateDisplay value={row.gameDate} />
                    </Td>
                    <Td>
                      <TickerDisplay
                        ticker={row.ticker}
                        title={row.gameDescription}
                        variant="compact"
                      />
                    </Td>
                    <Td>
                      <span
                        className={cn(
                          'rounded px-2 py-0.5 text-xs font-medium',
                          row.side === 'yes'
                            ? 'bg-positive/20 text-positive'
                            : 'bg-negative/20 text-negative'
                        )}
                      >
                        {row.positionLabel}
                      </span>
                    </Td>
                    <Td className="text-right font-mono">
                      {row.qtyOpen > 0 ? row.qtyOpen.toLocaleString() : '—'}
                    </Td>
                    <Td className="text-right font-mono">
                      {row.qtyClosed > 0 ? row.qtyClosed.toLocaleString() : '—'}
                    </Td>
                    <Td className="text-muted-foreground text-right font-mono">
                      {row.queuePosition !== null ? row.queuePosition.toLocaleString() : '—'}
                    </Td>
                    <Td className="text-center">
                      {row.topOfBook === null ? (
                        <span className="text-muted-foreground">—</span>
                      ) : row.topOfBook ? (
                        <span className="text-positive">✓</span>
                      ) : (
                        <span className="text-muted-foreground">✗</span>
                      )}
                    </Td>
                    <Td className="text-right font-mono">
                      {avgToShow !== null ? formatDualPriceCents(avgToShow) : '—'}
                    </Td>
                    <Td className="text-right font-mono">
                      {row.currentPriceCents !== null
                        ? formatDualPriceCents(row.currentPriceCents)
                        : '—'}
                    </Td>
                    <Td className="text-right font-mono font-semibold">
                      <Money
                        value={row.pnl}
                        variant="pnl"
                      />
                    </Td>
                    <Td className="text-center">
                      {row.orderId && onCancelOrder && (
                        <button
                          onClick={() => onCancelOrder(row.orderId!)}
                          className="hover:bg-destructive/20 hover:text-destructive rounded p-1 transition-colors"
                          title="Cancel order"
                        >
                          <X className="h-4 w-4" />
                        </button>
                      )}
                    </Td>
                  </tr>
                );
              })
            )}
          </tbody>
        </table>
      </div>

      {/* Show more/less */}
      {unifiedRows.length > maxRows && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
            aria-label={showAllRows ? 'Show fewer rows' : 'Show more rows'}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${unifiedRows.length}`
                : `Showing ${Math.min(maxRows, unifiedRows.length)} of ${unifiedRows.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}
    </div>
  );
}

export default CombinedOrdersPositionsPanel;
