/**
 * DashboardHeader - Controls for the Value Dashboard
 *
 * Contains mode toggles and settings. Feed status is shown in the app header.
 */

import { useState, useRef, useEffect, type ReactNode } from 'react';
import { cn } from '@/lib/utils';
import { Settings, Keyboard, Pencil } from 'lucide-react';
import { Toggle } from '@/components/atoms/Toggle';
import {
  type VolumeWindow,
  ALL_VOLUME_WINDOWS,
  formatWindowLabel,
} from '@/lib/nbaConsolidated/volumeSettings';
import {
  getStartTimeOffsetMinutes,
  setStartTimeOffsetMinutes,
} from '@/lib/nbaConsolidated/startTimeSettings';
import { useHotkeyContext, formatHotkey, type HotkeyId } from '@/lib/hotkeys';
import { HotkeyConfigModal } from '@/components/hotkeys';

export type OddsMode = 'maker' | 'taker';
export type ViewMode = 'two-row' | 'classic';

interface DashboardHeaderProps {
  /** Current odds display mode */
  oddsMode: OddsMode;
  /** Current view mode */
  viewMode: ViewMode;
  /** Handler for mode changes */
  onModeChange: (mode: OddsMode) => void;
  /** Handler for view mode changes */
  onViewModeChange: (mode: ViewMode) => void;
  /** Enabled volume windows */
  volumeWindows?: VolumeWindow[];
  /** Handler for volume window toggle */
  onToggleVolumeWindow?: (window: VolumeWindow) => void;
  /** Content to render on the left side (e.g., sport dropdown, market type tabs) */
  leftContent?: ReactNode;
  className?: string;
}

export function DashboardHeader({
  oddsMode,
  viewMode,
  onModeChange,
  onViewModeChange,
  volumeWindows = [],
  onToggleVolumeWindow,
  leftContent,
  className,
}: DashboardHeaderProps) {
  const [showSettings, setShowSettings] = useState(false);
  const [showHotkeys, setShowHotkeys] = useState(false);
  const [showHotkeyConfig, setShowHotkeyConfig] = useState(false);
  const [startTimeOffset, setStartTimeOffset] = useState(() => getStartTimeOffsetMinutes());
  const dropdownRef = useRef<HTMLDivElement>(null);
  const hotkeysRef = useRef<HTMLDivElement>(null);
  const { getHotkey } = useHotkeyContext();

  // Helper to get formatted hotkey display
  const getHotkeyDisplay = (id: HotkeyId): string => {
    const config = getHotkey(id);
    return config ? formatHotkey(config) : '—';
  };

  const handleOffsetChange = (value: number) => {
    setStartTimeOffset(value);
    setStartTimeOffsetMinutes(value);
  };

  // Close dropdowns when clicking outside
  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setShowSettings(false);
      }
      if (hotkeysRef.current && !hotkeysRef.current.contains(event.target as Node)) {
        setShowHotkeys(false);
      }
    }
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  return (
    <div className={cn('flex items-center justify-between', className)}>
      {/* Left side content (sport dropdown, market type tabs) */}
      <div className="flex items-center gap-4">{leftContent}</div>

      {/* Right side controls */}
      <div className="flex items-center gap-3">
        {/* View mode toggle */}
        <div className="border-border flex items-center overflow-hidden rounded border">
          <button
            onClick={() => onViewModeChange('two-row')}
            className={cn(
              'px-3 py-1.5 text-xs font-medium uppercase tracking-wider transition-colors',
              viewMode === 'two-row'
                ? 'bg-primary text-primary-foreground'
                : 'bg-muted hover:bg-muted/80'
            )}
            title="2 rows per game (one per team)"
          >
            2-Row
          </button>
          <button
            onClick={() => onViewModeChange('classic')}
            className={cn(
              'px-3 py-1.5 text-xs font-medium uppercase tracking-wider transition-colors',
              viewMode === 'classic'
                ? 'bg-primary text-primary-foreground'
                : 'bg-muted hover:bg-muted/80'
            )}
            title="1 row per game (classic view)"
          >
            Classic
          </button>
        </div>

        {/* Odds mode toggle buttons */}
        <div className="border-border flex items-center overflow-hidden rounded border">
          <button
            onClick={() => onModeChange('maker')}
            className={cn(
              'px-3 py-1.5 text-xs font-medium uppercase tracking-wider transition-colors',
              oddsMode === 'maker'
                ? 'bg-primary text-primary-foreground'
                : 'bg-muted hover:bg-muted/80'
            )}
          >
            Maker
          </button>
          <button
            onClick={() => onModeChange('taker')}
            className={cn(
              'px-3 py-1.5 text-xs font-medium uppercase tracking-wider transition-colors',
              oddsMode === 'taker'
                ? 'bg-primary text-primary-foreground'
                : 'bg-muted hover:bg-muted/80'
            )}
          >
            Taker
          </button>
        </div>

        {/* Hotkeys dropdown */}
        <div
          className="relative"
          ref={hotkeysRef}
        >
          <button
            onClick={() => setShowHotkeys((v) => !v)}
            className={cn(
              'border-border rounded border p-1.5 transition-colors',
              showHotkeys ? 'bg-primary text-primary-foreground' : 'bg-muted hover:bg-muted/80'
            )}
            title="Keyboard Shortcuts"
          >
            <Keyboard className="h-4 w-4" />
          </button>

          {showHotkeys && (
            <div className="bg-card border-border absolute right-0 z-50 mt-1 w-52 rounded-lg border shadow-lg">
              <div className="border-border flex items-center justify-between border-b p-2">
                <span className="text-muted-foreground text-xs font-medium uppercase tracking-wider">
                  Keyboard Shortcuts
                </span>
                <button
                  onClick={() => {
                    setShowHotkeys(false);
                    setShowHotkeyConfig(true);
                  }}
                  className="text-muted-foreground hover:text-foreground flex items-center gap-1 text-xs"
                  title="Customize shortcuts"
                >
                  <Pencil className="h-3 w-3" />
                </button>
              </div>
              <div className="space-y-1 p-2 text-sm">
                <div className="text-muted-foreground mb-1 text-[10px] font-medium uppercase tracking-wider">
                  Sport
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">NBA</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-nba')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">NFL</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-nfl')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">NHL</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-nhl')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">CBB</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-cbb')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">ATP</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-atp')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">WTA</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('sport-wta')}
                  </kbd>
                </div>
                <div className="border-border my-2 border-t" />
                <div className="text-muted-foreground mb-1 text-[10px] font-medium uppercase tracking-wider">
                  Market Type
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">Moneyline</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('market-moneyline')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">Spread</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('market-spread')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">Total</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('market-total')}
                  </kbd>
                </div>
                <div className="border-border my-2 border-t" />
                <div className="text-muted-foreground mb-1 text-[10px] font-medium uppercase tracking-wider">
                  Trading
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">New Order</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('new-order')}
                  </kbd>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-muted-foreground">Kill All Orders</span>
                  <kbd className="bg-muted rounded px-1.5 py-0.5 font-mono text-xs">
                    {getHotkeyDisplay('kill-all')}
                  </kbd>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Settings dropdown */}
        <div
          className="relative"
          ref={dropdownRef}
        >
          <button
            onClick={() => setShowSettings((v) => !v)}
            className={cn(
              'border-border rounded border p-1.5 transition-colors',
              showSettings ? 'bg-primary text-primary-foreground' : 'bg-muted hover:bg-muted/80'
            )}
            title="Settings"
          >
            <Settings className="h-4 w-4" />
          </button>

          {showSettings && (
            <div className="bg-card border-border absolute right-0 z-50 mt-1 w-56 rounded-lg border shadow-lg">
              {/* Start Time Offset */}
              <div className="border-border border-b p-2">
                <span className="text-muted-foreground text-xs font-medium uppercase tracking-wider">
                  Start Time Offset
                </span>
              </div>
              <div className="border-border border-b p-2">
                <div className="flex items-center gap-2">
                  <input
                    type="number"
                    value={startTimeOffset}
                    onChange={(e) => handleOffsetChange(parseInt(e.target.value, 10) || 0)}
                    className="border-border bg-background w-16 rounded border px-2 py-1 font-mono text-sm"
                    min={-30}
                    max={30}
                  />
                  <span className="text-muted-foreground text-xs">minutes</span>
                </div>
                <p className="text-muted-foreground mt-1 text-xs">
                  Adjust Kalshi's reported game start times
                </p>
              </div>

              {/* Volume Windows */}
              {onToggleVolumeWindow && (
                <>
                  <div className="border-border border-b p-2">
                    <span className="text-muted-foreground text-xs font-medium uppercase tracking-wider">
                      Volume Windows
                    </span>
                  </div>
                  <div className="space-y-2 p-2">
                    {ALL_VOLUME_WINDOWS.map((window) => {
                      const isEnabled = volumeWindows.includes(window);
                      return (
                        <Toggle
                          key={window}
                          label={formatWindowLabel(window)}
                          checked={isEnabled}
                          onChange={() => onToggleVolumeWindow(window)}
                          size="sm"
                        />
                      );
                    })}
                  </div>
                </>
              )}
            </div>
          )}
        </div>
      </div>

      {/* Hotkey Configuration Modal */}
      <HotkeyConfigModal
        isOpen={showHotkeyConfig}
        onClose={() => setShowHotkeyConfig(false)}
      />
    </div>
  );
}

export default DashboardHeader;
