/**
 * OpenOrdersPanel - Displays open orders for the NBA Value Dashboard
 *
 * Shows a table of open orders with the ability to cancel individual orders
 * or kill all orders at once.
 */

import { useMemo, useState } from 'react';
import { ChevronDown, ChevronUp, X } from 'lucide-react';
import { cn } from '@/lib/utils';
import type { KalshiOrder } from '@/lib/kalshiApi';
import type { PolyOpenOrder } from '@/lib/polymarket/types';
import { formatDualPriceCents } from '@/lib/nbaConsolidated/format';
import { TickerDisplay } from '@/components/atoms/TickerDisplay';
import { Th, Td } from '@/components/atoms/TableCells';
import { ConfirmModal } from '@/components/atoms/Modal';
import { DecreaseOrderQuantityEditor } from '@/components/molecules/DecreaseOrderQuantityEditor';

interface OpenOrdersPanelProps {
  kalshiOrders: KalshiOrder[];
  polyOrders: PolyOpenOrder[];
  polyStatus?: 'disconnected' | 'connected' | 'error';
  /** Optional mapping from ticker -> market title (displayed above ticker) */
  titleByTicker?: Record<string, string>;
  /** Optional mapping from Polymarket token ID (assetId) -> human-readable label */
  polyTokenLabels?: Record<string, string>;
  onCancelOrder?: (orderId: string) => void;
  onDecreaseOrder?: (orderId: string, nextQuantity: number) => Promise<void> | void;
  onKillAll?: () => void;
  onCancelPolyOrder?: (orderId: string) => void;
  onKillAllPolyOrders?: () => void;
  className?: string;
}

export function OpenOrdersPanel({
  kalshiOrders,
  polyOrders,
  polyStatus = 'disconnected',
  titleByTicker,
  polyTokenLabels,
  onCancelOrder,
  onDecreaseOrder,
  onKillAll,
  onCancelPolyOrder,
  onKillAllPolyOrders,
  className,
}: OpenOrdersPanelProps) {
  const [showAllRows, setShowAllRows] = useState(false);
  const [showKillAllConfirm, setShowKillAllConfirm] = useState(false);
  const [showPolyKillAllConfirm, setShowPolyKillAllConfirm] = useState(false);
  const maxRows = 10;

  const visibleKalshiOrders = useMemo(() => {
    if (showAllRows) return kalshiOrders;
    return kalshiOrders.slice(0, maxRows);
  }, [kalshiOrders, showAllRows]);

  const visiblePolyOrders = useMemo(() => {
    if (showAllRows) return polyOrders;
    // If Kalshi took some slots, reduce poly slots
    const remainingSlots = Math.max(0, maxRows - kalshiOrders.length);
    return polyOrders.slice(0, remainingSlots);
  }, [polyOrders, kalshiOrders.length, showAllRows]);

  const totalOrders = kalshiOrders.length + polyOrders.length;

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header - clean section title */}
      <div className="border-border flex items-center justify-between border-b px-4 py-4">
        <h2 className="text-foreground text-lg font-semibold">Open Orders</h2>
        {kalshiOrders.length > 0 && onKillAll && (
          <button
            onClick={() => setShowKillAllConfirm(true)}
            className="bg-destructive hover:bg-destructive/90 text-destructive-foreground rounded-md px-4 py-2 text-xs font-medium uppercase tracking-wider shadow-sm transition-colors"
          >
            Kill All
          </button>
        )}

        {/* Kill All Confirmation Modal */}
        <ConfirmModal
          isOpen={showKillAllConfirm}
          onClose={() => setShowKillAllConfirm(false)}
          onConfirm={() => onKillAll?.()}
          title="Cancel All Orders"
          message={`Are you sure you want to cancel all ${kalshiOrders.length} open order${kalshiOrders.length === 1 ? '' : 's'}? This action cannot be undone.`}
          confirmLabel="Kill All"
          cancelLabel="Keep Orders"
          variant="danger"
        />
      </div>

      {/* Kalshi */}
      <div className="border-border/50 border-b p-4">
        <div className="text-muted-foreground mb-3 text-xs font-semibold uppercase tracking-wider">
          Kalshi ({kalshiOrders.length})
        </div>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-muted/50 border-border border-b">
              <tr>
                <Th>Market</Th>
                <Th>Side</Th>
                <Th>Action</Th>
                <Th className="text-right">Qty</Th>
                <Th className="text-right">Price</Th>
                <Th className="text-right">Queue</Th>
                <Th className="text-center"> </Th>
              </tr>
            </thead>
            <tbody className="divide-border divide-y">
              {kalshiOrders.length === 0 ? (
                <tr>
                  <td
                    colSpan={7}
                    className="text-muted-foreground px-3 py-6 text-center"
                  >
                    No Kalshi open orders
                  </td>
                </tr>
              ) : (
                visibleKalshiOrders.map((o, rowIndex) => {
                  const px = o.side === 'yes' ? o.yes_price : o.no_price;
                  const title = titleByTicker?.[o.ticker];
                  const zebraStyle =
                    rowIndex % 2 === 1
                      ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
                      : undefined;
                  return (
                    <tr
                      key={o.order_id}
                      className="hover:bg-accent/50 transition-colors"
                      style={zebraStyle}
                    >
                      <Td>
                        <TickerDisplay
                          ticker={o.ticker}
                          title={title}
                          variant="compact"
                        />
                      </Td>
                      <Td className="font-mono text-xs">{o.side.toUpperCase()}</Td>
                      <Td className="font-mono text-xs">{o.action.toUpperCase()}</Td>
                      <Td className="text-right font-mono text-xs">
                        {onDecreaseOrder ? (
                          <DecreaseOrderQuantityEditor
                            currentQuantity={o.remaining_count}
                            onSubmit={async (nextQuantity) => {
                              await onDecreaseOrder(o.order_id, nextQuantity);
                            }}
                          />
                        ) : (
                          o.remaining_count.toLocaleString()
                        )}
                      </Td>
                      <Td className="text-right font-mono text-xs">
                        {formatDualPriceCents(px ?? null)}
                      </Td>
                      <Td className="text-muted-foreground text-right font-mono text-xs">
                        {o.queue_position?.toLocaleString() ?? '—'}
                      </Td>
                      <Td className="text-center">
                        {onCancelOrder && (
                          <button
                            onClick={() => onCancelOrder(o.order_id)}
                            className="hover:bg-destructive/20 hover:text-destructive rounded p-1 transition-colors"
                            title="Cancel Kalshi order"
                          >
                            <X className="h-4 w-4" />
                          </button>
                        )}
                      </Td>
                    </tr>
                  );
                })
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Polymarket */}
      <div className="p-4">
        <div className="mb-3 flex items-center justify-between">
          <div className="text-muted-foreground text-xs font-semibold uppercase tracking-wider">
            Polymarket ({polyOrders.length}) · {polyStatus}
          </div>
          {polyOrders.length > 0 && onKillAllPolyOrders && (
            <button
              onClick={() => setShowPolyKillAllConfirm(true)}
              className="bg-destructive hover:bg-destructive/90 text-destructive-foreground rounded-md px-4 py-2 text-xs font-medium uppercase tracking-wider shadow-sm transition-colors"
            >
              Kill All
            </button>
          )}
        </div>

        {/* Poly Kill All Confirmation Modal */}
        <ConfirmModal
          isOpen={showPolyKillAllConfirm}
          onClose={() => setShowPolyKillAllConfirm(false)}
          onConfirm={() => onKillAllPolyOrders?.()}
          title="Cancel All Polymarket Orders"
          message={`Are you sure you want to cancel all ${polyOrders.length} Polymarket open order${polyOrders.length === 1 ? '' : 's'}? This action cannot be undone.`}
          confirmLabel="Kill All"
          cancelLabel="Keep Orders"
          variant="danger"
        />
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-muted/50 border-border border-b">
              <tr>
                <Th>Market</Th>
                <Th>Side</Th>
                <Th className="text-right">Remaining</Th>
                <Th className="text-right">Price</Th>
                {onCancelPolyOrder && <Th className="text-center"> </Th>}
              </tr>
            </thead>
            <tbody className="divide-border divide-y">
              {polyOrders.length === 0 ? (
                <tr>
                  <td
                    colSpan={onCancelPolyOrder ? 5 : 4}
                    className="text-muted-foreground px-3 py-6 text-center"
                  >
                    No Polymarket open orders
                  </td>
                </tr>
              ) : (
                visiblePolyOrders.map((o, rowIndex) => {
                  const remaining = Math.max(0, (o.originalSize ?? 0) - (o.sizeMatched ?? 0));
                  const label = polyTokenLabels?.[o.assetId];
                  const zebraStyle =
                    rowIndex % 2 === 1
                      ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
                      : undefined;
                  return (
                    <tr
                      key={o.id}
                      className="hover:bg-accent/50 transition-colors"
                      style={zebraStyle}
                    >
                      <Td className="text-xs">
                        {label ? (
                          <span className="font-medium">{label}</span>
                        ) : (
                          <span
                            className="text-muted-foreground font-mono"
                            title={o.assetId}
                          >
                            {o.assetId.slice(0, 12)}…
                          </span>
                        )}
                      </Td>
                      <Td className="font-mono text-xs">{String(o.side).toUpperCase()}</Td>
                      <Td className="text-right font-mono text-xs">{remaining.toLocaleString()}</Td>
                      <Td className="text-right font-mono text-xs">
                        {Number.isFinite(o.price) ? Number(o.price).toFixed(4) : '—'}
                      </Td>
                      {onCancelPolyOrder && (
                        <Td className="text-center">
                          <button
                            onClick={() => onCancelPolyOrder(o.id)}
                            className="hover:bg-destructive/20 hover:text-destructive rounded p-1 transition-colors"
                            title="Cancel Polymarket order"
                          >
                            <X className="h-4 w-4" />
                          </button>
                        </Td>
                      )}
                    </tr>
                  );
                })
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Show more/less */}
      {totalOrders > maxRows && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
            aria-label={showAllRows ? 'Show fewer orders' : 'Show more orders'}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${totalOrders}`
                : `Showing ${Math.min(maxRows, totalOrders)} of ${totalOrders}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}
    </div>
  );
}

export default OpenOrdersPanel;
