/**
 * TwoRowMoneylines - Table showing odds comparison with 2 rows per game (one per team)
 *
 * This format matches traditional sportsbook displays where each team gets its own row.
 * Away team row includes the game time; home team row leaves time blank.
 */

import { cn } from '@/lib/utils';
import { LiquidityBar } from '@/components/atoms/LiquidityBar';
import { Odds } from '@/components/atoms/Odds';
import { DateDisplay } from '@/components/atoms/DateDisplay';
import { Th, Td as BaseTd } from '@/components/atoms/TableCells';
import { SortableTh } from '@/components/atoms/SortableTh';
import { Fragment, memo, useMemo, useState } from 'react';
import { useTableSort } from '@/hooks/useTableSort';
import type { ConsolidatedGameBooks, TeamSide, Venue } from '@/lib/nbaConsolidated/types';
import { ChevronDown, ChevronUp } from 'lucide-react';
import {
  applyTakerFee,
  computeOrderPriceFromRawBid,
  getTeamWinData,
  next3NoLadders,
  next3YesLadders,
  takerFeeForVenue,
  liquidityDollars,
} from '@/lib/nbaConsolidated/pricing';
import type { OddsMode } from './DashboardHeader';
import type { TradeClickRequest } from './ConsolidatedMoneylines';
import { getSeriesTicker, type Sport } from '@/lib/sportsDiscovery';
import { formatStartTimePt } from '@/lib/formatters';
import type { SportsLoadingState } from '@/lib/sportsStream';
import { LoadingPhaseStatus } from './LoadingPhaseStatus';

/** Table cell for two-row layout with improved spacing */
function Td({
  children,
  className,
  ...rest
}: React.TdHTMLAttributes<HTMLTableCellElement> & { children: React.ReactNode }) {
  return (
    <BaseTd
      {...rest}
      className={cn('border-0 py-2', className)}
    >
      {children}
    </BaseTd>
  );
}

interface TwoRowMoneylinesProps {
  games: ConsolidatedGameBooks[];
  isLoading?: boolean;
  loadingState?: SportsLoadingState;
  oddsMode: OddsMode;
  maxLiquidity?: number;
  expandedGameKeys: Set<string>;
  onToggleExpand: (gameKey: string) => void;
  onTradeClick?: (req: TradeClickRequest) => void;
  /** Navigate to MarketDetailView (primary click). Shift+click still opens trade modal. */
  onNavigateToMarket?: (marketTicker: string) => void;
  /** Sport for team name lookup (default: nba) */
  sport?: Sport;
  /** Show "no book" / "n/a" hints for empty data (dev explorer mode) */
  showEmptyHints?: boolean;
  className?: string;
}

export function TwoRowMoneylines({
  games,
  isLoading = false,
  loadingState,
  maxLiquidity = 100000,
  oddsMode,
  expandedGameKeys,
  onToggleExpand,
  onTradeClick,
  onNavigateToMarket,
  sport = 'nba',
  showEmptyHints = false,
  className,
}: TwoRowMoneylinesProps) {
  const [showAllRows, setShowAllRows] = useState(true);
  const maxGames = 10;

  const sortExtractors = useMemo(
    () => ({
      rotation: (game: ConsolidatedGameBooks) => game.awayRotation ?? null,
    }),
    []
  );

  const { sorted, sortKey, sortDir, handleSort } = useTableSort(games, sortExtractors, {
    defaultSortKey: 'time',
    defaultSortDir: 'asc',
  });

  const visibleGames = useMemo(() => {
    if (showAllRows) return sorted;
    return sorted.slice(0, maxGames);
  }, [sorted, showAllRows]);

  const isTakerMode = oddsMode === 'taker';

  const sportLabel = sport.toUpperCase().replace('-', ' ');
  const seriesTicker = getSeriesTicker(sport, 'moneyline');

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header with sport-prefixed title and series ticker */}
      <div className="border-border border-b px-4 py-4">
        <div className="flex items-center justify-between">
          <h2 className="text-foreground text-lg font-semibold">{sportLabel} Moneylines</h2>
          <div className="flex items-center gap-3">
            {seriesTicker && (
              <span className="text-muted-foreground font-mono text-xs">{seriesTicker}</span>
            )}
            <LoadingPhaseStatus
              isLoading={isLoading}
              loadingState={loadingState}
            />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full border-collapse border-spacing-0">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <SortableTh
                className="w-14"
                sortKey="rotation"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Roto
              </SortableTh>
              <SortableTh
                className="w-24"
                sortKey="date"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Date
              </SortableTh>
              <SortableTh
                className="w-16"
                sortKey="time"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Time PT
              </SortableTh>
              <SortableTh
                className="w-48"
                sortKey="team"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Team
              </SortableTh>
              <Th className="w-20 text-right">Kalshi</Th>
              <Th className="w-16 text-right">K Vol</Th>
              <Th className="w-24">K Liq</Th>
              <Th className="w-20 text-right">Poly</Th>
              <Th className="w-16 text-right">P Vol</Th>
              <Th className="w-24">P Liq</Th>
            </tr>
          </thead>
          <tbody>
            {isLoading && games.length === 0
              ? Array.from({ length: 10 }).map((_, i) => (
                  <tr
                    key={`loading-${i}`}
                    className="animate-pulse"
                  >
                    <Td>
                      <div className="bg-muted/40 h-3 w-10 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-16 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-28 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="p-1">
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="p-1">
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                  </tr>
                ))
              : visibleGames.map((game, gameIndex) => (
                  <GameRows
                    key={game.key}
                    game={game}
                    gameIndex={gameIndex}
                    isTakerMode={isTakerMode}
                    maxLiquidity={maxLiquidity}
                    expanded={expandedGameKeys.has(game.key)}
                    onToggleExpand={onToggleExpand}
                    onTradeClick={onTradeClick}
                    onNavigateToMarket={onNavigateToMarket}
                    oddsMode={oddsMode}
                    sport={sport}
                    showEmptyHints={showEmptyHints}
                  />
                ))}
          </tbody>
        </table>
      </div>

      {/* Expand / minimize (show more rows) */}
      {games.length > maxGames && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
            aria-label={showAllRows ? 'Show fewer games' : 'Show more games'}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${games.length}`
                : `Showing ${Math.min(maxGames, games.length)} of ${games.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}

      {/* Empty state */}
      {games.length === 0 && !isLoading && (
        <div className="text-muted-foreground p-8 text-center">No games available</div>
      )}
    </div>
  );
}

interface GameRowsProps {
  game: ConsolidatedGameBooks;
  gameIndex: number;
  isTakerMode: boolean;
  oddsMode?: OddsMode;
  maxLiquidity: number;
  expanded: boolean;
  onToggleExpand: (gameKey: string) => void;
  onTradeClick?: (req: TradeClickRequest) => void;
  onNavigateToMarket?: (marketTicker: string) => void;
  sport: Sport;
  showEmptyHints?: boolean;
}

const GameRows = memo(function GameRows({
  game,
  gameIndex,
  isTakerMode,
  oddsMode: _oddsMode,
  maxLiquidity,
  expanded,
  onToggleExpand,
  onTradeClick,
  onNavigateToMarket,
  sport: _sport,
  showEmptyHints = false,
}: GameRowsProps) {
  const awayName = game.awayName ?? game.awayCode;
  const homeName = game.homeName ?? game.homeCode;

  const awayK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'away',
    isTakerMode,
  });
  const homeK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'home',
    isTakerMode,
  });
  const awayP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'away',
    isTakerMode,
  });
  const homeP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'home',
    isTakerMode,
  });

  const handleTrade = (teamSide: TeamSide, venue: Venue, e: React.MouseEvent) => {
    const teamData =
      venue === 'kalshi'
        ? teamSide === 'away'
          ? awayK
          : homeK
        : teamSide === 'away'
          ? awayP
          : homeP;
    const marketTicker = teamData.marketTicker;
    if (!marketTicker) return;

    // Shift+click: open trade modal (legacy behavior)
    if (e.shiftKey && onTradeClick) {
      const teamLabel = teamSide === 'away' ? awayName : homeName;
      const orderPrice = computeOrderPriceFromRawBid(teamData.rawBidPriceCents, isTakerMode);
      if (orderPrice === null) return;

      if (venue === 'kalshi') {
        onTradeClick({
          venue: 'kalshi',
          marketTicker,
          side: 'no',
          priceCents: orderPrice,
          contracts: 100,
          isTakerMode,
          teamLabel,
          gameKey: game.key,
          gameDate: game.date,
          startTimePt: game.startTimePt,
        });
      } else {
        const opponentMarket =
          teamSide === 'away' ? game.polymarket?.markets.home : game.polymarket?.markets.away;
        onTradeClick({
          venue: 'polymarket',
          marketTicker,
          tokenId: opponentMarket?.tokenId,
          conditionId: opponentMarket?.conditionId,
          tickSize: opponentMarket?.tickSize,
          negRisk: opponentMarket?.negRisk,
          side: 'no',
          priceCents: orderPrice,
          contracts: 100,
          isTakerMode,
          teamLabel,
          gameKey: game.key,
          gameDate: game.date,
          startTimePt: game.startTimePt,
        });
      }
      return;
    }

    // Primary click: navigate to market detail
    if (onNavigateToMarket) {
      onNavigateToMarket(marketTicker);
    }
  };

  const hasKalshi = !!game.kalshi;
  const hasPoly = !!game.polymarket;

  const renderTeamRow = (teamSide: TeamSide, isFirstRow: boolean) => {
    const teamName = teamSide === 'away' ? awayName : homeName;
    const teamK = teamSide === 'away' ? awayK : homeK;
    const teamP = teamSide === 'away' ? awayP : homeP;

    const kDollars = liquidityDollars(teamK.liq, teamK.priceCents);
    const pDollars = liquidityDollars(teamP.liq, teamP.priceCents);

    // "no book" when venue is entirely missing, "—" when venue exists but price is null
    const kNullText = showEmptyHints && !hasKalshi ? 'no book' : '—';
    const pNullText = showEmptyHints && !hasPoly ? 'no book' : '—';

    // Add thick top border only for away (first) row to separate games (skip first game)
    const borderClass = isFirstRow && gameIndex > 0 ? 'border-t-2 border-border' : '';
    // Alternating row colors - both rows of a game share the same background
    const zebraStyle =
      gameIndex % 2 === 1
        ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
        : undefined;

    return (
      <tr
        key={`${game.key}-${teamSide}`}
        className={cn('hover:bg-accent/50 cursor-pointer transition-colors', borderClass)}
        style={zebraStyle}
        onClick={() => {
          onToggleExpand(game.key);
        }}
      >
        {/* Rotation number */}
        <Td className="text-muted-foreground font-mono text-xs">
          {teamSide === 'away'
            ? (game.awayRotation ?? (showEmptyHints ? 'n/a' : '—'))
            : (game.homeRotation ?? (showEmptyHints ? 'n/a' : '—'))}
        </Td>

        {/* Date (only show on first row) */}
        <Td className="text-muted-foreground font-mono text-xs">
          {isFirstRow ? <DateDisplay value={game.date} /> : ''}
        </Td>

        {/* Time (only show on first row) */}
        <Td className="text-muted-foreground font-mono text-xs">
          {isFirstRow ? formatStartTimePt(game.startTimePt) : ''}
        </Td>

        {/* Team Name */}
        <Td className="font-medium">{teamName}</Td>

        {/* Kalshi odds */}
        <Td
          className={cn('text-right font-mono', teamK.priceCents !== null && 'hover:underline')}
          onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
            e.stopPropagation();
            handleTrade(teamSide, 'kalshi', e);
          }}
          title={
            teamK.marketTicker
              ? `Click to view ${teamK.marketTicker} (Shift+click to trade)`
              : undefined
          }
        >
          <Odds
            cents={teamK.priceCents}
            format="dual"
            nullText={kNullText}
          />
        </Td>

        {/* Kalshi Volume (contracts) */}
        <Td className="text-muted-foreground text-right font-mono text-xs">
          {teamK.liq != null ? teamK.liq.toLocaleString() : '—'}
        </Td>

        {/* Kalshi Liquidity ($) */}
        <Td>
          <LiquidityBar
            value={kDollars ?? 0}
            maxValue={maxLiquidity}
          />
        </Td>

        {/* Polymarket odds */}
        <Td
          className={cn('text-right font-mono', teamP.priceCents !== null && 'hover:underline')}
          onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
            e.stopPropagation();
            handleTrade(teamSide, 'polymarket', e);
          }}
          title={teamP.marketTicker ? `Click to view market (Shift+click to trade)` : undefined}
        >
          <Odds
            cents={teamP.priceCents}
            format="dual"
            nullText={pNullText}
          />
        </Td>

        {/* Polymarket Volume (contracts) */}
        <Td className="text-muted-foreground text-right font-mono text-xs">
          {teamP.liq != null ? teamP.liq.toLocaleString() : '—'}
        </Td>

        {/* Polymarket Liquidity ($) */}
        <Td>
          <LiquidityBar
            value={pDollars ?? 0}
            maxValue={maxLiquidity}
          />
        </Td>
      </tr>
    );
  };

  const renderLadderRows = (teamSide: TeamSide) => {
    const teamK = teamSide === 'away' ? awayK : homeK;
    const teamP = teamSide === 'away' ? awayP : homeP;

    // In maker mode, ladder comes from NO bids; in taker mode, from YES bids (for inversion).
    const kLvls = isTakerMode
      ? next3YesLadders(teamK.yesLevels, teamK.rawBidPriceCents)
      : next3NoLadders(teamK.noLevels, teamK.rawBidPriceCents);
    const pLvls = isTakerMode
      ? next3YesLadders(teamP.yesLevels, teamP.rawBidPriceCents)
      : next3NoLadders(teamP.noLevels, teamP.rawBidPriceCents);

    const feeRate = takerFeeForVenue('kalshi');
    const polyFee = takerFeeForVenue('polymarket');

    return [0, 1, 2].map((idx) => {
      const kLvl = kLvls[idx] ?? null;
      const pLvl = pLvls[idx] ?? null;

      const kPrice = kLvl
        ? isTakerMode
          ? applyTakerFee(100 - kLvl.priceCents, feeRate)
          : kLvl.priceCents
        : null;
      const pPrice = pLvl
        ? isTakerMode
          ? applyTakerFee(100 - pLvl.priceCents, polyFee)
          : pLvl.priceCents
        : null;

      const kDollars = kLvl ? liquidityDollars(kLvl.size, kPrice) : null;
      const pDollars = pLvl ? liquidityDollars(pLvl.size, pPrice) : null;
      return (
        <tr
          key={`${game.key}-${teamSide}-ladder-${idx}`}
          className="border-0"
        >
          <Td> </Td>
          <Td> </Td>
          <Td> </Td>
          <Td> </Td>
          <Td className="text-right font-mono">
            <Odds
              cents={kPrice}
              format="dual"
            />
          </Td>
          <Td className="text-right font-mono text-xs">{kLvl ? kLvl.size.toLocaleString() : ''}</Td>
          <Td>
            <LiquidityBar
              value={kDollars ?? 0}
              maxValue={maxLiquidity}
            />
          </Td>
          <Td className="text-right font-mono">
            <Odds
              cents={pPrice}
              format="dual"
            />
          </Td>
          <Td className="text-right font-mono text-xs">{pLvl ? pLvl.size.toLocaleString() : ''}</Td>
          <Td>
            <LiquidityBar
              value={pDollars ?? 0}
              maxValue={maxLiquidity}
            />
          </Td>
        </tr>
      );
    });
  };

  return (
    <Fragment>
      {/* Away team row (first row - shows date/time) */}
      {renderTeamRow('away', true)}
      {expanded && renderLadderRows('away')}

      {/* Home team row (second row - no date/time) */}
      {renderTeamRow('home', false)}
      {expanded && renderLadderRows('home')}
    </Fragment>
  );
});

export default TwoRowMoneylines;
