/**
 * Toast - Notification system for alerts and messages
 * Supports multiple toast types, auto-dismiss, and actions
 */

import { useState, useEffect, createContext, useContext, useCallback } from 'react';
import { X, CheckCircle, AlertCircle, AlertTriangle, Info, Bell } from 'lucide-react';
import { cn } from '@/lib/utils';

export type ToastType = 'success' | 'error' | 'warning' | 'info' | 'alert';

export interface ToastData {
  id: string;
  type: ToastType;
  title: string;
  message?: string;
  duration?: number;
  action?: {
    label: string;
    onClick: () => void;
  };
}

interface ToastContextValue {
  toasts: ToastData[];
  addToast: (toast: Omit<ToastData, 'id'>) => string;
  removeToast: (id: string) => void;
  clearAll: () => void;
}

const ToastContext = createContext<ToastContextValue | null>(null);

export function useToast() {
  const context = useContext(ToastContext);
  if (!context) {
    throw new Error('useToast must be used within a ToastProvider');
  }
  return context;
}

export function ToastProvider({ children }: { children: React.ReactNode }) {
  const [toasts, setToasts] = useState<ToastData[]>([]);

  const addToast = useCallback((toast: Omit<ToastData, 'id'>) => {
    const id = `toast-${Date.now()}-${Math.random().toString(36).slice(2)}`;
    setToasts((prev) => [...prev, { ...toast, id }]);
    return id;
  }, []);

  const removeToast = useCallback((id: string) => {
    setToasts((prev) => prev.filter((t) => t.id !== id));
  }, []);

  const clearAll = useCallback(() => {
    setToasts([]);
  }, []);

  return (
    <ToastContext.Provider value={{ toasts, addToast, removeToast, clearAll }}>
      {children}
      <ToastContainer
        toasts={toasts}
        onRemove={removeToast}
      />
    </ToastContext.Provider>
  );
}

function ToastContainer({
  toasts,
  onRemove,
}: {
  toasts: ToastData[];
  onRemove: (id: string) => void;
}) {
  return (
    <div className="fixed bottom-4 right-4 z-50 flex max-w-sm flex-col gap-2">
      {toasts.map((toast) => (
        <ToastItem
          key={toast.id}
          toast={toast}
          onRemove={onRemove}
        />
      ))}
    </div>
  );
}

function ToastItem({ toast, onRemove }: { toast: ToastData; onRemove: (id: string) => void }) {
  const [isExiting, setIsExiting] = useState(false);

  useEffect(() => {
    const duration = toast.duration ?? 5000;
    if (duration > 0) {
      const timer = setTimeout(() => {
        setIsExiting(true);
        setTimeout(() => onRemove(toast.id), 200);
      }, duration);
      return () => clearTimeout(timer);
    }
  }, [toast.id, toast.duration, onRemove]);

  const handleClose = () => {
    setIsExiting(true);
    setTimeout(() => onRemove(toast.id), 200);
  };

  const icons: Record<ToastType, React.ReactNode> = {
    success: (
      <CheckCircle
        size={18}
        className="text-green-400"
      />
    ),
    error: (
      <AlertCircle
        size={18}
        className="text-red-400"
      />
    ),
    warning: (
      <AlertTriangle
        size={18}
        className="text-yellow-400"
      />
    ),
    info: (
      <Info
        size={18}
        className="text-blue-400"
      />
    ),
    alert: (
      <Bell
        size={18}
        className="text-primary"
      />
    ),
  };

  const borderColors: Record<ToastType, string> = {
    success: 'border-l-green-500',
    error: 'border-l-red-500',
    warning: 'border-l-yellow-500',
    info: 'border-l-blue-500',
    alert: 'border-l-primary',
  };

  return (
    <div
      className={cn(
        'bg-card border-border rounded-lg border border-l-4 p-4 shadow-lg transition-all duration-200',
        borderColors[toast.type],
        isExiting ? 'translate-x-4 opacity-0' : 'translate-x-0 opacity-100'
      )}
    >
      <div className="flex items-start gap-3">
        <span className="mt-0.5">{icons[toast.type]}</span>
        <div className="min-w-0 flex-1">
          <div className="text-sm font-medium">{toast.title}</div>
          {toast.message && (
            <div className="text-muted-foreground mt-0.5 text-xs">{toast.message}</div>
          )}
          {toast.action && (
            <button
              onClick={toast.action.onClick}
              className="text-primary mt-2 text-xs font-medium hover:underline"
            >
              {toast.action.label}
            </button>
          )}
        </div>
        <button
          onClick={handleClose}
          className="hover:bg-muted text-muted-foreground rounded p-1 transition-colors"
        >
          <X size={14} />
        </button>
      </div>
    </div>
  );
}

// Standalone toast display for use without context
export function Toast({ toast, onClose }: { toast: ToastData; onClose: () => void }) {
  return (
    <ToastItem
      toast={toast}
      onRemove={onClose}
    />
  );
}
