/**
 * TradeTape - Shows real-time stream of all market trades
 *
 * Displays a scrolling tape of recent trades in the market.
 * Clicking a trade sets the price in the Quick Order form.
 */

import { cn } from '@/lib/utils';
import { Badge } from '@/components/atoms/Badge';
import { useFlashNewItems } from '@/hooks/useFlash';
import type { KalshiTrade } from '@/lib/kalshiApi';

interface TradeTapeProps {
  trades: KalshiTrade[];
  loading?: boolean;
  className?: string;
  onTradeClick?: (trade: KalshiTrade) => void;
}

export function TradeTape({ trades, loading, className, onTradeClick }: TradeTapeProps) {
  // Track which trades are new and should flash
  const flashingTradeIds = useFlashNewItems(trades.map((t) => t.trade_id));
  const formatTime = (isoString: string): string => {
    const date = new Date(isoString);
    return date.toLocaleTimeString('en-US', {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      hour12: false,
    });
  };

  const formatPrice = (price: number): string => {
    return `${price}¢`;
  };

  return (
    <div className={cn('flex flex-col', className)}>
      {/* Header */}
      <div className="border-border/50 bg-muted/30 flex items-center gap-2 border-b px-3 py-2">
        <div className="h-2 w-2 animate-pulse rounded-full bg-green-500" />
        <h3 className="text-sm font-medium">Trade Tape</h3>
        <span className="text-muted-foreground text-xs">({trades.length} recent)</span>
      </div>

      {/* Trade list */}
      <div className="max-h-48 flex-1 overflow-y-auto">
        {loading ? (
          <div className="text-muted-foreground flex items-center justify-center py-8 text-sm">
            Loading trades...
          </div>
        ) : trades.length === 0 ? (
          <div className="text-muted-foreground flex items-center justify-center py-8 text-sm">
            No trades yet
          </div>
        ) : (
          <div className="divide-border/20 divide-y">
            {trades.map((trade) => (
              <TradeRow
                key={trade.trade_id}
                trade={trade}
                isFlashing={flashingTradeIds.has(trade.trade_id)}
                onClick={() => onTradeClick?.(trade)}
                formatTime={formatTime}
                formatPrice={formatPrice}
              />
            ))}
          </div>
        )}
      </div>
    </div>
  );
}

interface TradeRowProps {
  trade: KalshiTrade;
  isFlashing?: boolean;
  onClick?: () => void;
  formatTime: (isoString: string) => string;
  formatPrice: (price: number) => string;
}

function TradeRow({ trade, isFlashing, onClick, formatTime, formatPrice }: TradeRowProps) {
  const isBuy = trade.taker_side === 'yes';
  const price = trade.yes_price;

  return (
    <div
      onClick={onClick}
      className={cn(
        'flex items-center justify-between px-3 py-1.5 font-mono text-xs',
        'cursor-pointer transition-colors',
        'hover:bg-accent/40',
        isBuy
          ? 'hover:border-l-2 hover:border-l-green-500'
          : 'hover:border-l-2 hover:border-l-red-500',
        isFlashing && (isBuy ? 'animate-flash-up' : 'animate-flash-down')
      )}
    >
      {/* Time */}
      <span className="text-muted-foreground w-20">{formatTime(trade.created_time)}</span>

      {/* Side badge */}
      <Badge
        variant={isBuy ? 'success' : 'danger'}
        size="xs"
        className="w-10 justify-center"
      >
        {isBuy ? 'BUY' : 'SELL'}
      </Badge>

      {/* Price */}
      <span
        className={cn('w-12 text-right font-semibold', isBuy ? 'text-green-500' : 'text-red-500')}
      >
        {formatPrice(price)}
      </span>

      {/* Quantity */}
      <span className="text-muted-foreground w-16 text-right">{trade.count.toLocaleString()}</span>
    </div>
  );
}

export default TradeTape;
