/**
 * DevExplorerView — Sports inventory browser (dev tool).
 *
 * Activity-first: series sorted by volume, markets filtered by live activity.
 * Two-panel layout: series list (left) + market table (right).
 *
 * When a known sports series is selected, spins up a live SportsStream and
 * renders the same table components used in the main dashboard
 * (TwoRowMoneylines / TwoRowSpreads / TwoRowTotals).
 */

import { useState, useMemo, useCallback, useRef, useEffect } from 'react';
import { RefreshCw, Radio } from 'lucide-react';
import { cn } from '@/lib/utils';
import type { KalshiApiClient } from '@/lib/kalshiApi';
import type { KalshiMarket, KalshiOrderbook, KalshiCredentials } from '@/types';
import { useDevModeData } from '@/hooks/useDevModeData';
import { Toggle } from '@/components/atoms/Toggle';
import { getConfigFromSeriesTicker } from '@/lib/sportsDiscovery';
import {
  createSportsStream,
  type SportsStream,
  type GameData,
  type MoneylineGameData,
  type SpreadGameData,
  type TotalGameData,
  type SportsLoadingState,
  moneylineToConsolidated,
} from '@/lib/sportsStream';
import { initAuth } from '@/lib/kalshiAuth';
import { TwoRowMoneylines } from '@/components/nba-value-dashboard/TwoRowMoneylines';
import { TwoRowSpreads, TwoRowTotals } from '@/components/sports';
import type { Sport, MarketType } from '@/lib/sportsDiscovery';
import type { ConsolidatedGameBooks } from '@/lib/nbaConsolidated/types';
import { useRotationNumbers } from '@/hooks/useRotationNumbers';

/** Human-readable labels for known Kalshi sports series */
const SERIES_LABELS: Record<string, string> = {
  KXMLBGAME: 'MLB Games',
  KXUFCFIGHT: 'UFC Fights',
  KXNCAAWBGAME: "Women's CBB",
  KXNBAGAME: 'NBA Games',
  KXNBASPREAD: 'NBA Spreads',
  KXNBATOTAL: 'NBA Totals',
  KXNHLGAME: 'NHL Games',
  KXNHLSPREAD: 'NHL Spreads',
  KXNHLTOTAL: 'NHL Totals',
  KXNCAAMBGAME: "Men's CBB",
  KXNCAAMBSPREAD: 'CBB Spreads',
  KXNCAAMBTOTAL: 'CBB Totals',
  KXNFLGAME: 'NFL Games',
  KXNFLSPREAD: 'NFL Spreads',
  KXNFLTOTAL: 'NFL Totals',
  KXATPMATCH: 'ATP Tennis',
  KXWTAMATCH: 'WTA Tennis',
};

function getSeriesLabel(ticker: string, title?: string): string {
  const upper = ticker.toUpperCase();
  if (SERIES_LABELS[upper]) return SERIES_LABELS[upper];
  if (title) return title;
  return upper.startsWith('KX') ? upper.slice(2) : upper;
}

interface DevExplorerViewProps {
  apiClientRef: React.RefObject<KalshiApiClient | null>;
  credentials?: KalshiCredentials | null;
  onPlaceOrder?: (ticker: string, price?: number) => void;
}

export function DevExplorerView({ apiClientRef, credentials, onPlaceOrder }: DevExplorerViewProps) {
  const devData = useDevModeData(apiClientRef, true);
  const [activeOnly, setActiveOnly] = useState(true);

  // Live sports stream state
  const sportsStreamRef = useRef<SportsStream | null>(null);
  const activeStreamKeyRef = useRef<string>('');
  const [liveGames, setLiveGames] = useState<GameData[]>([]);
  const [liveLoading, setLiveLoading] = useState(false);
  const [liveLoadingState, setLiveLoadingState] = useState<SportsLoadingState | undefined>();
  const [liveConfig, setLiveConfig] = useState<{
    sport: Sport;
    marketType: MarketType;
  } | null>(null);

  // Moneyline table state
  const [expandedGameKeys, setExpandedGameKeys] = useState<Set<string>>(new Set());
  const handleToggleExpand = useCallback((gameKey: string) => {
    setExpandedGameKeys((prev) => {
      const next = new Set(prev);
      if (next.has(gameKey)) next.delete(gameKey);
      else next.add(gameKey);
      return next;
    });
  }, []);

  // Stop current stream
  const stopStream = useCallback(() => {
    if (sportsStreamRef.current) {
      sportsStreamRef.current.stop();
      sportsStreamRef.current = null;
    }
    activeStreamKeyRef.current = '';
    setLiveGames([]);
    setLiveLoading(false);
    setLiveLoadingState(undefined);
    setLiveConfig(null);
    setExpandedGameKeys(new Set());
  }, []);

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      if (sportsStreamRef.current) {
        sportsStreamRef.current.stop();
        sportsStreamRef.current = null;
      }
    };
  }, []);

  // Start a sports stream for a known series
  const startSportsStream = useCallback(
    async (sport: Sport, marketType: MarketType) => {
      const streamKey = `${sport}-${marketType}`;
      if (activeStreamKeyRef.current === streamKey && sportsStreamRef.current) return;

      // Stop existing stream
      if (sportsStreamRef.current) {
        sportsStreamRef.current.stop();
        sportsStreamRef.current = null;
      }

      if (!apiClientRef.current || !credentials) return;

      activeStreamKeyRef.current = streamKey;
      setLiveConfig({ sport, marketType });
      setLiveLoading(true);
      setLiveGames([]);
      setExpandedGameKeys(new Set());

      try {
        const { privateKey: cryptoKey } = await initAuth(
          credentials.accessKeyId,
          credentials.privateKeyPem
        );

        const stream = createSportsStream(
          {
            api: apiClientRef.current,
            accessKeyId: credentials.accessKeyId,
            privateKey: cryptoKey,
            environment: credentials.environment,
            useRelay: true,
            useMock: false,
          },
          { sport, marketType, daysAhead: 14 }
        );

        sportsStreamRef.current = stream;

        stream.onUpdate((update) => {
          // Guard against stale callbacks from a previous stream
          if (activeStreamKeyRef.current !== streamKey) return;
          setLiveGames(update.games);
          setLiveLoadingState(update.loadingState);
          setLiveLoading(update.loadingState.isLoading);
        });

        await stream.start();
      } catch (err) {
        console.error(`[DevExplorer] Failed to start ${streamKey} stream:`, err);
        setLiveLoading(false);
        setLiveLoadingState(undefined);
      }
    },
    [apiClientRef, credentials]
  );

  // Sort series by volume descending
  const sortedSeries = useMemo(() => {
    const list = [...devData.series].sort((a, b) => (b.volume ?? 0) - (a.volume ?? 0));
    if (activeOnly) {
      return list.filter((s) => (s.volume ?? 0) > 0);
    }
    return list;
  }, [devData.series, activeOnly]);

  // Filter markets by activity
  const filteredMarkets = useMemo(() => {
    if (!activeOnly) return devData.markets;
    return devData.markets.filter((m) => m.volume > 0 || m.open_interest > 0);
  }, [devData.markets, activeOnly]);

  // Sort filtered markets by volume descending
  const sortedMarkets = useMemo(() => {
    return [...filteredMarkets].sort((a, b) => b.volume - a.volume);
  }, [filteredMarkets]);

  // Stats for selected series
  const marketStats = useMemo(() => {
    const total = devData.markets.length;
    const active = devData.markets.filter((m) => m.volume > 0 || m.open_interest > 0).length;
    const totalVol = devData.markets.reduce((sum, m) => sum + m.volume, 0);
    const totalOI = devData.markets.reduce((sum, m) => sum + m.open_interest, 0);
    return { total, active, totalVol, totalOI };
  }, [devData.markets]);

  const handleSelectSeries = useCallback(
    (ticker: string) => {
      devData.selectSeries(ticker);

      // Check if this series maps to a known sport+marketType
      const config = getConfigFromSeriesTicker(ticker);
      if (config && credentials) {
        startSportsStream(config.sport, config.marketType);
      } else {
        stopStream();
      }
    },
    [devData, credentials, startSportsStream, stopStream]
  );

  const selectedSeriesObj = devData.series.find(
    (s) => (s.series_ticker || s.ticker) === devData.selectedTicker
  );

  // Convert moneyline games to consolidated format for TwoRowMoneylines
  const consolidatedMoneylineGames: ConsolidatedGameBooks[] = useMemo(() => {
    if (!liveConfig || liveConfig.marketType !== 'moneyline') return [];
    return (liveGames as MoneylineGameData[]).map(
      moneylineToConsolidated
    ) as ConsolidatedGameBooks[];
  }, [liveGames, liveConfig]);

  // Determine if we're showing a live sports view
  const isLiveView = liveConfig !== null;

  return (
    <div className="flex h-full flex-col gap-4 p-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-foreground text-lg font-semibold">Sports Inventory</h1>
          <p className="text-muted-foreground text-xs">
            Browse live Kalshi sports series — select a known sport to stream live data
          </p>
        </div>
        <div className="flex items-center gap-3">
          {isLiveView && (
            <span className="flex items-center gap-1.5 text-xs text-emerald-400">
              <Radio className="h-3 w-3 animate-pulse" />
              live
            </span>
          )}
          <Toggle
            label="Active only"
            checked={activeOnly}
            onChange={() => setActiveOnly((v) => !v)}
            size="sm"
          />
          <button
            onClick={devData.refresh}
            disabled={devData.loading || devData.seriesLoading}
            className="text-muted-foreground hover:text-foreground transition-colors disabled:opacity-50"
            title="Refresh"
          >
            <RefreshCw
              className={cn(
                'h-4 w-4',
                (devData.loading || devData.seriesLoading) && 'animate-spin'
              )}
            />
          </button>
        </div>
      </div>

      {/* Two-panel layout */}
      <div className="flex min-h-0 flex-1 gap-4">
        {/* Left: Series list */}
        <div className="bg-card border-border flex w-64 shrink-0 flex-col rounded-lg border">
          <div className="border-border border-b px-3 py-2">
            <span className="text-muted-foreground text-[10px] font-medium uppercase tracking-wider">
              Series ({sortedSeries.length})
            </span>
          </div>
          <div className="flex-1 overflow-y-auto">
            {devData.seriesLoading && sortedSeries.length === 0 ? (
              <div className="text-muted-foreground p-3 text-xs">Loading series...</div>
            ) : sortedSeries.length === 0 ? (
              <div className="text-muted-foreground p-3 text-xs">
                {activeOnly ? 'No series with activity' : 'No sports series found'}
              </div>
            ) : (
              sortedSeries.map((s) => {
                const ticker = s.series_ticker || s.ticker || '';
                const label = getSeriesLabel(ticker, s.title);
                const volume = s.volume ?? 0;
                const isSelected = devData.selectedTicker === ticker;
                const hasLiveSupport = !!getConfigFromSeriesTicker(ticker);

                return (
                  <button
                    key={ticker}
                    onClick={() => handleSelectSeries(ticker)}
                    className={cn(
                      'flex w-full items-center justify-between px-3 py-2 text-left transition-colors',
                      isSelected
                        ? 'bg-amber-600/15 text-amber-400'
                        : 'text-foreground hover:bg-muted/50'
                    )}
                  >
                    <div className="min-w-0">
                      <div className="flex items-center gap-1.5">
                        <span className="truncate text-sm font-medium">{label}</span>
                        {hasLiveSupport && (
                          <Radio className="h-2.5 w-2.5 shrink-0 text-emerald-500/60" />
                        )}
                      </div>
                      <div className="text-muted-foreground font-mono text-[10px]">{ticker}</div>
                    </div>
                    {volume > 0 && (
                      <span className="text-muted-foreground ml-2 shrink-0 font-mono text-[10px]">
                        {volume >= 1000 ? `${(volume / 1000).toFixed(1)}k` : volume}
                      </span>
                    )}
                  </button>
                );
              })
            )}
          </div>
        </div>

        {/* Right: Market table (live sports view or raw fallback) */}
        <div className="bg-card border-border flex min-w-0 flex-1 flex-col rounded-lg border">
          {!devData.selectedTicker ? (
            <div className="text-muted-foreground flex flex-1 items-center justify-center text-sm">
              Select a series to view markets
            </div>
          ) : isLiveView ? (
            <LiveSportsPanel
              config={liveConfig}
              games={liveGames}
              consolidatedGames={consolidatedMoneylineGames}
              loading={liveLoading}
              loadingState={liveLoadingState}
              seriesLabel={getSeriesLabel(devData.selectedTicker, selectedSeriesObj?.title)}
              seriesTicker={devData.selectedTicker}
              expandedGameKeys={expandedGameKeys}
              onToggleExpand={handleToggleExpand}
              onPlaceOrder={onPlaceOrder}
            />
          ) : (
            <RawMarketPanel
              seriesLabel={getSeriesLabel(devData.selectedTicker, selectedSeriesObj?.title)}
              seriesTicker={devData.selectedTicker}
              marketStats={marketStats}
              sortedMarkets={sortedMarkets}
              orderbooks={devData.orderbooks}
              loading={devData.loading}
              activeOnly={activeOnly}
              onPlaceOrder={onPlaceOrder}
            />
          )}
        </div>
      </div>
    </div>
  );
}

// ---------------------------------------------------------------------------
// Live Sports Panel — renders TwoRowMoneylines / TwoRowSpreads / TwoRowTotals
// ---------------------------------------------------------------------------

function LiveSportsPanel({
  config,
  games,
  consolidatedGames,
  loading,
  loadingState,
  seriesLabel,
  seriesTicker,
  expandedGameKeys,
  onToggleExpand,
  onPlaceOrder,
}: {
  config: { sport: Sport; marketType: MarketType };
  games: GameData[];
  consolidatedGames: ConsolidatedGameBooks[];
  loading: boolean;
  loadingState?: SportsLoadingState;
  seriesLabel: string;
  seriesTicker: string;
  expandedGameKeys: Set<string>;
  onToggleExpand: (gameKey: string) => void;
  onPlaceOrder?: (ticker: string, price?: number) => void;
}) {
  // Enrich moneyline games with rotation numbers from The Odds API
  const enrichedMoneylineGames = useRotationNumbers(config.sport, consolidatedGames);

  const gameCount = games.length;

  // Data coverage summary
  const dataSummary = useMemo(() => {
    if (gameCount === 0) return null;
    if (config.marketType === 'moneyline') {
      const withKalshi = enrichedMoneylineGames.filter((g) => !!g.kalshi).length;
      const withPoly = enrichedMoneylineGames.filter((g) => !!g.polymarket).length;
      return { withKalshi, withPoly, total: enrichedMoneylineGames.length };
    }
    if (config.marketType === 'spread') {
      const sg = games as SpreadGameData[];
      const withKalshi = sg.filter((g) => g.markets.size > 0).length;
      const withPoly = sg.filter((g) => !!g.polymarketMain).length;
      return { withKalshi, withPoly, total: sg.length };
    }
    if (config.marketType === 'total') {
      const tg = games as TotalGameData[];
      const withKalshi = tg.filter((g) => g.markets.size > 0).length;
      const withPoly = tg.filter((g) => !!g.polymarketMain).length;
      return { withKalshi, withPoly, total: tg.length };
    }
    return null;
  }, [games, enrichedMoneylineGames, gameCount, config.marketType]);

  // Map market tickers → game keys so clicking odds cells can expand the row
  const tickerToGameKey = useMemo(() => {
    const map = new Map<string, string>();
    if (config.marketType === 'moneyline') {
      for (const g of enrichedMoneylineGames) {
        // Kalshi market tickers
        if (g.kalshi?.markets.away.marketTicker) map.set(g.kalshi.markets.away.marketTicker, g.key);
        if (g.kalshi?.markets.home.marketTicker) map.set(g.kalshi.markets.home.marketTicker, g.key);
        // Polymarket tickers
        if (g.polymarket?.markets.away.marketTicker)
          map.set(g.polymarket.markets.away.marketTicker, g.key);
        if (g.polymarket?.markets.home.marketTicker)
          map.set(g.polymarket.markets.home.marketTicker, g.key);
      }
    }
    // Spreads/totals: use eventTicker as key
    for (const g of games) {
      if ('markets' in g && g.marketType === 'spread') {
        const sg = g as SpreadGameData;
        for (const [, info] of sg.markets) {
          if (info.book.marketTicker) map.set(info.book.marketTicker, sg.eventTicker);
        }
      }
      if ('markets' in g && g.marketType === 'total') {
        const tg = g as TotalGameData;
        for (const [, book] of tg.markets) {
          if (book.marketTicker) map.set(book.marketTicker, tg.eventTicker);
        }
      }
    }
    return map;
  }, [enrichedMoneylineGames, games, config.marketType]);

  // Primary click on odds cell → toggle row expansion (same as main dashboard)
  const handleNavigateToMarket = useCallback(
    (marketTicker: string) => {
      const gameKey = tickerToGameKey.get(marketTicker);
      if (gameKey) {
        onToggleExpand(gameKey);
      }
    },
    [tickerToGameKey, onToggleExpand]
  );

  // Trade click handler — adapts TradeClickRequest to onPlaceOrder
  const handleTradeClick = useCallback(
    (req: { marketTicker: string; priceCents?: number }) => {
      if (!onPlaceOrder) return;
      const price = req.priceCents != null ? req.priceCents / 100 : undefined;
      onPlaceOrder(req.marketTicker, price);
    },
    [onPlaceOrder]
  );

  return (
    <>
      {/* Header */}
      <div className="border-border flex flex-col gap-1 border-b px-3 py-2">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <span className="text-sm font-medium text-amber-400">{seriesLabel}</span>
            <span className="text-muted-foreground font-mono text-[10px]">{seriesTicker}</span>
            <span className="text-muted-foreground text-xs">
              {gameCount} {gameCount === 1 ? 'game' : 'games'}
            </span>
          </div>
          <div className="flex items-center gap-2">
            {loading && <span className="text-muted-foreground text-[10px]">streaming...</span>}
            {!loading && gameCount > 0 && (
              <span className="text-xs text-emerald-400/70">
                <Radio className="mr-1 inline h-3 w-3" />
                live
              </span>
            )}
          </div>
        </div>
        {/* Data coverage summary */}
        {dataSummary && !loading && (
          <div className="text-muted-foreground flex items-center gap-3 text-[10px]">
            <span>
              Kalshi: {dataSummary.withKalshi}/{dataSummary.total} w/ books
            </span>
            <span>
              Poly: {dataSummary.withPoly}/{dataSummary.total} w/ books
            </span>
          </div>
        )}
      </div>

      {/* Table */}
      <div className="flex-1 overflow-auto">
        {config.marketType === 'moneyline' && (
          <TwoRowMoneylines
            games={enrichedMoneylineGames}
            isLoading={loading}
            loadingState={loadingState}
            oddsMode="maker"
            expandedGameKeys={expandedGameKeys}
            onToggleExpand={onToggleExpand}
            onTradeClick={handleTradeClick}
            onNavigateToMarket={handleNavigateToMarket}
            sport={config.sport}
            showEmptyHints
          />
        )}
        {config.marketType === 'spread' && (
          <TwoRowSpreads
            games={games as SpreadGameData[]}
            sport={config.sport}
            isLoading={loading}
            loadingState={loadingState}
            onTradeClick={handleTradeClick}
            onNavigateToMarket={handleNavigateToMarket}
            showEmptyHints
          />
        )}
        {config.marketType === 'total' && (
          <TwoRowTotals
            games={games as TotalGameData[]}
            sport={config.sport}
            isLoading={loading}
            loadingState={loadingState}
            onTradeClick={handleTradeClick}
            onNavigateToMarket={handleNavigateToMarket}
            showEmptyHints
          />
        )}
      </div>
    </>
  );
}

// ---------------------------------------------------------------------------
// Raw Market Panel — original flat table for unknown series
// ---------------------------------------------------------------------------

function RawMarketPanel({
  seriesLabel,
  seriesTicker,
  marketStats,
  sortedMarkets,
  orderbooks,
  loading,
  activeOnly,
  onPlaceOrder,
}: {
  seriesLabel: string;
  seriesTicker: string;
  marketStats: { total: number; active: number; totalVol: number; totalOI: number };
  sortedMarkets: KalshiMarket[];
  orderbooks: Map<string, KalshiOrderbook>;
  loading: boolean;
  activeOnly: boolean;
  onPlaceOrder?: (ticker: string, price?: number) => void;
}) {
  return (
    <>
      {/* Market table header */}
      <div className="border-border flex items-center justify-between border-b px-3 py-2">
        <div className="flex items-center gap-3">
          <span className="text-sm font-medium text-amber-400">{seriesLabel}</span>
          <span className="text-muted-foreground font-mono text-[10px]">{seriesTicker}</span>
          <span className="text-muted-foreground text-xs">
            {activeOnly
              ? `${marketStats.active} active / ${marketStats.total} total`
              : `${marketStats.total} markets`}
          </span>
          {marketStats.totalVol > 0 && (
            <span className="text-muted-foreground text-xs">
              Vol: {marketStats.totalVol.toLocaleString()} | OI:{' '}
              {marketStats.totalOI.toLocaleString()}
            </span>
          )}
        </div>
        <div className="flex items-center gap-2">
          {onPlaceOrder && (
            <span className="text-muted-foreground text-[10px]">click row to trade</span>
          )}
          {loading && <span className="text-muted-foreground text-[10px]">refreshing...</span>}
        </div>
      </div>

      {/* Market table */}
      <div className="flex-1 overflow-auto">
        {loading && sortedMarkets.length === 0 ? (
          <div className="text-muted-foreground p-4 text-center text-xs">Loading markets...</div>
        ) : sortedMarkets.length === 0 ? (
          <div className="text-muted-foreground p-4 text-center text-xs">
            {activeOnly
              ? 'No markets with activity in this series'
              : 'No open markets in this series'}
          </div>
        ) : (
          <table className="w-full text-xs">
            <thead className="sticky top-0">
              <tr className="border-border bg-card border-b">
                <th className="text-muted-foreground px-3 py-2 text-left font-medium">Ticker</th>
                <th className="text-muted-foreground px-3 py-2 text-left font-medium">Title</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Yes Bid</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Yes Ask</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Last</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Volume</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">OI</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Liq</th>
              </tr>
            </thead>
            <tbody>
              {sortedMarkets.map((m) => (
                <MarketRow
                  key={m.market_ticker}
                  market={m}
                  orderbook={orderbooks.get(m.market_ticker)}
                  onPlaceOrder={onPlaceOrder}
                />
              ))}
            </tbody>
          </table>
        )}
      </div>
    </>
  );
}

function MarketRow({
  market: m,
  orderbook: ob,
  onPlaceOrder,
}: {
  market: KalshiMarket;
  orderbook?: KalshiOrderbook;
  onPlaceOrder?: (ticker: string, price?: number) => void;
}) {
  const yesBid = ob?.yes?.[0]?.[0] ?? m.yes_bid;
  const noBid = ob?.no?.[0]?.[0] ?? m.no_bid;
  const yesAsk = noBid != null ? 100 - noBid : m.yes_ask;
  const lastPrice = m.last_price;

  const handleClick = onPlaceOrder
    ? () => {
        // Convert cents (0-100) to decimal (0-1) for OrderTicket
        const price = yesBid != null ? yesBid / 100 : undefined;
        onPlaceOrder(m.market_ticker, price);
      }
    : undefined;

  return (
    <tr
      className={cn(
        'border-border hover:bg-muted/30 border-b last:border-b-0',
        onPlaceOrder && 'cursor-pointer'
      )}
      onClick={handleClick}
    >
      <td className="px-3 py-1.5 font-mono text-[10px]">{m.market_ticker}</td>
      <td className="max-w-xs truncate px-3 py-1.5">{m.title}</td>
      <td className="px-3 py-1.5 text-right font-mono">{yesBid != null ? `${yesBid}¢` : '—'}</td>
      <td className="px-3 py-1.5 text-right font-mono">{yesAsk != null ? `${yesAsk}¢` : '—'}</td>
      <td
        className={cn(
          'px-3 py-1.5 text-right font-mono',
          lastPrice != null ? 'text-foreground' : 'text-muted-foreground'
        )}
      >
        {lastPrice != null ? `${lastPrice}¢` : '—'}
      </td>
      <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
        {m.volume.toLocaleString()}
      </td>
      <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
        {m.open_interest.toLocaleString()}
      </td>
      <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
        {m.liquidity.toLocaleString()}
      </td>
    </tr>
  );
}
