/**
 * GateView - Password gate screen
 *
 * Displays before the Kalshi login screen. Users must enter the correct
 * password to proceed. Password is verified via SHA-256 hash comparison.
 */

import { useState, useCallback } from 'react';

interface GateViewProps {
  onUnlock: (verifiedHash: string) => void;
}

/**
 * Hash a string using SHA-256 via Web Crypto API
 */
async function sha256(message: string): Promise<string> {
  const encoder = new TextEncoder();
  const data = encoder.encode(message);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map((b) => b.toString(16).padStart(2, '0')).join('');
}

export function GateView({ onUnlock }: GateViewProps) {
  const [password, setPassword] = useState('');
  const [error, setError] = useState<string | null>(null);
  const [isVerifying, setIsVerifying] = useState(false);

  const expectedHash = import.meta.env.VITE_DASHBOARD_PASSWORD_HASH || '';

  const handleSubmit = useCallback(
    async (e: React.FormEvent) => {
      e.preventDefault();

      if (!password.trim()) {
        setError('Please enter a password');
        return;
      }

      if (!expectedHash) {
        setError('Dashboard access is not configured');
        return;
      }

      setIsVerifying(true);
      setError(null);

      try {
        const inputHash = await sha256(password);

        if (inputHash === expectedHash) {
          onUnlock(inputHash);
        } else {
          setError('Incorrect password');
          setPassword('');
        }
      } catch (err) {
        console.error('Hash verification failed:', err);
        setError('Verification failed');
      } finally {
        setIsVerifying(false);
      }
    },
    [password, expectedHash, onUnlock]
  );

  return (
    <div className="bg-background flex min-h-screen flex-col items-center justify-center p-4">
      <div className="w-full max-w-md">
        {/* Logo/Title */}
        <div className="mb-8 text-center">
          <h1 className="text-foreground text-2xl font-bold">Kalshi Terminal</h1>
          <p className="text-muted-foreground mt-2 text-sm">Enter password to continue</p>
        </div>

        {/* Password Form */}
        <div className="bg-card border-border rounded-lg border p-6">
          <form onSubmit={handleSubmit}>
            <div className="form-group">
              <label htmlFor="gate-password">Dashboard Password:</label>
              <input
                type="password"
                id="gate-password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="Enter password"
                autoFocus
                disabled={isVerifying}
              />
            </div>
            <div className="form-group">
              <button
                type="submit"
                className="btn-primary w-full"
                disabled={isVerifying || !password.trim()}
              >
                {isVerifying ? 'Verifying...' : 'Continue'}
              </button>
            </div>
          </form>
        </div>

        {/* Error Display */}
        {error && (
          <div className="bg-destructive/10 border-destructive/20 mt-4 rounded-lg border p-3">
            <p className="text-destructive text-sm">{error}</p>
          </div>
        )}
      </div>
    </div>
  );
}
