/**
 * MarketDetailView - Market Maker Optimized Trading Interface
 *
 * Designed for market makers based on the BuildingForMarketMakers guide.
 * Prioritizes: Position, Queue Position, Active Orders, P&L, Order Entry
 * De-prioritizes: Charts (collapsible), News, Historical data
 *
 * Layout:
 * - Header: Market info, prices, P&L (always visible)
 * - Main: Order Book | Order Entry | Orders & Positions
 * - Footer: Recent Fills Tape, Collapsible Chart
 */

import { useState, useEffect, useCallback, useRef, useMemo } from 'react';
import { Link } from 'react-router-dom';
import {
  ChevronRight,
  TrendingUp,
  TrendingDown,
  Activity,
  RefreshCw,
  Wifi,
  X,
  Zap,
  BarChart3,
  Keyboard,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { CandlestickChart } from '@/components/organisms/CandlestickChart';
import { OrderBook } from '@/components/organisms/OrderBook';
import { SweepConfirmPanel } from '@/components/molecules/SweepConfirmPanel';
import { TradeTape } from '@/components/organisms/TradeTape';
import { LoadingSpinner } from '@/components/atoms/LoadingState';
import { Badge } from '@/components/atoms/Badge';
import { SideBadge } from '@/components/atoms/SideBadge';
import { Tooltip } from '@/components/atoms/Tooltip';
import { useToast } from '@/components/organisms/Toast';
import { createMarketStream, type MarketStream, type OrderbookUpdate } from '@/lib/marketStream';
import { initAuth } from '@/lib/kalshiAuth';
import { useFlashOnChange } from '@/hooks/useFlash';
import type {
  KalshiApiClient,
  KalshiCandlestick,
  KalshiOrder,
  KalshiFill,
  KalshiTrade,
} from '@/lib/kalshiApi';
import type {
  KalshiMarket,
  KalshiCredentials,
  PriceUpdate,
  KalshiSeries,
  KalshiEvent,
  Position,
} from '@/types';
import type { MockCandle } from '@/mock/candlesticks';

interface MarketDetailViewProps {
  market?: KalshiMarket;
  marketTicker?: string;
  apiClient: KalshiApiClient;
  credentials?: KalshiCredentials;
  className?: string;
}

interface StreamingPrices {
  lastPrice: number | null;
  yesBid: number | null;
  yesAsk: number | null;
  previousPrice: number | null;
}

interface OrderbookLevel {
  price: number;
  quantity: number;
  cumulative: number;
}

// Simple fill type for the tape
interface TapeFill {
  id: string;
  timestamp: Date;
  side: 'yes' | 'no';
  action: 'buy' | 'sell';
  price: number;
  quantity: number;
  isMine?: boolean;
}

export function MarketDetailView({
  market: marketProp,
  marketTicker,
  apiClient,
  credentials,
  className,
}: MarketDetailViewProps) {
  const { addToast } = useToast();

  // ==================== MARKET DATA STATE ====================
  const [market, setMarket] = useState<KalshiMarket | null>(marketProp ?? null);
  const [marketLoading, setMarketLoading] = useState(!marketProp);
  const [marketError, setMarketError] = useState<string | null>(null);

  // Breadcrumb data
  const [seriesData, setSeriesData] = useState<KalshiSeries | null>(null);
  const [eventData, setEventData] = useState<KalshiEvent | null>(null);

  // Orderbook
  const [bids, setBids] = useState<OrderbookLevel[]>([]);
  const [asks, setAsks] = useState<OrderbookLevel[]>([]);
  const [orderbookLoading, setOrderbookLoading] = useState(true);

  // Chart (collapsible - low priority for MMs)
  const [chartData, setChartData] = useState<MockCandle[]>([]);
  const [chartLoading, setChartLoading] = useState(true);

  // Streaming prices
  const [prices, setPrices] = useState<StreamingPrices>({
    lastPrice: marketProp?.last_price ?? null,
    yesBid: marketProp?.yes_bid ?? null,
    yesAsk: marketProp?.yes_ask ?? null,
    previousPrice: marketProp?.previous_price ?? null,
  });

  // ==================== MM-CRITICAL STATE ====================
  // Position in this market
  const [position, setPosition] = useState<Position | null>(null);
  const [positionLoading, setPositionLoading] = useState(true);

  // Active orders
  const [activeOrders, setActiveOrders] = useState<KalshiOrder[]>([]);
  const [ordersLoading, setOrdersLoading] = useState(true);

  // Recent fills (user's fills)
  const [recentFills, setRecentFills] = useState<TapeFill[]>([]);
  const [fillsLoading, setFillsLoading] = useState(true);

  // Market trades (public trade tape)
  const [marketTrades, setMarketTrades] = useState<KalshiTrade[]>([]);
  const [tradesLoading, setTradesLoading] = useState(true);

  // Order form state
  const [orderSide, setOrderSide] = useState<'buy' | 'sell'>('buy');
  const [orderPosition, setOrderPosition] = useState<'yes' | 'no'>('no'); // Default to NO for MMs
  const [orderPrice, setOrderPrice] = useState<number>(50);
  const [orderQuantity, setOrderQuantity] = useState<number>(10);
  const [orderSubmitting, setOrderSubmitting] = useState(false);
  const [orderError, setOrderError] = useState<string | null>(null);

  // Sweep state
  const [sweepState, setSweepState] = useState<{
    side: 'buy' | 'sell';
    position: 'yes' | 'no';
    limitPrice: number;
    levels: Array<{ price: number; quantity: number }>;
    totalContracts: number;
  } | null>(null);
  const [sweepSubmitting, setSweepSubmitting] = useState(false);
  const [sweepError, setSweepError] = useState<string | null>(null);

  // Keyboard shortcuts active
  const [showShortcuts, setShowShortcuts] = useState(false);

  // Stream refs
  const streamRef = useRef<MarketStream | null>(null);
  const pollingRef = useRef<NodeJS.Timeout | null>(null);
  const [isStreaming, setIsStreaming] = useState(false);
  const [isOrderbookStreaming, setIsOrderbookStreaming] = useState(false);

  // ==================== FETCH MARKET DATA ====================
  useEffect(() => {
    if (marketProp) {
      setMarket(marketProp);
      setPrices({
        lastPrice: marketProp.last_price,
        yesBid: marketProp.yes_bid,
        yesAsk: marketProp.yes_ask,
        previousPrice: marketProp.previous_price,
      });
      // Set default order price to current bid
      if (marketProp.yes_bid) {
        setOrderPrice(marketProp.yes_bid);
      }
      return;
    }

    if (!marketTicker) return;

    const fetchMarket = async () => {
      setMarketLoading(true);
      setMarketError(null);

      try {
        const parts = marketTicker.split('-');
        const eventTicker = parts.length > 1 ? parts.slice(0, -1).join('-') : marketTicker;
        const markets = await apiClient.getMarkets({ event_ticker: eventTicker, limit: 100 });
        const found = markets.find((m) => m.market_ticker === marketTicker);

        if (!found) {
          setMarketError(`Market not found: ${marketTicker}`);
          setMarketLoading(false);
          return;
        }

        setMarket(found);
        setPrices({
          lastPrice: found.last_price,
          yesBid: found.yes_bid,
          yesAsk: found.yes_ask,
          previousPrice: found.previous_price,
        });
        if (found.yes_bid) {
          setOrderPrice(found.yes_bid);
        }
        setMarketLoading(false);
      } catch (err) {
        setMarketError(err instanceof Error ? err.message : 'Failed to load market');
        setMarketLoading(false);
      }
    };

    fetchMarket();
  }, [marketProp, marketTicker, apiClient]);

  // Fetch breadcrumb data
  useEffect(() => {
    if (!market) return;

    const fetchBreadcrumbData = async () => {
      try {
        const allSeries = await apiClient.getSeries();
        const series = allSeries.find(
          (s) => s.series_ticker === market.series_ticker || s.ticker === market.series_ticker
        );
        setSeriesData(series ?? null);

        if (market.event_ticker) {
          const events = await apiClient.getEvents({
            series_ticker: market.series_ticker,
            limit: 200,
          });
          const event = events.find((e) => e.event_ticker === market.event_ticker);
          if (event) {
            setEventData(event);
          }
        }
      } catch (err) {
        console.warn('Could not fetch breadcrumb data:', err);
      }
    };

    fetchBreadcrumbData();
  }, [market, apiClient]);

  // Build breadcrumbs
  const breadcrumbs = useMemo(() => {
    if (!market) return [];

    const category = seriesData?.category ?? eventData?.category;
    const crumbs: { label: string; to?: string; ticker?: string }[] = [
      { label: 'Categories', to: '/app/explore' },
    ];

    if (category) {
      crumbs.push({ label: category, to: `/app/explore/${encodeURIComponent(category)}` });
    }

    if (market.series_ticker) {
      crumbs.push({
        label: seriesData?.title ?? market.series_ticker,
        ticker: market.series_ticker,
        to: category
          ? `/app/explore/${encodeURIComponent(category)}/${encodeURIComponent(market.series_ticker)}`
          : undefined,
      });
    }

    if (market.event_ticker) {
      // Build event link - try full path, fallback to simpler paths
      let eventPath: string | undefined;
      if (category && market.series_ticker) {
        eventPath = `/app/explore/${encodeURIComponent(category)}/${encodeURIComponent(market.series_ticker)}/${encodeURIComponent(market.event_ticker)}`;
      } else if (market.series_ticker) {
        // Link to series page with event
        eventPath = `/app/explore?series=${encodeURIComponent(market.series_ticker)}&event=${encodeURIComponent(market.event_ticker)}`;
      }
      crumbs.push({
        label: eventData?.title ?? market.event_ticker,
        ticker: market.event_ticker,
        to: eventPath,
      });
    }

    crumbs.push({ label: market.title });
    return crumbs;
  }, [market, seriesData, eventData]);

  // ==================== FETCH ORDERBOOK ====================
  const fetchOrderbook = useCallback(async () => {
    if (!market) return;

    try {
      const response = await apiClient.getOrderbook(market.market_ticker);
      if (!response) {
        setBids([]);
        setAsks([]);
        return;
      }

      const rawResponse = response as unknown as Record<string, unknown>;
      const orderbookData = rawResponse.orderbook ?? rawResponse;
      const orderbook = orderbookData as Record<string, unknown>;

      const yesData = orderbook.yes as Array<[number, number]> | undefined;
      const noData = orderbook.no as Array<[number, number]> | undefined;

      const yesBids = parseOrderbookSide(yesData);
      const noBids = parseOrderbookSide(noData);
      const yesAsks = noBids.map((level) => ({ ...level, price: 100 - level.price }));

      const sortedBids = addCumulatives(yesBids.sort((a, b) => b.price - a.price));
      const sortedAsks = addCumulatives(yesAsks.sort((a, b) => a.price - b.price));

      setBids(sortedBids);
      setAsks(sortedAsks);
    } catch (err) {
      console.error('Failed to fetch orderbook:', err);
    } finally {
      setOrderbookLoading(false);
    }
  }, [apiClient, market?.market_ticker]);

  // ==================== FETCH POSITION (MM CRITICAL) ====================
  const fetchPosition = useCallback(async () => {
    if (!market || !credentials) {
      setPositionLoading(false);
      return;
    }

    try {
      const positions = await apiClient.getPositions();
      const pos = positions.find((p) => p.ticker === market.market_ticker);

      if (pos) {
        const currentPrice = prices.lastPrice ?? market.last_price ?? 50;
        const avgPrice = pos.average_price;
        const quantity = Math.abs(pos.position);
        const unrealizedPnl = ((currentPrice - avgPrice) * quantity) / 100;

        setPosition({
          ticker: pos.ticker,
          side: pos.side as 'yes' | 'no',
          quantity,
          averagePrice: avgPrice,
          currentPrice,
          realizedPnl: pos.realized_pnl ?? 0,
          unrealizedPnl,
          totalPnl: (pos.realized_pnl ?? 0) + unrealizedPnl,
          pnlPercent: avgPrice > 0 ? ((currentPrice - avgPrice) / avgPrice) * 100 : 0,
        });
      } else {
        setPosition(null);
      }
    } catch (err) {
      console.error('Failed to fetch position:', err);
      setPosition(null);
    } finally {
      setPositionLoading(false);
    }
  }, [apiClient, market?.market_ticker, credentials, prices.lastPrice]);

  // ==================== FETCH ACTIVE ORDERS (MM CRITICAL) ====================
  const fetchActiveOrders = useCallback(async () => {
    if (!market || !credentials) {
      setOrdersLoading(false);
      return;
    }

    try {
      const orders = await apiClient.getOpenOrders();
      // Filter to just this market's orders
      const marketOrders = orders.filter((o) => o.ticker === market.market_ticker);
      setActiveOrders(marketOrders);
    } catch (err) {
      console.error('Failed to fetch orders:', err);
      setActiveOrders([]);
    } finally {
      setOrdersLoading(false);
    }
  }, [apiClient, market?.market_ticker, credentials]);

  // ==================== FETCH FILLS (USER'S FILLS) ====================
  const fetchFills = useCallback(async () => {
    if (!market || !credentials) {
      setFillsLoading(false);
      return;
    }

    try {
      const fills = await apiClient.getFilledOrders({ limit: 20 });
      // Filter to just this market's fills
      const marketFills = fills.filter((f) => f.ticker === market.market_ticker);
      const tapeFills: TapeFill[] = marketFills.map((f: KalshiFill) => ({
        id: f.fill_id ?? String(Math.random()),
        timestamp: new Date(f.created_at),
        side: f.side as 'yes' | 'no',
        action: f.action as 'buy' | 'sell',
        price: f.price,
        quantity: f.count,
        isMine: true, // These are the user's fills
      }));
      setRecentFills(tapeFills);
    } catch (err) {
      console.error('Failed to fetch fills:', err);
      setRecentFills([]);
    } finally {
      setFillsLoading(false);
    }
  }, [apiClient, market?.market_ticker, credentials]);

  // ==================== FETCH MARKET TRADES (PUBLIC TAPE) ====================
  const fetchMarketTrades = useCallback(async () => {
    if (!market) {
      setTradesLoading(false);
      return;
    }

    try {
      const trades = await apiClient.getMarketTrades({ ticker: market.market_ticker, limit: 50 });
      setMarketTrades(trades);
    } catch (err) {
      console.error('Failed to fetch market trades:', err);
      setMarketTrades([]);
    } finally {
      setTradesLoading(false);
    }
  }, [apiClient, market?.market_ticker]);

  // ==================== FETCH CANDLESTICKS ====================
  const fetchCandlesticks = useCallback(async () => {
    if (!market) return;

    try {
      const now = Math.floor(Date.now() / 1000);
      const oneDayAgo = now - 24 * 60 * 60;

      const candlesticks = await apiClient.getCandlesticks(
        market.series_ticker,
        market.market_ticker,
        {
          start_ts: oneDayAgo,
          end_ts: now,
          period_interval: 1,
        }
      );

      const validCandlesticks = candlesticks.filter(
        (c) =>
          c.price?.open != null &&
          c.price?.high != null &&
          c.price?.low != null &&
          c.price?.close != null
      );

      if (validCandlesticks.length > 0) {
        setChartData(validCandlesticks.map(convertToChartCandle));
      } else {
        setChartData(generatePlaceholderChart(market.last_price ?? 50));
      }
    } catch (err) {
      console.error('Failed to fetch candlesticks:', err);
      setChartData(generatePlaceholderChart(market?.last_price ?? 50));
    } finally {
      setChartLoading(false);
    }
  }, [apiClient, market?.series_ticker, market?.market_ticker, market?.last_price]);

  // ==================== ORDER SUBMISSION ====================
  const handleSubmitOrder = useCallback(async () => {
    setOrderError(null);

    if (!credentials) {
      setOrderError('No credentials configured. Go to Settings to connect.');
      return;
    }
    if (!market) {
      setOrderError('Market not loaded. Please wait or refresh.');
      return;
    }
    if (orderSubmitting) {
      return;
    }

    setOrderSubmitting(true);
    try {
      await apiClient.placeOrder({
        ticker: market.market_ticker,
        side: orderPosition,
        action: orderSide,
        type: 'limit',
        count: orderQuantity,
        price: orderPrice,
      });

      // Show success toast
      addToast({
        type: 'success',
        title: 'Order Submitted',
        message: `${orderSide.toUpperCase()} ${orderQuantity} ${orderPosition.toUpperCase()} @ ${orderPrice}¢`,
        duration: 3000,
      });

      // Refresh orders after submission
      await fetchActiveOrders();
      // Reset quantity
      setOrderQuantity(10);
    } catch (err) {
      const message = err instanceof Error ? err.message : 'Unknown error';
      console.error('Failed to submit order:', err);
      setOrderError(message);
    } finally {
      setOrderSubmitting(false);
    }
  }, [
    market,
    credentials,
    apiClient,
    orderSide,
    orderPosition,
    orderPrice,
    orderQuantity,
    fetchActiveOrders,
    orderSubmitting,
    addToast,
  ]);

  // ==================== CANCEL ORDER ====================
  const handleCancelOrder = useCallback(
    async (orderId: string) => {
      try {
        await apiClient.cancelOrder(orderId);
        await fetchActiveOrders();
      } catch (err) {
        console.error('Failed to cancel order:', err);
      }
    },
    [apiClient, fetchActiveOrders]
  );

  // ==================== CANCEL ALL ORDERS ====================
  const handleCancelAll = useCallback(async () => {
    if (!market) return;
    try {
      for (const order of activeOrders) {
        await apiClient.cancelOrder(order.order_id);
      }
      await fetchActiveOrders();
    } catch (err) {
      console.error('Failed to cancel all orders:', err);
    }
  }, [market, activeOrders, apiClient, fetchActiveOrders]);

  // ==================== ORDER BOOK CLICK HANDLER ====================
  // Clicking a bid level = sell at that price (hit the bid)
  // Clicking an ask level = buy at that price (lift the offer)
  // Shift+click = stage a sweep
  const handleOrderbookLevelClick = useCallback(
    (
      price: number,
      _quantity: number,
      _cumulative: number,
      side: 'bid' | 'ask',
      shiftKey: boolean
    ) => {
      if (shiftKey) {
        // Sweep mode: collect all levels up to the clicked price
        const sweepSide: 'buy' | 'sell' = side === 'ask' ? 'buy' : 'sell';
        const source = side === 'ask' ? asks : bids;
        const sweptLevels = source
          .filter((l) => (side === 'ask' ? l.price <= price : l.price >= price))
          .map((l) => ({ price: l.price, quantity: l.quantity }));

        if (sweptLevels.length === 0) return;

        const totalContracts = sweptLevels.reduce((sum, l) => sum + l.quantity, 0);

        setSweepState({
          side: sweepSide,
          position: 'yes',
          limitPrice: price,
          levels: sweptLevels,
          totalContracts,
        });
        setSweepError(null);
      } else {
        // Normal click: populate Quick Order form
        setOrderPrice(price);
        setOrderSide(side === 'bid' ? 'sell' : 'buy');
      }
    },
    [asks, bids]
  );

  // ==================== SWEEP SUBMIT HANDLER ====================
  const handleSweepSubmit = useCallback(async () => {
    if (!sweepState || !market || !credentials) return;

    setSweepSubmitting(true);
    setSweepError(null);

    try {
      await apiClient.placeOrder({
        ticker: market.market_ticker,
        side: sweepState.position,
        action: sweepState.side,
        type: 'limit',
        count: sweepState.totalContracts,
        price: sweepState.limitPrice,
        time_in_force: 'immediate_or_cancel',
      });

      addToast({
        type: 'success',
        title: 'Sweep Executed',
        message: `${sweepState.side.toUpperCase()} ${sweepState.totalContracts.toLocaleString()} ${sweepState.position.toUpperCase()} @ ${sweepState.limitPrice}¢ (IOC)`,
        duration: 4000,
      });

      setSweepState(null);
      await fetchActiveOrders();
      await fetchPosition();
    } catch (err) {
      const message = err instanceof Error ? err.message : 'Sweep failed';
      setSweepError(message);
    } finally {
      setSweepSubmitting(false);
    }
  }, [sweepState, market, credentials, apiClient, addToast, fetchActiveOrders, fetchPosition]);

  // ==================== TRADE TAPE CLICK HANDLER ====================
  const handleTradeClick = useCallback((trade: KalshiTrade) => {
    setOrderPrice(trade.yes_price);
  }, []);

  // ==================== KEYBOARD SHORTCUTS ====================
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      // Skip if typing in an input
      if (e.target instanceof HTMLInputElement || e.target instanceof HTMLTextAreaElement) {
        return;
      }

      // B = Buy
      if (e.key === 'b' || e.key === 'B') {
        e.preventDefault();
        e.stopPropagation();
        setOrderSide('buy');
        return;
      }
      // S = Sell
      if (e.key === 's' || e.key === 'S') {
        e.preventDefault();
        e.stopPropagation();
        setOrderSide('sell');
        return;
      }
      // Y = Yes position
      if (e.key === 'y' || e.key === 'Y') {
        e.preventDefault();
        e.stopPropagation();
        setOrderPosition('yes');
        return;
      }
      // N = No position (prevent App.tsx from opening order ticket)
      if (e.key === 'n' || e.key === 'N') {
        e.preventDefault();
        e.stopPropagation();
        setOrderPosition('no');
        return;
      }
      // Enter = Submit order
      if (e.key === 'Enter' && e.ctrlKey) {
        e.preventDefault();
        e.stopPropagation();
        handleSubmitOrder();
        return;
      }
      // Escape = dismiss sweep panel (plain) or cancel all (Shift)
      if (e.key === 'Escape') {
        e.preventDefault();
        e.stopPropagation();
        if (e.shiftKey) {
          handleCancelAll();
        } else {
          setSweepState(null);
          setSweepError(null);
        }
        return;
      }
      // ? = Show shortcuts
      if (e.key === '?') {
        e.preventDefault();
        e.stopPropagation();
        setShowShortcuts((prev) => !prev);
        return;
      }
    };

    // Use capture phase to run before App.tsx global handlers
    window.addEventListener('keydown', handleKeyDown, { capture: true });
    return () => window.removeEventListener('keydown', handleKeyDown, { capture: true });
  }, [handleSubmitOrder, handleCancelAll]);

  // ==================== PRICE UPDATE HANDLER ====================
  const handlePriceUpdate = useCallback(
    (update: PriceUpdate) => {
      if (!market || update.ticker !== market.market_ticker) return;

      const yesBidCents =
        update.yes_bid !== undefined ? Math.round(update.yes_bid * 100) : undefined;
      const yesAskCents =
        update.yes_ask !== undefined ? Math.round(update.yes_ask * 100) : undefined;

      setPrices((prev) => ({
        lastPrice:
          yesBidCents !== undefined && yesAskCents !== undefined
            ? Math.round((yesBidCents + yesAskCents) / 2)
            : prev.lastPrice,
        yesBid: yesBidCents ?? prev.yesBid,
        yesAsk: yesAskCents ?? prev.yesAsk,
        previousPrice: prev.lastPrice,
      }));

      const newPrice =
        yesBidCents !== undefined && yesAskCents !== undefined
          ? Math.round((yesBidCents + yesAskCents) / 2)
          : null;

      if (newPrice !== null) {
        setChartData((prev) => updateChartWithTick(prev, newPrice));
      }
    },
    [market?.market_ticker]
  );

  const handleOrderbookUpdate = useCallback(
    (update: OrderbookUpdate) => {
      if (!market || update.ticker !== market.market_ticker) return;

      // Convert to the same shape the existing parser expects.
      const yesData = update.yes.map((l) => [l.price, l.quantity] as [number, number]);
      const noData = update.no.map((l) => [l.price, l.quantity] as [number, number]);

      const yesBids = parseOrderbookSide(yesData);
      const noBids = parseOrderbookSide(noData);
      const yesAsks = noBids.map((level) => ({ ...level, price: 100 - level.price }));

      const sortedBids = addCumulatives(yesBids.sort((a, b) => b.price - a.price));
      const sortedAsks = addCumulatives(yesAsks.sort((a, b) => a.price - b.price));

      setBids(sortedBids);
      setAsks(sortedAsks);
      setOrderbookLoading(false);
      setIsOrderbookStreaming(true);
    },
    [market?.market_ticker]
  );

  // ==================== INITIAL DATA FETCH ====================
  useEffect(() => {
    fetchOrderbook();
    fetchCandlesticks();
    fetchPosition();
    fetchActiveOrders();
    fetchFills();
    fetchMarketTrades();
  }, [
    fetchOrderbook,
    fetchCandlesticks,
    fetchPosition,
    fetchActiveOrders,
    fetchFills,
    fetchMarketTrades,
  ]);

  // ==================== STREAMING SETUP ====================
  useEffect(() => {
    let mounted = true;

    const setupStreaming = async () => {
      if (!credentials) {
        pollingRef.current = setInterval(() => {
          fetchOrderbook();
          fetchPosition();
          fetchActiveOrders();
          fetchMarketTrades();
        }, 5000);
        return;
      }

      try {
        const { privateKey } = await initAuth(credentials.accessKeyId, credentials.privateKeyPem);
        if (!mounted) return;

        const stream = createMarketStream(true);
        streamRef.current = stream;

        stream.onPriceUpdate(handlePriceUpdate);
        stream.onOrderbookUpdate(handleOrderbookUpdate);
        stream.onError((error) => {
          console.error('Stream error:', error);
          setIsStreaming(false);
          setIsOrderbookStreaming(false);
          if (!pollingRef.current) {
            pollingRef.current = setInterval(() => {
              fetchOrderbook();
              fetchPosition();
              fetchActiveOrders();
              fetchMarketTrades();
            }, 5000);
          }
        });

        await stream.connect(credentials.accessKeyId, privateKey, credentials.environment);

        if (!mounted) {
          stream.disconnect();
          return;
        }

        if (market) {
          stream.subscribe([market.market_ticker]);
          // Live depth (snapshots + deltas) if Kalshi provides it.
          stream.subscribeOrderbook([market.market_ticker]);

          // If we don't get any depth updates soon, keep polling.
          setTimeout(() => {
            if (mounted && !isOrderbookStreaming) {
              setIsOrderbookStreaming(false); // UI indicator only; polling already runs as fallback
            }
          }, 5000);
        }
        setIsStreaming(true);

        // Poll orders/trades/position even when streaming. Orderbook polling handled by separate effect.
        pollingRef.current = setInterval(() => {
          fetchActiveOrders();
          fetchPosition();
          fetchMarketTrades();
        }, 3000);
      } catch (error) {
        console.error('Failed to set up streaming:', error);
        if (mounted && !pollingRef.current) {
          pollingRef.current = setInterval(() => {
            fetchOrderbook();
            fetchPosition();
            fetchActiveOrders();
            fetchMarketTrades();
          }, 5000);
        }
      }
    };

    setupStreaming();

    return () => {
      mounted = false;
      if (pollingRef.current) {
        clearInterval(pollingRef.current);
        pollingRef.current = null;
      }
      if (streamRef.current) {
        streamRef.current.disconnect();
        streamRef.current = null;
      }
      setIsStreaming(false);
      setIsOrderbookStreaming(false);
    };
  }, [
    credentials,
    market?.market_ticker,
    fetchPosition,
    fetchActiveOrders,
    fetchMarketTrades,
    handlePriceUpdate,
    handleOrderbookUpdate,
  ]);

  // ==================== ORDERBOOK POLLING (fallback when not streaming) ====================
  useEffect(() => {
    if (isOrderbookStreaming) return;

    const interval = setInterval(fetchOrderbook, 3000);
    return () => clearInterval(interval);
  }, [isOrderbookStreaming, fetchOrderbook]);

  // ==================== COMPUTED VALUES ====================
  const spread =
    prices.yesBid !== null && prices.yesAsk !== null ? prices.yesAsk - prices.yesBid : undefined;
  const priceChange =
    prices.lastPrice !== null && prices.previousPrice !== null
      ? prices.lastPrice - prices.previousPrice
      : null;

  // Calculate total P&L using LIVE streaming price (not stale position.currentPrice)
  const liveTotalPnL = useMemo(() => {
    if (!position) return 0;

    const currentPrice = prices.lastPrice ?? position.currentPrice ?? 50;
    const avgPrice = position.averagePrice;
    const quantity = position.quantity;

    // For YES positions: profit when price goes up
    // For NO positions: profit when price goes down (100 - price goes up)
    const unrealizedPnl =
      position.side === 'yes'
        ? ((currentPrice - avgPrice) * quantity) / 100
        : ((100 - currentPrice - (100 - avgPrice)) * quantity) / 100;

    return (position.realizedPnl ?? 0) + unrealizedPnl;
  }, [position, prices.lastPrice]);

  // Also compute live unrealized P&L for the position card
  const liveUnrealizedPnl = useMemo(() => {
    if (!position) return 0;

    const currentPrice = prices.lastPrice ?? position.currentPrice ?? 50;
    const avgPrice = position.averagePrice;
    const quantity = position.quantity;

    return position.side === 'yes'
      ? ((currentPrice - avgPrice) * quantity) / 100
      : ((100 - currentPrice - (100 - avgPrice)) * quantity) / 100;
  }, [position, prices.lastPrice]);

  // ==================== FLASH EFFECTS FOR LIVE DATA ====================
  const lastPriceFlash = useFlashOnChange(prices.lastPrice, { trackDirection: true });
  const bidFlash = useFlashOnChange(prices.yesBid, { trackDirection: true });
  const askFlash = useFlashOnChange(prices.yesAsk, { trackDirection: true });
  const spreadFlash = useFlashOnChange(spread);
  const pnlFlash = useFlashOnChange(liveTotalPnL, { trackDirection: true });
  const unrealizedPnlFlash = useFlashOnChange(liveUnrealizedPnl, { trackDirection: true });

  // ==================== LOADING STATE ====================
  if (marketLoading) {
    return (
      <div
        className={cn('bg-background flex h-full flex-col items-center justify-center', className)}
      >
        <LoadingSpinner size="lg" />
        <p className="text-muted-foreground mt-4">Loading market...</p>
      </div>
    );
  }

  // ==================== ERROR STATE ====================
  if (marketError || !market) {
    return (
      <div
        className={cn('bg-background flex h-full flex-col items-center justify-center', className)}
      >
        <p className="text-destructive">{marketError ?? 'Market not found'}</p>
        <Link
          to="/app/explore"
          className="text-primary mt-4 hover:underline"
        >
          ← Back to Explore
        </Link>
      </div>
    );
  }

  // ==================== RENDER ====================
  return (
    <div className={cn('bg-background flex h-full flex-col', className)}>
      {/* ========== BREADCRUMB ========== */}
      <div className="border-border flex items-center gap-2 border-b px-4 py-2 text-sm">
        {breadcrumbs.map((crumb, i) => {
          // Don't show ticker badge if the label is already the ticker
          const showTickerBadge = crumb.ticker && crumb.label !== crumb.ticker;
          const isLast = i === breadcrumbs.length - 1;

          return (
            <div
              key={i}
              className="flex items-center gap-2"
            >
              {i > 0 && <ChevronRight className="text-muted-foreground h-4 w-4" />}
              {crumb.to ? (
                <Link
                  to={crumb.to}
                  className="text-muted-foreground hover:text-foreground transition-colors hover:underline"
                >
                  {crumb.label}
                </Link>
              ) : crumb.ticker ? (
                <button
                  onClick={() => navigator.clipboard.writeText(crumb.ticker!)}
                  className="text-foreground max-w-[300px] cursor-pointer truncate font-medium hover:underline"
                  title={`Click to copy: ${crumb.ticker}`}
                >
                  {crumb.label}
                </button>
              ) : (
                <span className="text-foreground max-w-[300px] truncate font-medium">
                  {crumb.label}
                </span>
              )}
              {showTickerBadge && !isLast && crumb.to && (
                <Link
                  to={crumb.to}
                  className="text-muted-foreground bg-muted hover:bg-muted/80 hover:text-foreground rounded px-1.5 py-0.5 font-mono text-xs transition-colors"
                  title={crumb.ticker}
                >
                  {crumb.ticker}
                </Link>
              )}
            </div>
          );
        })}
      </div>

      {/* ========== HEADER: Market Info + Prices + P&L ========== */}
      <div className="border-border bg-card/50 flex-shrink-0 border-b">
        <div className="flex items-center gap-4 px-4 py-3">
          {/* Market Title */}
          <div className="min-w-0 flex-1">
            <div className="flex items-center gap-2">
              <h1 className="truncate text-lg font-semibold">{market.title}</h1>
              <span className="text-muted-foreground bg-muted rounded px-2 py-0.5 font-mono text-xs">
                {market.market_ticker}
              </span>
            </div>
          </div>

          {/* Live Prices - All items aligned at top */}
          <div className="flex items-start gap-5">
            {/* LAST Price with Change Indicator */}
            <div className="flex items-end gap-2">
              {priceChange !== null && (
                <span
                  className={cn(
                    'mb-0.5 flex items-center gap-1 font-mono text-sm',
                    priceChange > 0
                      ? 'text-green-500'
                      : priceChange < 0
                        ? 'text-red-400'
                        : 'text-muted-foreground'
                  )}
                >
                  {priceChange > 0 ? (
                    <TrendingUp className="h-4 w-4" />
                  ) : priceChange < 0 ? (
                    <TrendingDown className="h-4 w-4" />
                  ) : null}
                  {priceChange > 0 ? '+' : ''}
                  {priceChange}¢
                </span>
              )}
              <div>
                <div className="text-muted-foreground mb-1 text-[10px] uppercase tracking-wider">
                  Last
                </div>
                <div
                  className={cn(
                    '-mx-1.5 rounded px-1.5 font-mono text-2xl font-bold',
                    lastPriceFlash
                  )}
                >
                  {prices.lastPrice !== null ? `${prices.lastPrice}¢` : '--'}
                </div>
              </div>
            </div>

            {/* BID */}
            <div>
              <div className="text-muted-foreground mb-1 text-[10px] uppercase tracking-wider">
                Bid
              </div>
              <div
                className={cn(
                  '-mx-1.5 rounded px-1.5 font-mono text-lg font-bold text-green-500',
                  bidFlash
                )}
              >
                {prices.yesBid !== null ? `${prices.yesBid}¢` : '--'}
              </div>
            </div>

            {/* ASK */}
            <div>
              <div className="text-muted-foreground mb-1 text-[10px] uppercase tracking-wider">
                Ask
              </div>
              <div
                className={cn(
                  '-mx-1.5 rounded px-1.5 font-mono text-lg font-bold text-red-400',
                  askFlash
                )}
              >
                {prices.yesAsk !== null ? `${prices.yesAsk}¢` : '--'}
              </div>
            </div>

            {/* SPREAD */}
            {spread !== undefined && (
              <div>
                <div className="text-muted-foreground mb-1 text-[10px] uppercase tracking-wider">
                  Spread
                </div>
                <div
                  className={cn('-mx-1.5 rounded px-1.5 font-mono text-lg font-bold', spreadFlash)}
                >
                  {spread}¢
                </div>
              </div>
            )}

            {/* P&L Display */}
            <div>
              <div className="text-muted-foreground mb-1 text-[10px] uppercase tracking-wider">
                P&L
              </div>
              <div
                className={cn(
                  '-mx-1.5 rounded px-1.5 font-mono text-lg font-bold',
                  liveTotalPnL >= 0 ? 'text-green-500' : 'text-red-500',
                  pnlFlash
                )}
              >
                {liveTotalPnL >= 0 ? '+' : ''}${liveTotalPnL.toFixed(2)}
              </div>
            </div>
          </div>

          {/* Controls */}
          <div className="flex items-center gap-2">
            <Tooltip content="Keyboard shortcuts (?)">
              <button
                onClick={() => setShowShortcuts(!showShortcuts)}
                className="hover:bg-muted text-muted-foreground hover:text-foreground rounded-lg p-2 transition-colors"
              >
                <Keyboard className="h-4 w-4" />
              </button>
            </Tooltip>
            <button
              onClick={() => {
                fetchOrderbook();
                fetchPosition();
                fetchActiveOrders();
                fetchFills();
              }}
              className="hover:bg-muted text-muted-foreground hover:text-foreground rounded-lg p-2 transition-colors"
              title="Refresh"
            >
              <RefreshCw className="h-4 w-4" />
            </button>
            <div className="text-muted-foreground flex items-center gap-1 text-xs">
              {isStreaming ? (
                <>
                  <Wifi className="h-3 w-3 animate-pulse text-green-500" />
                  <span>Live</span>
                </>
              ) : (
                <>
                  <Activity className="h-3 w-3 animate-pulse text-yellow-500" />
                  <span>Polling</span>
                </>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* ========== KEYBOARD SHORTCUTS PANEL ========== */}
      {showShortcuts && (
        <div className="bg-muted/50 border-border flex items-center gap-6 border-b px-4 py-2 text-xs">
          <span className="font-medium">Shortcuts:</span>
          <span>
            <kbd className="bg-muted rounded px-1">B</kbd> Buy
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">S</kbd> Sell
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">Y</kbd> Yes
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">N</kbd> No
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">Ctrl+Enter</kbd> Submit
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">Shift+Click</kbd> Sweep
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">Esc</kbd> Cancel Sweep
          </span>
          <span>
            <kbd className="bg-muted rounded px-1">Shift+Esc</kbd> Cancel All
          </span>
        </div>
      )}

      {/* ========== MAIN CONTENT: 3-Column Layout ========== */}
      <div className="grid h-[500px] grid-cols-12 gap-4 p-4">
        {/* Column 1: Order Book (4 cols) */}
        <div className="col-span-4 flex min-h-0 flex-col">
          {orderbookLoading ? (
            <div className="bg-card border-border/50 flex flex-1 items-center justify-center rounded-lg border">
              <LoadingSpinner size="sm" />
            </div>
          ) : bids.length === 0 && asks.length === 0 ? (
            <div className="bg-card border-border/50 flex flex-1 items-center justify-center rounded-lg border">
              <div className="text-muted-foreground text-sm">No orderbook data</div>
            </div>
          ) : (
            <OrderBook
              bids={bids}
              asks={asks}
              spread={spread}
              lastPrice={prices.lastPrice ?? undefined}
              className="flex-1"
              onLevelClick={handleOrderbookLevelClick}
            />
          )}
        </div>

        {/* Column 2: Order Entry / Sweep Confirm (3 cols) */}
        <div className="col-span-3">
          {sweepState ? (
            <SweepConfirmPanel
              side={sweepState.side}
              position={sweepState.position}
              levels={sweepState.levels}
              limitPrice={sweepState.limitPrice}
              totalContracts={sweepState.totalContracts}
              submitting={sweepSubmitting}
              error={sweepError}
              onConfirm={handleSweepSubmit}
              onCancel={() => {
                setSweepState(null);
                setSweepError(null);
              }}
            />
          ) : (
            <div className="bg-card border-border/50 rounded-lg border p-4">
              <h3 className="mb-4 flex items-center gap-2 text-sm font-semibold">
                <Zap className="text-primary h-4 w-4" />
                Quick Order
              </h3>

              {/* Buy/Sell Toggle */}
              <div className="mb-3 grid grid-cols-2 gap-2">
                <button
                  onClick={() => setOrderSide('buy')}
                  className={cn(
                    'rounded-lg py-2 text-sm font-semibold transition-colors',
                    orderSide === 'buy'
                      ? 'bg-green-500 text-white'
                      : 'bg-muted text-muted-foreground hover:bg-muted/80'
                  )}
                >
                  BUY
                </button>
                <button
                  onClick={() => setOrderSide('sell')}
                  className={cn(
                    'rounded-lg py-2 text-sm font-semibold transition-colors',
                    orderSide === 'sell'
                      ? 'bg-red-500 text-white'
                      : 'bg-muted text-muted-foreground hover:bg-muted/80'
                  )}
                >
                  SELL
                </button>
              </div>

              {/* Yes/No Toggle */}
              <div className="mb-3">
                <label className="text-muted-foreground mb-1 block text-xs">Position</label>
                <div className="grid grid-cols-2 gap-2">
                  <button
                    onClick={() => setOrderPosition('yes')}
                    className={cn(
                      'rounded-lg py-1.5 text-sm transition-colors',
                      orderPosition === 'yes'
                        ? 'bg-primary/20 text-primary border-primary border'
                        : 'bg-muted text-muted-foreground hover:bg-muted/80 border border-transparent'
                    )}
                  >
                    YES
                  </button>
                  <button
                    onClick={() => setOrderPosition('no')}
                    className={cn(
                      'rounded-lg py-1.5 text-sm transition-colors',
                      orderPosition === 'no'
                        ? 'border border-purple-500 bg-purple-500/20 text-purple-400'
                        : 'bg-muted text-muted-foreground hover:bg-muted/80 border border-transparent'
                    )}
                  >
                    NO
                  </button>
                </div>
              </div>

              {/* Price Input */}
              <div className="mb-3">
                <label className="text-muted-foreground mb-1 block text-xs">Price</label>
                <div className="relative">
                  <input
                    type="number"
                    value={orderPrice}
                    onChange={(e) => setOrderPrice(Number(e.target.value))}
                    min={1}
                    max={99}
                    className="bg-muted border-border h-10 w-full rounded-lg border px-3 pr-8 font-mono text-sm"
                  />
                  <span className="text-muted-foreground absolute right-3 top-1/2 -translate-y-1/2 text-sm">
                    ¢
                  </span>
                </div>
              </div>

              {/* Quantity Input */}
              <div className="mb-4">
                <label className="text-muted-foreground mb-1 block text-xs">Quantity</label>
                <input
                  type="number"
                  value={orderQuantity}
                  onChange={(e) => setOrderQuantity(Number(e.target.value))}
                  min={1}
                  className="bg-muted border-border h-10 w-full rounded-lg border px-3 font-mono text-sm"
                />
                {/* Quick quantity buttons */}
                <div className="mt-2 flex gap-1">
                  {[10, 25, 50, 100].map((q) => (
                    <button
                      key={q}
                      onClick={() => setOrderQuantity(q)}
                      className="bg-muted hover:bg-muted/80 flex-1 rounded py-1 text-xs transition-colors"
                    >
                      {q}
                    </button>
                  ))}
                </div>
              </div>

              {/* Order Summary */}
              <div className="bg-muted/50 mb-4 space-y-1 rounded-lg p-2 text-xs">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Est. Cost</span>
                  <span className="font-mono">
                    ${((orderPrice * orderQuantity) / 100).toFixed(2)}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Max Profit</span>
                  <span className="font-mono text-green-500">
                    ${(((100 - orderPrice) * orderQuantity) / 100).toFixed(2)}
                  </span>
                </div>
              </div>

              {/* Submit Button */}
              <button
                onClick={handleSubmitOrder}
                disabled={!credentials || orderSubmitting}
                className={cn(
                  'w-full rounded-lg py-3 font-semibold text-white transition-colors disabled:opacity-50',
                  orderSide === 'buy'
                    ? 'bg-green-500 hover:bg-green-600'
                    : 'bg-red-500 hover:bg-red-600'
                )}
              >
                {orderSubmitting
                  ? 'Submitting...'
                  : `${orderSide.toUpperCase()} ${orderPosition.toUpperCase()}`}
              </button>

              {!credentials && (
                <p className="mt-2 text-center text-xs text-yellow-500">
                  Connect credentials to trade
                </p>
              )}

              {orderError && <p className="mt-2 text-center text-xs text-red-500">{orderError}</p>}
            </div>
          )}
        </div>

        {/* Column 3: Orders & Position (5 cols) */}
        <div className="col-span-5 flex min-h-0 flex-col gap-4">
          {/* Position Card (MM CRITICAL) */}
          <div className="bg-card border-border/50 rounded-lg border p-4">
            <h3 className="mb-3 text-sm font-semibold">My Position</h3>
            {positionLoading ? (
              <div className="flex items-center justify-center py-4">
                <LoadingSpinner size="sm" />
              </div>
            ) : position ? (
              <div className="grid grid-cols-4 gap-3 text-sm">
                <div>
                  <div className="text-muted-foreground text-xs">Side</div>
                  <SideBadge
                    side={position.side}
                    size="sm"
                  />
                </div>
                <div>
                  <div className="text-muted-foreground text-xs">Qty</div>
                  <div className="font-mono font-medium">{position.quantity}</div>
                </div>
                <div>
                  <div className="text-muted-foreground text-xs">Avg Price</div>
                  <div className="font-mono">${position.averagePrice.toFixed(2)}</div>
                </div>
                <div>
                  <div className="text-muted-foreground text-xs">Unrealized</div>
                  <div
                    className={cn(
                      '-mx-1 rounded px-1 font-mono font-medium',
                      liveUnrealizedPnl >= 0 ? 'text-green-500' : 'text-red-500',
                      unrealizedPnlFlash
                    )}
                  >
                    {liveUnrealizedPnl >= 0 ? '+' : ''}${liveUnrealizedPnl.toFixed(2)}
                  </div>
                </div>
              </div>
            ) : (
              <div className="text-muted-foreground py-4 text-center text-sm">
                No position in this market
              </div>
            )}
          </div>

          {/* Active Orders (MM CRITICAL) */}
          <div className="bg-card border-border/50 flex min-h-0 flex-1 flex-col rounded-lg border">
            <div className="border-border/50 flex items-center justify-between border-b px-4 py-3">
              <h3 className="text-muted-foreground text-xs font-semibold uppercase tracking-wider">
                Active Orders ({activeOrders.length})
              </h3>
              {activeOrders.length > 0 && (
                <button
                  onClick={handleCancelAll}
                  className="rounded px-2 py-1 text-xs font-medium uppercase tracking-wider text-red-400 transition-colors hover:bg-red-500/20"
                >
                  Cancel All
                </button>
              )}
            </div>
            <div className="min-h-0 flex-1 overflow-y-auto">
              {ordersLoading ? (
                <div className="flex items-center justify-center py-8">
                  <LoadingSpinner size="sm" />
                </div>
              ) : activeOrders.length === 0 ? (
                <div className="text-muted-foreground py-8 text-center text-sm">
                  No active orders
                </div>
              ) : (
                <table className="w-full">
                  <thead className="bg-muted/30 sticky top-0">
                    <tr className="text-muted-foreground text-[10px] font-medium uppercase tracking-wider">
                      <th className="px-3 py-2 text-left">Side</th>
                      <th className="px-3 py-2 text-left">Action</th>
                      <th className="px-3 py-2 text-right">Qty</th>
                      <th className="px-3 py-2 text-right">Price</th>
                      <th className="px-3 py-2 text-right">Queue</th>
                      <th className="w-8 px-3 py-2"></th>
                    </tr>
                  </thead>
                  <tbody className="divide-border/30 divide-y">
                    {activeOrders.map((order) => (
                      <tr
                        key={order.order_id}
                        className="hover:bg-muted/20 transition-colors"
                      >
                        <td className="px-3 py-2">
                          <span
                            className={cn(
                              'rounded px-1.5 py-0.5 text-[10px] font-semibold uppercase',
                              order.side === 'yes'
                                ? 'bg-emerald-500/20 text-emerald-400'
                                : 'bg-red-500/20 text-red-400'
                            )}
                          >
                            {order.side}
                          </span>
                        </td>
                        <td className="px-3 py-2">
                          <span
                            className={cn(
                              'rounded px-1.5 py-0.5 text-[10px] font-semibold uppercase',
                              order.action === 'buy'
                                ? 'bg-green-500/20 text-green-400'
                                : 'bg-orange-500/20 text-orange-400'
                            )}
                          >
                            {order.action}
                          </span>
                        </td>
                        <td className="px-3 py-2 text-right font-mono text-xs">
                          {order.remaining_count}
                        </td>
                        <td className="px-3 py-2 text-right font-mono text-xs">
                          {order.side === 'yes' ? order.yes_price : order.no_price}¢
                        </td>
                        <td className="text-muted-foreground px-3 py-2 text-right font-mono text-xs">
                          {order.queue_position?.toLocaleString() ?? '—'}
                        </td>
                        <td className="px-2 py-2">
                          <button
                            onClick={() => handleCancelOrder(order.order_id)}
                            className="text-muted-foreground rounded p-1 transition-colors hover:bg-red-500/20 hover:text-red-400"
                            title="Cancel order"
                          >
                            <X size={12} />
                          </button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* ========== FOOTER: Trade Tape + My Fills + Collapsible Chart ========== */}
      <div className="border-border border-t">
        {/* Trade Tape (public market trades) */}
        <div className="divide-border/50 bg-card/30 grid grid-cols-2 divide-x">
          {/* Trade Tape */}
          <TradeTape
            trades={marketTrades}
            loading={tradesLoading}
            onTradeClick={handleTradeClick}
          />

          {/* My Fills */}
          <div className="flex flex-col">
            <div className="border-border/50 bg-muted/30 flex items-center gap-2 border-b px-3 py-2">
              <Zap className="text-primary h-3 w-3" />
              <h3 className="text-sm font-medium">My Fills</h3>
              <span className="text-muted-foreground text-xs">({recentFills.length})</span>
            </div>
            <div className="max-h-48 flex-1 overflow-y-auto">
              {fillsLoading ? (
                <div className="text-muted-foreground flex items-center justify-center py-8 text-sm">
                  Loading...
                </div>
              ) : recentFills.length === 0 ? (
                <div className="text-muted-foreground flex items-center justify-center py-8 text-sm">
                  No fills yet
                </div>
              ) : (
                <div className="divide-border/20 divide-y">
                  {recentFills.slice(0, 20).map((fill) => (
                    <div
                      key={fill.id}
                      className="hover:bg-accent/40 flex cursor-pointer items-center justify-between px-3 py-1.5 font-mono text-xs"
                      onClick={() => setOrderPrice(fill.price)}
                    >
                      <span className="text-muted-foreground w-20">
                        {fill.timestamp.toLocaleTimeString('en-US', {
                          hour: '2-digit',
                          minute: '2-digit',
                          second: '2-digit',
                          hour12: false,
                        })}
                      </span>
                      <Badge
                        variant={fill.action === 'buy' ? 'success' : 'danger'}
                        size="xs"
                        className="w-10 justify-center"
                      >
                        {fill.action.toUpperCase()}
                      </Badge>
                      <span
                        className={cn(
                          'w-12 text-right font-semibold',
                          fill.action === 'buy' ? 'text-green-500' : 'text-red-500'
                        )}
                      >
                        {fill.price}¢
                      </span>
                      <span className="text-muted-foreground w-16 text-right">
                        {fill.quantity.toLocaleString()}
                      </span>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Price Chart */}
        <div className="border-border/50 border-t">
          <div className="text-muted-foreground flex items-center gap-2 px-4 py-2 text-sm">
            <BarChart3 className="h-4 w-4" />
            <span>Price Chart (24h)</span>
          </div>
          <div className="p-4 pt-0">
            {chartLoading ? (
              <div className="flex h-48 items-center justify-center">
                <LoadingSpinner size="sm" />
              </div>
            ) : (
              <CandlestickChart
                data={chartData}
                title=""
                height={200}
                showVolume={false}
              />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

// ==================== HELPER FUNCTIONS ====================

function parseOrderbookSide(
  data: Array<[number, number]> | undefined
): Array<{ price: number; quantity: number }> {
  if (!data || !Array.isArray(data)) return [];
  return data.map(([price, quantity]) => ({ price, quantity }));
}

function addCumulatives(levels: Array<{ price: number; quantity: number }>): OrderbookLevel[] {
  let cumulative = 0;
  return levels.map((level) => {
    cumulative += level.quantity;
    return { ...level, cumulative };
  });
}

function convertToChartCandle(candle: KalshiCandlestick): MockCandle {
  return {
    timestamp: candle.end_period_ts * 1000,
    open: candle.price.open!,
    high: candle.price.high!,
    low: candle.price.low!,
    close: candle.price.close!,
    volume: candle.volume,
  };
}

function generatePlaceholderChart(basePrice: number): MockCandle[] {
  const candles: MockCandle[] = [];
  const now = Date.now();
  const intervalMs = 5 * 60 * 1000;
  const count = 50;
  let price = basePrice;

  for (let i = 0; i < count; i++) {
    const timestamp = now - (count - i) * intervalMs;
    const volatility = 0.02;
    const change = (Math.random() - 0.5) * volatility * price;
    const open = price;
    const close = Math.max(1, Math.min(99, price + change));
    const high = Math.max(open, close) + Math.random() * 2;
    const low = Math.min(open, close) - Math.random() * 2;

    candles.push({
      timestamp,
      open,
      high: Math.min(99, high),
      low: Math.max(1, low),
      close,
      volume: Math.floor(Math.random() * 10000) + 1000,
    });

    price = close;
  }

  return candles;
}

function updateChartWithTick(candles: MockCandle[], newPrice: number): MockCandle[] {
  if (candles.length === 0) return candles;

  const lastCandle = candles[candles.length - 1];
  const now = Date.now();
  const intervalMs = 60 * 1000;
  const candleEndTime = lastCandle.timestamp + intervalMs;

  if (now < candleEndTime) {
    const updatedCandle: MockCandle = {
      ...lastCandle,
      close: newPrice,
      high: Math.max(lastCandle.high, newPrice),
      low: Math.min(lastCandle.low, newPrice),
      volume: lastCandle.volume + Math.floor(Math.random() * 100),
    };
    return [...candles.slice(0, -1), updatedCandle];
  }

  const newCandle: MockCandle = {
    timestamp: candleEndTime,
    open: lastCandle.close,
    high: Math.max(lastCandle.close, newPrice),
    low: Math.min(lastCandle.close, newPrice),
    close: newPrice,
    volume: Math.floor(Math.random() * 1000) + 100,
  };

  const newCandles = [...candles, newCandle];
  if (newCandles.length > 200) {
    return newCandles.slice(-200);
  }
  return newCandles;
}
