/**
 * NBAView - NBA markets display
 *
 * Shows NBA game markets with live pricing.
 */

import { useEffect, useRef } from 'react';
import type { NBAMarketRow, CandlestickPeriod, Candlestick } from '@/types';
import { createPortfolioChart, type ChartManager } from '@/lib/chartManager';
import { getMarketState } from '@/lib/stream';
import { CandlestickAggregator } from '@/lib/candlestickAggregator';
import type { Portfolio } from '@/types';
import MarketsTable from '@/components/MarketsTable';

interface NBAViewProps {
  markets: NBAMarketRow[];
  chartPeriod: CandlestickPeriod;
  onMarketClick: (marketTicker: string, price?: number) => void;
  onChartPeriodChange: (period: CandlestickPeriod) => void;
}

export function NBAView({
  markets,
  chartPeriod,
  onMarketClick,
  onChartPeriodChange,
}: NBAViewProps) {
  const portfolioChartContainerRef = useRef<HTMLDivElement>(null);
  const portfolioChartRef = useRef<ChartManager | null>(null);
  const chartHasDataRef = useRef(false);

  // Initialize portfolio chart
  useEffect(() => {
    if (portfolioChartContainerRef.current && !portfolioChartRef.current) {
      const chart = createPortfolioChart(portfolioChartContainerRef.current, true);
      chart.init();
      portfolioChartRef.current = chart;
    }

    return () => {
      if (portfolioChartRef.current) {
        portfolioChartRef.current.destroy();
        portfolioChartRef.current = null;
      }
    };
  }, []);

  // Update the chart from live market ticks aggregated into a simple "NBA basket".
  // Note: this is not a true account-value chart yet; it's a useful live signal until
  // we wire portfolio accounting + fills.
  useEffect(() => {
    const chart = portfolioChartRef.current;
    if (!chart) return;

    const ms = getMarketState();
    if (!ms) return;

    const tickers = (markets ?? []).map((m) => m.market_ticker).filter(Boolean);
    if (tickers.length === 0) return;

    const maxTickers = 20; // keep it light
    const selectedTickers = tickers.slice(0, maxTickers);

    const aggregator = new CandlestickAggregator();
    const portfolio: Portfolio = {
      id: 'nba-basket',
      name: 'NBA Basket',
      markets: selectedTickers.map((t) => ({ ticker: t, weight: 1 / selectedTickers.length })),
    };

    const updateOnce = () => {
      const marketCandles = new Map<string, Candlestick[]>();
      for (const t of selectedTickers) {
        marketCandles.set(t, ms.getCandles(t, chartPeriod, 200));
      }
      const portfolioCandles = aggregator.aggregatePortfolio(portfolio, chartPeriod, marketCandles);
      if (portfolioCandles.length > 0) {
        chart.updateData(portfolioCandles);
        chartHasDataRef.current = true;
      }
    };

    updateOnce();
    const interval = window.setInterval(updateOnce, 3000);
    return () => window.clearInterval(interval);
  }, [markets, chartPeriod]);

  const handleMarketClick = (market: NBAMarketRow) => {
    const midPrice = market.awayProb || market.homeProb || undefined;
    onMarketClick(market.market_ticker, midPrice);
  };

  return (
    <div className="space-y-6 p-4">
      {/* Portfolio Chart */}
      <div className="bg-card border-border rounded-lg border p-4">
        <div className="mb-4 flex items-center justify-between">
          <h3 className="text-sm font-medium">Portfolio Price History</h3>
          <select
            className="bg-background border-border rounded border px-2 py-1 text-xs"
            value={chartPeriod}
            onChange={(e) => onChartPeriodChange(e.target.value as CandlestickPeriod)}
          >
            <option value="1m">1m</option>
            <option value="5m">5m</option>
            <option value="15m">15m</option>
            <option value="1h">1h</option>
            <option value="1d">1d</option>
          </select>
        </div>
        <div className="relative">
          <div
            ref={portfolioChartContainerRef}
            className="h-48"
          />
          {/* Best-effort guidance when no live ticks have arrived yet */}
          {!chartHasDataRef.current && (
            <div className="text-muted-foreground pointer-events-none absolute inset-0 flex items-center justify-center text-xs">
              Waiting for live data…
            </div>
          )}
        </div>
      </div>

      {/* Markets Table */}
      <div className="bg-card border-border rounded-lg border">
        <div className="border-border border-b p-4">
          <h3 className="text-sm font-medium">NBA Markets</h3>
          <p className="text-muted-foreground mt-1 text-xs">{markets.length} markets available</p>
        </div>
        <MarketsTable
          markets={markets}
          onMarketClick={handleMarketClick}
        />
      </div>
    </div>
  );
}
