/**
 * OrdersView - Order management
 *
 * Shows open orders with actions to cancel orders and place new ones.
 */

import { useState } from 'react';
import { Plus, X, Ban, Clock, ChevronDown, ChevronUp } from 'lucide-react';
import type { KalshiOrder } from '@/lib/kalshiApi';
import { cn } from '@/lib/utils';

interface OrdersViewProps {
  orders: KalshiOrder[];
  onNewOrder: () => void;
  onKillAll: () => void;
  onCancelOrder: (orderId: string) => void;
}

type SortKey =
  | 'created_time'
  | 'ticker'
  | 'side'
  | 'action'
  | 'yes_price'
  | 'remaining_count'
  | 'status';

export function OrdersView({ orders, onNewOrder, onKillAll, onCancelOrder }: OrdersViewProps) {
  const [sortKey, setSortKey] = useState<SortKey>('created_time');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('desc');
  const [filter, setFilter] = useState<'all' | 'buy' | 'sell'>('all');
  const [cancellingOrder, setCancellingOrder] = useState<string | null>(null);

  const handleSort = (key: SortKey) => {
    if (sortKey === key) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
      return;
    }
    setSortKey(key);
    setSortDirection('desc');
  };

  const handleCancelOrder = async (orderId: string) => {
    setCancellingOrder(orderId);
    try {
      await onCancelOrder(orderId);
    } finally {
      setCancellingOrder(null);
    }
  };

  // Filter orders
  const filteredOrders = orders.filter((o) => {
    if (filter === 'all') return true;
    return o.action === filter;
  });

  // Sort orders
  const sortedOrders = [...filteredOrders].sort((a, b) => {
    const aVal = a[sortKey];
    const bVal = b[sortKey];
    const modifier = sortDirection === 'asc' ? 1 : -1;

    if (typeof aVal === 'string' && typeof bVal === 'string') {
      return aVal.localeCompare(bVal) * modifier;
    }
    if (aVal === undefined && bVal === undefined) return 0;
    if (aVal === undefined) return 1;
    if (bVal === undefined) return -1;
    return ((aVal as number) - (bVal as number)) * modifier;
  });

  const buyOrders = orders.filter((o) => o.action === 'buy');
  const sellOrders = orders.filter((o) => o.action === 'sell');
  const getOrderPrice = (o: KalshiOrder) => (o.side === 'yes' ? o.yes_price : o.no_price) ?? 0;
  const totalBuyValue =
    buyOrders.reduce((sum, o) => sum + getOrderPrice(o) * o.remaining_count, 0) / 100;
  const totalSellValue =
    sellOrders.reduce((sum, o) => sum + getOrderPrice(o) * o.remaining_count, 0) / 100;

  const SortHeader = ({ label, sortKeyProp }: { label: string; sortKeyProp: SortKey }) => (
    <button
      onClick={() => handleSort(sortKeyProp)}
      className="hover:text-foreground flex items-center gap-1 transition-colors"
    >
      {label}
      {sortKey === sortKeyProp &&
        (sortDirection === 'asc' ? <ChevronUp size={12} /> : <ChevronDown size={12} />)}
    </button>
  );

  const formatTime = (timestamp: string) => {
    const date = new Date(timestamp);
    return date.toLocaleTimeString('en-US', {
      hour: '2-digit',
      minute: '2-digit',
      hour12: false,
    });
  };

  return (
    <div className="space-y-6 p-4">
      {/* Summary Cards */}
      <div className="grid grid-cols-2 gap-4 md:grid-cols-4">
        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Open Orders</div>
          <div className="mt-1 text-2xl font-bold">{orders.length}</div>
          <div className="text-muted-foreground mt-1 text-xs">
            <span className="text-green-500">{buyOrders.length} buy</span>
            {' / '}
            <span className="text-red-500">{sellOrders.length} sell</span>
          </div>
        </div>

        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Buy Value</div>
          <div className="mt-1 text-2xl font-bold text-green-500">${totalBuyValue.toFixed(2)}</div>
        </div>

        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Sell Value</div>
          <div className="mt-1 text-2xl font-bold text-red-500">${totalSellValue.toFixed(2)}</div>
        </div>

        <div className="bg-card border-border flex items-center justify-center gap-2 rounded-lg border p-4">
          <button
            onClick={onNewOrder}
            className="bg-primary text-primary-foreground hover:bg-primary/90 flex items-center gap-2 rounded-lg px-3 py-2 text-sm transition-colors"
          >
            <Plus size={14} />
            New Order
          </button>
          <button
            onClick={onKillAll}
            disabled={orders.length === 0}
            className="bg-destructive text-destructive-foreground hover:bg-destructive/90 flex items-center gap-2 rounded-lg px-3 py-2 text-sm transition-colors disabled:cursor-not-allowed disabled:opacity-50"
          >
            <Ban size={14} />
            Cancel All
          </button>
        </div>
      </div>

      {/* Orders Table */}
      <div className="bg-card border-border rounded-lg border">
        {/* Header */}
        <div className="border-border flex items-center justify-between border-b p-4">
          <h3 className="font-semibold">Open Orders</h3>

          {/* Filter tabs */}
          <div className="flex items-center gap-2">
            {(['all', 'buy', 'sell'] as const).map((f) => (
              <button
                key={f}
                onClick={() => setFilter(f)}
                className={cn(
                  'rounded-full px-3 py-1 text-xs transition-colors',
                  filter === f
                    ? 'bg-primary text-primary-foreground'
                    : 'text-muted-foreground hover:text-foreground hover:bg-muted'
                )}
              >
                {f === 'all' ? 'All' : f === 'buy' ? 'Buy Orders' : 'Sell Orders'}
              </button>
            ))}
          </div>
        </div>

        {/* Table */}
        {orders.length === 0 ? (
          <div className="text-muted-foreground p-8 text-center">
            <p className="text-sm">No open orders</p>
            <button
              onClick={onNewOrder}
              className="bg-primary text-primary-foreground hover:bg-primary/90 mt-4 rounded-lg px-4 py-2 text-sm transition-colors"
            >
              Place Your First Order
            </button>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="text-muted-foreground border-border border-b text-xs">
                  <th className="p-3 text-left font-medium">
                    <SortHeader
                      label="Time"
                      sortKeyProp="created_time"
                    />
                  </th>
                  <th className="p-3 text-left font-medium">
                    <SortHeader
                      label="Market"
                      sortKeyProp="ticker"
                    />
                  </th>
                  <th className="p-3 text-center font-medium">
                    <SortHeader
                      label="Action"
                      sortKeyProp="action"
                    />
                  </th>
                  <th className="p-3 text-center font-medium">
                    <SortHeader
                      label="Side"
                      sortKeyProp="side"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">
                    <SortHeader
                      label="Price"
                      sortKeyProp="yes_price"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">
                    <SortHeader
                      label="Qty"
                      sortKeyProp="remaining_count"
                    />
                  </th>
                  <th className="hidden p-3 text-center font-medium md:table-cell">Queue</th>
                  <th className="p-3 text-center font-medium">
                    <SortHeader
                      label="Status"
                      sortKeyProp="status"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">Actions</th>
                </tr>
              </thead>
              <tbody>
                {sortedOrders.map((order) => {
                  const isCancelling = cancellingOrder === order.order_id;

                  return (
                    <tr
                      key={order.order_id}
                      className="border-border hover:bg-muted/30 border-b transition-colors"
                    >
                      <td className="p-3">
                        <div className="text-muted-foreground flex items-center gap-1.5 text-sm">
                          <Clock size={12} />
                          {formatTime(order.created_time)}
                        </div>
                      </td>
                      <td className="p-3">
                        <div className="font-mono text-sm">{order.ticker}</div>
                      </td>
                      <td className="p-3 text-center">
                        <span
                          className={cn(
                            'rounded px-2 py-0.5 text-xs font-semibold',
                            order.action === 'buy'
                              ? 'bg-green-500/20 text-green-400'
                              : 'bg-red-500/20 text-red-400'
                          )}
                        >
                          {order.action.toUpperCase()}
                        </span>
                      </td>
                      <td className="p-3 text-center">
                        <span
                          className={cn(
                            'rounded px-2 py-0.5 text-xs font-semibold',
                            order.side === 'yes'
                              ? 'bg-blue-500/20 text-blue-400'
                              : 'bg-orange-500/20 text-orange-400'
                          )}
                        >
                          {order.side.toUpperCase()}
                        </span>
                      </td>
                      <td className="p-3 text-right text-sm font-medium">
                        {getOrderPrice(order)}¢
                      </td>
                      <td className="p-3 text-right text-sm font-medium">
                        {order.remaining_count}
                      </td>
                      <td className="text-muted-foreground hidden p-3 text-center text-sm md:table-cell">
                        {order.queue_position ?? '—'}
                      </td>
                      <td className="p-3 text-center">
                        <span
                          className={cn(
                            'rounded px-2 py-0.5 text-xs font-medium',
                            order.status === 'resting' && 'bg-yellow-500/20 text-yellow-400',
                            order.status === 'pending' && 'bg-blue-500/20 text-blue-400',
                            order.status === 'filled' && 'bg-green-500/20 text-green-400',
                            order.status === 'cancelled' && 'bg-gray-500/20 text-gray-400'
                          )}
                        >
                          {order.status.charAt(0).toUpperCase() + order.status.slice(1)}
                        </span>
                      </td>
                      <td className="p-3 text-right">
                        <button
                          onClick={() => handleCancelOrder(order.order_id)}
                          disabled={isCancelling}
                          className={cn(
                            'text-muted-foreground hover:text-destructive hover:bg-destructive/10 rounded p-1.5 transition-colors',
                            isCancelling && 'cursor-not-allowed opacity-50'
                          )}
                          title="Cancel order"
                        >
                          {isCancelling ? (
                            <div className="h-3.5 w-3.5 animate-spin rounded-full border-2 border-current border-t-transparent" />
                          ) : (
                            <X size={14} />
                          )}
                        </button>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  );
}
