/**
 * PositionsView - Portfolio positions display
 *
 * Shows current positions with P&L and ability to close positions.
 */

import { useState } from 'react';
import { X, Plus, TrendingUp, TrendingDown, ChevronDown, ChevronUp } from 'lucide-react';
import type { Position, NBAMarketRow } from '@/types';
import { cn } from '@/lib/utils';

interface PositionsViewProps {
  positions: Position[];
  totalPnl: number;
  markets: NBAMarketRow[];
  onClosePosition?: (ticker: string, side: 'yes' | 'no', quantity: number) => Promise<void>;
  onNewOrder?: (marketTicker?: string, price?: number) => void;
}

type SortKey =
  | 'ticker'
  | 'side'
  | 'quantity'
  | 'averagePrice'
  | 'currentPrice'
  | 'totalPnl'
  | 'pnlPercent';

export function PositionsView({
  positions,
  totalPnl,
  markets,
  onClosePosition,
  onNewOrder,
}: PositionsViewProps) {
  const [sortKey, setSortKey] = useState<SortKey>('totalPnl');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('desc');
  const [filter, setFilter] = useState<'all' | 'profit' | 'loss'>('all');
  const [closingPosition, setClosingPosition] = useState<string | null>(null);

  // Create market name map for positions display
  const marketsDataMap = new Map<string, { awayTeam: string; homeTeam: string }>();
  markets.forEach((market) => {
    marketsDataMap.set(market.market_ticker, {
      awayTeam: market.awayTeam,
      homeTeam: market.homeTeam,
    });
  });

  const handleSort = (key: SortKey) => {
    if (sortKey === key) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
      return;
    }
    setSortKey(key);
    setSortDirection('desc');
  };

  const handleClosePosition = async (position: Position) => {
    if (!onClosePosition) return;

    const positionKey = `${position.ticker}-${position.side}`;
    setClosingPosition(positionKey);

    try {
      await onClosePosition(position.ticker, position.side, position.quantity);
    } finally {
      setClosingPosition(null);
    }
  };

  // Filter positions
  const filteredPositions = positions.filter((p) => {
    if (filter === 'all') return true;
    if (filter === 'profit') return p.totalPnl >= 0;
    return p.totalPnl < 0;
  });

  // Sort positions
  const sortedPositions = [...filteredPositions].sort((a, b) => {
    const aVal = a[sortKey];
    const bVal = b[sortKey];
    const modifier = sortDirection === 'asc' ? 1 : -1;

    if (typeof aVal === 'string' && typeof bVal === 'string') {
      return aVal.localeCompare(bVal) * modifier;
    }
    if (aVal === null && bVal === null) return 0;
    if (aVal === null) return 1;
    if (bVal === null) return -1;
    return ((aVal as number) - (bVal as number)) * modifier;
  });

  const totalValue = positions.reduce((sum, p) => {
    const currentPrice = p.currentPrice ?? p.averagePrice;
    return sum + currentPrice * p.quantity;
  }, 0);

  const inProfit = positions.filter((p) => p.totalPnl >= 0).length;
  const inLoss = positions.filter((p) => p.totalPnl < 0).length;

  const SortHeader = ({ label, sortKeyProp }: { label: string; sortKeyProp: SortKey }) => (
    <button
      onClick={() => handleSort(sortKeyProp)}
      className="hover:text-foreground flex items-center gap-1 transition-colors"
    >
      {label}
      {sortKey === sortKeyProp &&
        (sortDirection === 'asc' ? <ChevronUp size={12} /> : <ChevronDown size={12} />)}
    </button>
  );

  return (
    <div className="space-y-6 p-4">
      {/* Summary Cards */}
      <div className="grid grid-cols-2 gap-4 md:grid-cols-4">
        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Total P&L</div>
          <div
            className={cn(
              'mt-1 text-2xl font-bold',
              totalPnl >= 0 ? 'text-green-500' : 'text-red-500'
            )}
          >
            {totalPnl >= 0 ? '+' : ''}${totalPnl.toFixed(2)}
          </div>
        </div>

        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Market Value</div>
          <div className="mt-1 text-2xl font-bold">${totalValue.toFixed(2)}</div>
        </div>

        <div className="bg-card border-border rounded-lg border p-4">
          <div className="text-muted-foreground text-xs uppercase tracking-wide">Positions</div>
          <div className="mt-1 text-2xl font-bold">{positions.length}</div>
          <div className="text-muted-foreground mt-1 text-xs">
            <span className="text-green-500">{inProfit} profit</span>
            {' / '}
            <span className="text-red-500">{inLoss} loss</span>
          </div>
        </div>

        <div className="bg-card border-border flex items-center justify-center rounded-lg border p-4">
          {onNewOrder && (
            <button
              onClick={() => onNewOrder()}
              className="bg-primary text-primary-foreground hover:bg-primary/90 flex items-center gap-2 rounded-lg px-4 py-2 transition-colors"
            >
              <Plus size={16} />
              New Order
            </button>
          )}
        </div>
      </div>

      {/* Positions Table */}
      <div className="bg-card border-border rounded-lg border">
        {/* Header */}
        <div className="border-border flex items-center justify-between border-b p-4">
          <h3 className="font-semibold">Open Positions</h3>

          {/* Filter tabs */}
          <div className="flex items-center gap-2">
            {(['all', 'profit', 'loss'] as const).map((f) => (
              <button
                key={f}
                onClick={() => setFilter(f)}
                className={cn(
                  'rounded-full px-3 py-1 text-xs transition-colors',
                  filter === f
                    ? 'bg-primary text-primary-foreground'
                    : 'text-muted-foreground hover:text-foreground hover:bg-muted'
                )}
              >
                {f === 'all' ? 'All' : f === 'profit' ? 'In Profit' : 'In Loss'}
              </button>
            ))}
          </div>
        </div>

        {/* Table */}
        {positions.length === 0 ? (
          <div className="text-muted-foreground p-8 text-center">
            <p className="text-sm">No open positions</p>
            {onNewOrder && (
              <button
                onClick={() => onNewOrder()}
                className="bg-primary text-primary-foreground hover:bg-primary/90 mt-4 rounded-lg px-4 py-2 text-sm transition-colors"
              >
                Place Your First Order
              </button>
            )}
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="text-muted-foreground border-border border-b text-xs">
                  <th className="p-3 text-left font-medium">
                    <SortHeader
                      label="Market"
                      sortKeyProp="ticker"
                    />
                  </th>
                  <th className="p-3 text-center font-medium">
                    <SortHeader
                      label="Side"
                      sortKeyProp="side"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">
                    <SortHeader
                      label="Qty"
                      sortKeyProp="quantity"
                    />
                  </th>
                  <th className="hidden p-3 text-right font-medium md:table-cell">
                    <SortHeader
                      label="Avg Price"
                      sortKeyProp="averagePrice"
                    />
                  </th>
                  <th className="hidden p-3 text-right font-medium md:table-cell">
                    <SortHeader
                      label="Current"
                      sortKeyProp="currentPrice"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">
                    <SortHeader
                      label="P&L"
                      sortKeyProp="totalPnl"
                    />
                  </th>
                  <th className="p-3 text-right font-medium">Actions</th>
                </tr>
              </thead>
              <tbody>
                {sortedPositions.map((position, index) => {
                  const market = marketsDataMap.get(position.ticker);
                  const marketName = market
                    ? `${market.awayTeam} @ ${market.homeTeam}`
                    : position.ticker;
                  const positionKey = `${position.ticker}-${position.side}`;
                  const isClosing = closingPosition === positionKey;
                  const priceUp =
                    position.currentPrice !== null && position.currentPrice > position.averagePrice;
                  const priceDown =
                    position.currentPrice !== null && position.currentPrice < position.averagePrice;

                  return (
                    <tr
                      key={`${position.ticker}-${position.side}-${index}`}
                      className="border-border hover:bg-muted/30 border-b transition-colors"
                    >
                      <td className="p-3">
                        <div className="text-sm font-medium">{marketName}</div>
                        <div className="text-muted-foreground font-mono text-xs">
                          {position.ticker}
                        </div>
                      </td>
                      <td className="p-3 text-center">
                        <span
                          className={cn(
                            'rounded px-2 py-0.5 text-xs font-semibold',
                            position.side === 'yes'
                              ? 'bg-green-500/20 text-green-400'
                              : 'bg-red-500/20 text-red-400'
                          )}
                        >
                          {position.side.toUpperCase()}
                        </span>
                      </td>
                      <td className="p-3 text-right text-sm font-medium">{position.quantity}</td>
                      <td className="hidden p-3 text-right text-sm md:table-cell">
                        ${position.averagePrice.toFixed(2)}
                      </td>
                      <td className="hidden p-3 text-right md:table-cell">
                        <div className="flex items-center justify-end gap-1">
                          <span className="text-sm">
                            {position.currentPrice !== null
                              ? `$${position.currentPrice.toFixed(2)}`
                              : '—'}
                          </span>
                          {priceUp && (
                            <TrendingUp
                              size={12}
                              className="text-green-500"
                            />
                          )}
                          {priceDown && (
                            <TrendingDown
                              size={12}
                              className="text-red-500"
                            />
                          )}
                        </div>
                      </td>
                      <td className="p-3 text-right">
                        <div className="flex flex-col items-end">
                          <span
                            className={cn(
                              'font-semibold',
                              position.totalPnl >= 0 ? 'text-green-500' : 'text-red-500'
                            )}
                          >
                            {position.totalPnl >= 0 ? '+' : ''}${position.totalPnl.toFixed(2)}
                          </span>
                          <span
                            className={cn(
                              'text-xs',
                              position.pnlPercent >= 0 ? 'text-green-500' : 'text-red-500'
                            )}
                          >
                            ({position.pnlPercent >= 0 ? '+' : ''}
                            {position.pnlPercent.toFixed(1)}%)
                          </span>
                        </div>
                      </td>
                      <td className="p-3 text-right">
                        <div className="flex items-center justify-end gap-2">
                          {onNewOrder && (
                            <button
                              onClick={() =>
                                onNewOrder(position.ticker, position.currentPrice ?? undefined)
                              }
                              className="text-muted-foreground hover:text-primary hover:bg-primary/10 rounded p-1.5 transition-colors"
                              title="Add to position"
                            >
                              <Plus size={14} />
                            </button>
                          )}
                          {onClosePosition && (
                            <button
                              onClick={() => handleClosePosition(position)}
                              disabled={isClosing}
                              className={cn(
                                'text-muted-foreground hover:text-destructive hover:bg-destructive/10 rounded p-1.5 transition-colors',
                                isClosing && 'cursor-not-allowed opacity-50'
                              )}
                              title="Close position (market sell)"
                            >
                              {isClosing ? (
                                <div className="h-3.5 w-3.5 animate-spin rounded-full border-2 border-current border-t-transparent" />
                              ) : (
                                <X size={14} />
                              )}
                            </button>
                          )}
                        </div>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  );
}
