/**
 * SettingsView - Application settings
 *
 * Displays connection info and preferences.
 */

import { useEffect, useState } from 'react';
import type { PricingPOV, OddsFormat } from '@/types';
import type { KalshiCredentials } from '@/types';
import type { DataMode } from '@/lib/dataMode';
import { Dropdown } from '@/components/atoms/Dropdown';
import { SecretCredentialField } from '@/components/atoms/SecretCredentialField';
import type { KalshiProfile } from '@/lib/kalshiProfiles';
import type { PolyUserConfig } from '@/lib/polymarket/client';
import { getOddsApiKey, setOddsApiKey } from '@/lib/oddsApi';

interface SettingsViewProps {
  pov: PricingPOV;
  oddsFormat: OddsFormat;
  dataMode?: DataMode;
  onPOVToggle: () => void;
  onOddsFormatToggle: () => void;
  onDataModeToggle?: () => void;
  connectionStatus: string;
  environment: 'prod' | 'demo';
  kalshiProfiles: KalshiProfile[];
  activeKalshiProfileId: string | null;
  onKalshiSetActive?: (profileId: string) => void;
  onKalshiDeleteProfile?: (profileId: string) => void;
  onKalshiSaveProfile?: (args: {
    id?: string;
    name?: string;
    accessKey: string;
    privateKey: string;
  }) => void;
  kalshiCredentials?: KalshiCredentials | null;
  onKalshiReconnect?: (args: {
    id?: string;
    name?: string;
    accessKey: string;
    privateKey: string;
    remember: boolean;
  }) => Promise<void>;
  polyConfig?: PolyUserConfig | null;
  polyStatus?: 'disconnected' | 'connected' | 'error';
  onPolyConnect?: (args: {
    privateKey: string;
    signatureType: 0 | 1 | 2;
    funder?: string;
  }) => Promise<void>;
  onPolyDisconnect?: () => void;
}

export function SettingsView({
  pov,
  oddsFormat,
  dataMode,
  onPOVToggle,
  onOddsFormatToggle,
  onDataModeToggle,
  connectionStatus,
  environment,
  kalshiProfiles,
  activeKalshiProfileId,
  onKalshiSetActive,
  onKalshiDeleteProfile,
  onKalshiSaveProfile,
  kalshiCredentials,
  onKalshiReconnect,
  polyConfig,
  polyStatus = 'disconnected',
  onPolyConnect,
  onPolyDisconnect,
}: SettingsViewProps) {
  const [selectedKalshiProfileId, setSelectedKalshiProfileId] = useState<string>('');
  const [kalshiProfileName, setKalshiProfileName] = useState('');
  const [kalshiAccessKey, setKalshiAccessKey] = useState('');
  const [kalshiPrivateKey, setKalshiPrivateKey] = useState('');
  const [kalshiSaving, setKalshiSaving] = useState(false);
  const [kalshiError, setKalshiError] = useState<string | null>(null);

  const [polyPrivateKey, setPolyPrivateKey] = useState('');
  const [polyFunder, setPolyFunder] = useState('');
  const [polySignatureType, setPolySignatureType] = useState<0 | 1 | 2>(0);
  const [polySaving, setPolySaving] = useState(false);
  const [polyError, setPolyError] = useState<string | null>(null);

  const [oddsApiKeyValue, setOddsApiKeyValue] = useState(() => getOddsApiKey());
  const [oddsApiSaved, setOddsApiSaved] = useState(false);

  useEffect(() => {
    const active =
      kalshiProfiles.find((p) => p.id === activeKalshiProfileId) ||
      kalshiProfiles.find((p) => p.accessKey === kalshiCredentials?.accessKeyId) ||
      null;
    if (active) {
      setSelectedKalshiProfileId(active.id);
      setKalshiProfileName(active.name || active.accessKey);
      setKalshiAccessKey(active.accessKey);
      setKalshiPrivateKey(active.privateKey);
      return;
    }

    setSelectedKalshiProfileId('');
    const accessKey = kalshiCredentials?.accessKeyId ?? '';
    setKalshiProfileName(accessKey);
    setKalshiAccessKey(accessKey);
    setKalshiPrivateKey(kalshiCredentials?.privateKeyPem ?? '');
  }, [
    kalshiProfiles,
    activeKalshiProfileId,
    kalshiCredentials?.accessKeyId,
    kalshiCredentials?.privateKeyPem,
  ]);

  useEffect(() => {
    setPolyPrivateKey(polyConfig?.privateKey ?? '');
    setPolyFunder(polyConfig?.funder ?? '');
    setPolySignatureType((polyConfig?.signatureType ?? 0) as 0 | 1 | 2);
  }, [polyConfig?.privateKey, polyConfig?.funder, polyConfig?.signatureType]);

  return (
    <div className="max-w-2xl space-y-6 p-4">
      {/* Connection Info */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">Connection</h3>
        <div className="space-y-3">
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground text-sm">Status</span>
            <span className="text-sm font-medium">{connectionStatus}</span>
          </div>
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground text-sm">Environment</span>
            <span className="text-sm font-medium uppercase">{environment}</span>
          </div>
        </div>
      </div>

      {/* Kalshi Credentials */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">Kalshi Credentials</h3>
        <div className="space-y-3">
          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">Saved Profiles</label>
            <Dropdown
              options={kalshiProfiles.map((p) => ({
                value: p.id,
                label: p.id === activeKalshiProfileId ? `${p.name} (active)` : p.name,
                description: p.accessKey,
              }))}
              value={selectedKalshiProfileId}
              onChange={(value) => {
                const profile = kalshiProfiles.find((p) => p.id === value);
                if (!profile) return;
                setSelectedKalshiProfileId(profile.id);
                setKalshiProfileName(profile.name || profile.accessKey);
                setKalshiAccessKey(profile.accessKey);
                setKalshiPrivateKey(profile.privateKey);
                setKalshiError(null);
                onKalshiSetActive?.(profile.id);
              }}
              placeholder="No saved profiles"
              disabled={kalshiSaving || kalshiProfiles.length === 0}
            />
            <div className="flex items-center gap-2">
              <button
                onClick={() => {
                  setSelectedKalshiProfileId('');
                  setKalshiProfileName('');
                  setKalshiAccessKey('');
                  setKalshiPrivateKey('');
                  setKalshiError(null);
                }}
                disabled={kalshiSaving}
                className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
              >
                New Profile
              </button>
              {selectedKalshiProfileId && (
                <button
                  onClick={() => {
                    const accessKey = kalshiAccessKey.trim();
                    setSelectedKalshiProfileId('');
                    setKalshiProfileName(
                      `${(kalshiProfileName || accessKey || 'Profile').trim()} Copy`
                    );
                    setKalshiError(null);
                  }}
                  disabled={kalshiSaving}
                  className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
                >
                  Duplicate
                </button>
              )}
            </div>
          </div>

          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">Profile Name</label>
            <input
              value={kalshiProfileName}
              onChange={(e) => setKalshiProfileName(e.target.value)}
              className="border-border bg-background h-10 w-full rounded-lg border px-3 text-sm"
              placeholder="Defaults to Access Key ID if blank"
              disabled={kalshiSaving}
            />
          </div>

          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">Access Key ID</label>
            <SecretCredentialField
              value={kalshiAccessKey}
              onChange={setKalshiAccessKey}
              placeholder="Your Kalshi access key id"
              disabled={kalshiSaving}
            />
          </div>

          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">
              Private Key (RSA PEM)
            </label>
            <SecretCredentialField
              value={kalshiPrivateKey}
              onChange={setKalshiPrivateKey}
              placeholder="-----BEGIN PRIVATE KEY-----"
              disabled={kalshiSaving}
              multiline
            />
          </div>

          {kalshiError && <div className="text-sm text-red-400">{kalshiError}</div>}

          <div className="flex items-center gap-2">
            <button
              onClick={async () => {
                if (!onKalshiSaveProfile) return;
                const accessKey = kalshiAccessKey.trim();
                const privateKey = kalshiPrivateKey.trim();
                if (!accessKey || !privateKey) {
                  setKalshiError('Access Key ID and Private Key are required.');
                  return;
                }
                const name = kalshiProfileName.trim() || accessKey;
                onKalshiSaveProfile({
                  id: selectedKalshiProfileId || undefined,
                  name,
                  accessKey,
                  privateKey,
                });
                setKalshiProfileName(name);
                setKalshiError(null);
              }}
              disabled={!onKalshiSaveProfile || kalshiSaving}
              className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
            >
              Save Profile
            </button>
            <button
              onClick={async () => {
                if (!onKalshiReconnect) return;
                setKalshiError(null);
                setKalshiSaving(true);
                try {
                  const accessKey = kalshiAccessKey.trim();
                  const privateKey = kalshiPrivateKey.trim();
                  if (!accessKey || !privateKey) {
                    throw new Error('Access Key ID and Private Key are required.');
                  }
                  const name = kalshiProfileName.trim() || accessKey;
                  await onKalshiReconnect({
                    id: selectedKalshiProfileId || undefined,
                    name,
                    accessKey,
                    privateKey,
                    remember: true,
                  });
                  setKalshiProfileName(name);
                } catch (err) {
                  setKalshiError(err instanceof Error ? err.message : String(err));
                } finally {
                  setKalshiSaving(false);
                }
              }}
              disabled={!onKalshiReconnect || kalshiSaving}
              className="bg-primary text-primary-foreground hover:bg-primary/90 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
            >
              {kalshiSaving ? 'Reconnecting…' : 'Save + Reconnect'}
            </button>
            <button
              onClick={async () => {
                if (!onKalshiReconnect) return;
                setKalshiError(null);
                setKalshiSaving(true);
                try {
                  const accessKey = kalshiAccessKey.trim();
                  const privateKey = kalshiPrivateKey.trim();
                  if (!accessKey || !privateKey) {
                    throw new Error('Access Key ID and Private Key are required.');
                  }
                  await onKalshiReconnect({
                    accessKey,
                    privateKey,
                    remember: false,
                  });
                } catch (err) {
                  setKalshiError(err instanceof Error ? err.message : String(err));
                } finally {
                  setKalshiSaving(false);
                }
              }}
              disabled={!onKalshiReconnect || kalshiSaving}
              className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
            >
              Reconnect (Session Only)
            </button>
            {selectedKalshiProfileId && (
              <button
                onClick={() => {
                  onKalshiDeleteProfile?.(selectedKalshiProfileId);
                  setSelectedKalshiProfileId('');
                  setKalshiProfileName(kalshiAccessKey.trim());
                  setKalshiError(null);
                }}
                disabled={!onKalshiDeleteProfile || kalshiSaving}
                className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
              >
                Delete Profile
              </button>
            )}
          </div>
        </div>
      </div>

      {/* Display Preferences */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">Display Preferences</h3>
        <div className="space-y-4">
          {/* POV Setting */}
          <div className="flex items-center justify-between">
            <div>
              <span className="text-sm">Pricing Point of View</span>
              <p className="text-muted-foreground mt-0.5 text-xs">
                Maker shows bid/ask, Taker shows executable prices
              </p>
            </div>
            <button
              onClick={onPOVToggle}
              className="bg-muted hover:bg-muted/80 rounded px-3 py-1.5 text-xs font-medium transition-colors"
            >
              {pov === 'maker' ? 'Maker' : 'Taker'}
            </button>
          </div>

          {/* Odds Format Setting */}
          <div className="flex items-center justify-between">
            <div>
              <span className="text-sm">Odds Format</span>
              <p className="text-muted-foreground mt-0.5 text-xs">
                How probabilities are displayed
              </p>
            </div>
            <button
              onClick={onOddsFormatToggle}
              className="bg-muted hover:bg-muted/80 rounded px-3 py-1.5 text-xs font-medium uppercase transition-colors"
            >
              {oddsFormat === 'probability'
                ? 'Percent'
                : oddsFormat === 'decimal'
                  ? 'Decimal'
                  : 'American'}
            </button>
          </div>

          {/* Data Source Setting */}
          {onDataModeToggle && (
            <div className="flex items-center justify-between">
              <div>
                <span className="text-sm">Data Source</span>
                <p className="text-muted-foreground mt-0.5 text-xs">
                  Smart Relay caches market data server-side for faster loading
                </p>
              </div>
              <button
                onClick={onDataModeToggle}
                className="bg-muted hover:bg-muted/80 rounded px-3 py-1.5 text-xs font-medium transition-colors"
              >
                {dataMode === 'smart-relay' ? 'Smart Relay' : 'Standard'}
              </button>
            </div>
          )}
        </div>
      </div>

      {/* Polymarket */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">Polymarket</h3>
        <div className="space-y-3">
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground text-sm">Status</span>
            <span className="text-sm font-medium uppercase">{polyStatus}</span>
          </div>

          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">Private Key (EOA)</label>
            <SecretCredentialField
              value={polyPrivateKey}
              onChange={setPolyPrivateKey}
              placeholder="0x..."
              disabled={polySaving}
            />
          </div>

          <div className="grid grid-cols-2 gap-3">
            <div className="space-y-2">
              <label className="text-muted-foreground text-xs font-medium">Signature Type</label>
              <select
                value={polySignatureType}
                onChange={(e) => setPolySignatureType(Number(e.target.value) as 0 | 1 | 2)}
                className="border-border bg-background h-10 w-full rounded-lg border px-3 text-sm"
                disabled={polySaving}
              >
                <option value={0}>0 (EOA)</option>
                <option value={1}>1 (Magic / Email)</option>
                <option value={2}>2 (Gnosis Safe)</option>
              </select>
            </div>
            <div className="space-y-2">
              <label className="text-muted-foreground text-xs font-medium">
                Funder Address (optional)
              </label>
              <input
                value={polyFunder}
                onChange={(e) => setPolyFunder(e.target.value)}
                className="border-border bg-background h-10 w-full rounded-lg border px-3 font-mono text-sm"
                placeholder="0x..."
                disabled={polySaving}
              />
            </div>
          </div>

          {polyError && <div className="text-sm text-red-400">{polyError}</div>}

          <div className="flex items-center gap-2">
            <button
              onClick={async () => {
                if (!onPolyConnect) return;
                setPolyError(null);
                setPolySaving(true);
                try {
                  await onPolyConnect({
                    privateKey: polyPrivateKey,
                    signatureType: polySignatureType,
                    funder: polyFunder || undefined,
                  });
                } catch (err) {
                  setPolyError(err instanceof Error ? err.message : String(err));
                } finally {
                  setPolySaving(false);
                }
              }}
              disabled={!onPolyConnect || polySaving}
              className="bg-primary text-primary-foreground hover:bg-primary/90 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
            >
              {polySaving ? 'Connecting…' : 'Connect'}
            </button>
            <button
              onClick={() => onPolyDisconnect?.()}
              disabled={!onPolyDisconnect || polySaving}
              className="bg-muted hover:bg-muted/80 rounded px-3 py-2 text-xs font-medium transition-colors disabled:opacity-50"
            >
              Disconnect
            </button>
          </div>
          <p className="text-muted-foreground text-xs">
            This stores your Polymarket API credentials locally in the browser. Use a dedicated key
            and treat it as sensitive.
          </p>
        </div>
      </div>

      {/* Odds API */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">The Odds API</h3>
        <div className="space-y-3">
          <div className="space-y-2">
            <label className="text-muted-foreground text-xs font-medium">API Key</label>
            <SecretCredentialField
              value={oddsApiKeyValue}
              onChange={(v) => {
                setOddsApiKeyValue(v);
                setOddsApiSaved(false);
              }}
              placeholder="Your the-odds-api.com API key"
            />
          </div>
          <div className="flex items-center gap-2">
            <button
              onClick={() => {
                setOddsApiKey(oddsApiKeyValue);
                setOddsApiSaved(true);
              }}
              className="bg-primary text-primary-foreground hover:bg-primary/90 rounded px-3 py-2 text-xs font-medium transition-colors"
            >
              Save Key
            </button>
            {oddsApiSaved && (
              <span className="text-xs text-green-400">Saved — reload page to fetch new data</span>
            )}
          </div>
          <p className="text-muted-foreground text-xs">
            Powers rotation numbers and external sportsbook odds. Get a key at{' '}
            <a
              href="https://the-odds-api.com"
              target="_blank"
              rel="noopener noreferrer"
              className="text-primary underline"
            >
              the-odds-api.com
            </a>
          </p>
        </div>
      </div>

      {/* Keyboard Shortcuts */}
      <div className="bg-card border-border rounded-lg border p-4">
        <h3 className="mb-4 text-sm font-medium">Keyboard Shortcuts</h3>
        <div className="space-y-2 text-sm">
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground">New Order</span>
            <kbd className="bg-muted rounded px-2 py-0.5 font-mono text-xs">N</kbd>
          </div>
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground">Cancel All Orders</span>
            <kbd className="bg-muted rounded px-2 py-0.5 font-mono text-xs">K</kbd>
          </div>
        </div>
      </div>
    </div>
  );
}
