/**
 * TwoRowSpreads - Table showing spreads with TWO rows per game
 *
 * Columns: Date | Time | Team | Spread | Odds | Liq
 *
 * Format:
 *   Army      -2.5    -114 (53c)
 *   Colgate   +2.5    +118 (46c)
 */

import { cn } from '@/lib/utils';
import { LiquidityBar } from '@/components/atoms/LiquidityBar';
import { Odds } from '@/components/atoms/Odds';
import { DateDisplay } from '@/components/atoms/DateDisplay';
import { Th, Td as BaseTd } from '@/components/atoms/TableCells';
import { SortableTh } from '@/components/atoms/SortableTh';
import { useMemo, useState, Fragment } from 'react';
import { ChevronDown, ChevronUp } from 'lucide-react';
import { liquidityDollars } from '@/lib/nbaConsolidated/pricing';
import { type VolumeWindow, formatWindowLabel } from '@/lib/nbaConsolidated/volumeSettings';
import { getSeriesTicker, type Sport } from '@/lib/sportsDiscovery';
import {
  getBestBid,
  type SpreadGameData,
  type SpreadMarketInfo,
  type SportsLoadingState,
} from '@/lib/sportsStream';
import type { TradeClickRequest } from '@/components/nba-value-dashboard/ConsolidatedMoneylines';
import { LoadingPhaseStatus } from '@/components/nba-value-dashboard/LoadingPhaseStatus';
import { useTableSort, type ValueExtractor } from '@/hooks/useTableSort';
import { formatStartTimePt } from '@/lib/formatters';

/** Table cell with consistent padding */
function Td({
  children,
  className,
  ...rest
}: React.TdHTMLAttributes<HTMLTableCellElement> & { children?: React.ReactNode }) {
  return (
    <BaseTd
      {...rest}
      className={cn('border-0 py-2', className)}
    >
      {children}
    </BaseTd>
  );
}

interface TwoRowSpreadsProps {
  games: SpreadGameData[];
  /** Show "no book" / "n/a" hints for empty data (dev explorer mode) */
  showEmptyHints?: boolean;
  sport: Sport;
  isLoading?: boolean;
  loadingState?: SportsLoadingState;
  maxLiquidity?: number;
  volumeWindows?: VolumeWindow[];
  onTradeClick?: (req: TradeClickRequest) => void;
  /** Navigate to MarketDetailView (primary click). Shift+click still opens trade modal. */
  onNavigateToMarket?: (marketTicker: string) => void;
  className?: string;
}

/** Find main spread (closest to 50c) and return the spread info */
function findMainSpread(game: SpreadGameData): {
  favoriteCode: string;
  floorStrike: number;
  info: SpreadMarketInfo;
} | null {
  let mainSpread: { favoriteCode: string; floorStrike: number; info: SpreadMarketInfo } | null =
    null;
  let minDiff = Infinity;

  for (const [_key, info] of game.markets) {
    const yesBid = getBestBid(info.book, 'yes');
    if (yesBid.priceCents === null) continue;

    const diff = Math.abs(yesBid.priceCents - 50);
    if (diff < minDiff) {
      minDiff = diff;
      mainSpread = {
        favoriteCode: info.teamCode,
        floorStrike: info.floorStrike,
        info,
      };
    }
  }

  return mainSpread;
}

export function TwoRowSpreads({
  games,
  showEmptyHints = false,
  sport,
  isLoading = false,
  loadingState,
  maxLiquidity = 100000,
  volumeWindows = [],
  onTradeClick,
  onNavigateToMarket,
  className,
}: TwoRowSpreadsProps) {
  const [showAllRows, setShowAllRows] = useState(true);
  const maxGames = 10;

  const sportLabel = sport.toUpperCase().replace('-', ' ');
  const seriesTicker = getSeriesTicker(sport, 'spread');

  const extractors = useMemo<Record<string, ValueExtractor<SpreadGameData>>>(
    () => ({
      spread: (g) => {
        const main = findMainSpread(g);
        return main ? main.floorStrike : null;
      },
      kalshi: (g) => {
        const main = findMainSpread(g);
        return main ? getBestBid(main.info.book, 'yes').priceCents : null;
      },
      poly: (g) => {
        const polyMain = g.polymarketMain ?? null;
        return polyMain ? getBestBid(polyMain.markets.favorite ?? null, 'yes').priceCents : null;
      },
      kliq: (g) => {
        const main = findMainSpread(g);
        if (!main) return null;
        const bid = getBestBid(main.info.book, 'yes');
        return liquidityDollars(bid.size, bid.priceCents) ?? null;
      },
      pliq: (g) => {
        const polyMain = g.polymarketMain ?? null;
        if (!polyMain) return null;
        const bid = getBestBid(polyMain.markets.favorite ?? null, 'yes');
        const dollars = liquidityDollars(bid.size, bid.priceCents);
        return dollars && dollars > 0 ? dollars : (polyMain.liquidityUsd ?? 0) / 2 || null;
      },
    }),
    []
  );

  const { sorted, sortKey, sortDir, handleSort } = useTableSort(games, extractors, {
    defaultSortKey: 'time',
    defaultSortDir: 'asc',
  });

  const visibleGames = useMemo(() => {
    if (showAllRows) return sorted;
    return sorted.slice(0, maxGames);
  }, [sorted, showAllRows]);

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header with series ticker */}
      <div className="border-border border-b px-4 py-4">
        <div className="flex items-center justify-between">
          <h2 className="text-foreground text-lg font-semibold">{sportLabel} Spreads</h2>
          <div className="flex items-center gap-3">
            {seriesTicker && (
              <span className="text-muted-foreground font-mono text-xs">{seriesTicker}</span>
            )}
            <LoadingPhaseStatus
              isLoading={isLoading}
              loadingState={loadingState}
            />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full border-collapse border-spacing-0">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <SortableTh
                className="w-24"
                sortKey="date"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Date
              </SortableTh>
              <SortableTh
                className="w-20"
                sortKey="time"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Time PT
              </SortableTh>
              <SortableTh
                className="w-48"
                sortKey="team"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Team
              </SortableTh>
              <SortableTh
                className="w-16 text-right"
                sortKey="spread"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Spread
              </SortableTh>
              <SortableTh
                className="w-20 text-right"
                sortKey="kalshi"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Kalshi
              </SortableTh>
              <SortableTh
                className="w-24"
                sortKey="kliq"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                K Liq
              </SortableTh>
              <SortableTh
                className="w-20 text-right"
                sortKey="poly"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Poly
              </SortableTh>
              <SortableTh
                className="w-24"
                sortKey="pliq"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                P Liq
              </SortableTh>
              {volumeWindows.map((w) => (
                <Th
                  key={`vol-${w}`}
                  className="w-14 text-right"
                >
                  {formatWindowLabel(w)}
                </Th>
              ))}
            </tr>
          </thead>
          <tbody>
            {isLoading && games.length === 0
              ? Array.from({ length: 8 }).map((_, i) => (
                  <tr
                    key={`loading-${i}`}
                    className="animate-pulse"
                  >
                    <Td>
                      <div className="bg-muted/40 h-3 w-16 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-14 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-28 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                    {volumeWindows.map((w) => (
                      <Td
                        key={`loading-vol-${i}-${w}`}
                        className="text-right"
                      >
                        <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                      </Td>
                    ))}
                  </tr>
                ))
              : visibleGames.map((game, gameIndex) => (
                  <GameRows
                    key={game.eventTicker}
                    game={game}
                    maxLiquidity={maxLiquidity}
                    volumeWindows={volumeWindows}
                    gameIndex={gameIndex}
                    showEmptyHints={showEmptyHints}
                    onTradeClick={onTradeClick}
                    onNavigateToMarket={onNavigateToMarket}
                  />
                ))}
          </tbody>
        </table>
      </div>

      {/* Show more/less */}
      {games.length > maxGames && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${games.length}`
                : `Showing ${Math.min(maxGames, games.length)} of ${games.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}

      {/* Empty state */}
      {games.length === 0 && !isLoading && (
        <div className="text-muted-foreground p-8 text-center">No games available</div>
      )}
    </div>
  );
}

interface GameRowsProps {
  game: SpreadGameData;
  maxLiquidity: number;
  volumeWindows: VolumeWindow[];
  gameIndex: number;
  showEmptyHints?: boolean;
  onTradeClick?: (req: TradeClickRequest) => void;
  onNavigateToMarket?: (marketTicker: string) => void;
}

function GameRows({
  game,
  maxLiquidity,
  volumeWindows,
  gameIndex,
  showEmptyHints = false,
  onTradeClick,
  onNavigateToMarket,
}: GameRowsProps) {
  const awayDisplay = game.awayName ?? game.awayCode;
  const homeDisplay = game.homeName ?? game.homeCode;

  const mainSpread = useMemo(() => findMainSpread(game), [game]);
  const polyMain = game.polymarketMain ?? null;

  // Null text for empty venue data
  const kNullText = showEmptyHints && !mainSpread ? 'no book' : '—';
  const pNullText = showEmptyHints && !polyMain ? 'no book' : '—';
  const polyPreferredFavoriteSide = polyMain?.favoriteSide ?? 'away';

  // Determine which team is favored based on mainSpread.favoriteCode
  const isFavoriteAway = mainSpread
    ? mainSpread.favoriteCode.toUpperCase() === game.awayCode.toUpperCase()
    : polyPreferredFavoriteSide === 'away';

  // Get YES/NO bids from the main spread market
  // YES = favorite covers, NO = underdog covers
  const yesBid = getBestBid(mainSpread?.info.book ?? null, 'yes');
  const noBid = getBestBid(mainSpread?.info.book ?? null, 'no');

  // Assign spreads: favorite gets negative, underdog gets positive
  const floorStrike = mainSpread?.floorStrike ?? polyMain?.line ?? 0;
  const awaySpread = isFavoriteAway ? -floorStrike : +floorStrike;
  const homeSpread = isFavoriteAway ? +floorStrike : -floorStrike;

  // Assign odds: favorite uses YES, underdog uses NO
  const awayOdds = isFavoriteAway ? yesBid : noBid;
  const homeOdds = isFavoriteAway ? noBid : yesBid;

  // Optional Polymarket main spread.
  const polyFavoriteBid = getBestBid(polyMain?.markets.favorite ?? null, 'yes');
  const polyUnderdogBid = getBestBid(polyMain?.markets.underdog ?? null, 'yes');
  const awayPolyOdds = isFavoriteAway ? polyFavoriteBid : polyUnderdogBid;
  const homePolyOdds = isFavoriteAway ? polyUnderdogBid : polyFavoriteBid;

  // Calculate liquidity for each side
  const awayLiquidity = liquidityDollars(awayOdds.size, awayOdds.priceCents) ?? 0;
  const homeLiquidity = liquidityDollars(homeOdds.size, homeOdds.priceCents) ?? 0;
  const awayPolyLiqFromBook = liquidityDollars(awayPolyOdds.size, awayPolyOdds.priceCents) ?? 0;
  const homePolyLiqFromBook = liquidityDollars(homePolyOdds.size, homePolyOdds.priceCents) ?? 0;
  const polyHalfLiq = (polyMain?.liquidityUsd ?? 0) / 2;
  const awayPolyLiquidity = awayPolyLiqFromBook > 0 ? awayPolyLiqFromBook : polyHalfLiq;
  const homePolyLiquidity = homePolyLiqFromBook > 0 ? homePolyLiqFromBook : polyHalfLiq;

  // Format spread with sign
  const formatSpread = (spread: number): string => {
    if (spread === 0) return '—';
    const sign = spread > 0 ? '+' : '';
    return `${sign}${spread.toFixed(1)}`;
  };

  // Handle trade click
  // Primary click: navigate to market detail
  // Shift+click: open trade modal (legacy behavior)
  const handleTrade = (
    teamSide: 'away' | 'home',
    venue: 'kalshi' | 'polymarket',
    e: React.MouseEvent
  ) => {
    const isFavorite = teamSide === 'away' ? isFavoriteAway : !isFavoriteAway;

    // Shift+click: open trade modal (legacy behavior)
    if (e.shiftKey && onTradeClick) {
      const teamLabel = teamSide === 'away' ? awayDisplay : homeDisplay;
      const spreadLabel = formatSpread(teamSide === 'away' ? awaySpread : homeSpread);

      if (venue === 'kalshi') {
        if (!mainSpread) return;
        const marketTicker = mainSpread.info.book.marketTicker;
        if (!marketTicker) return;
        const side = isFavorite ? 'yes' : 'no';
        const priceCents = isFavorite ? yesBid.priceCents : noBid.priceCents;
        if (priceCents === null) return;
        onTradeClick({
          venue: 'kalshi',
          marketTicker,
          side,
          priceCents,
          contracts: 100,
          isTakerMode: false,
          teamLabel: `${teamLabel} ${spreadLabel}`,
          gameKey: game.eventTicker,
          gameDate: game.date,
          startTimePt: game.startTimePt,
        });
        return;
      }

      const polyBook = isFavorite ? polyMain?.markets.favorite : polyMain?.markets.underdog;
      const priceCents = teamSide === 'away' ? awayPolyOdds.priceCents : homePolyOdds.priceCents;
      if (!polyBook?.tokenId || !polyBook.marketTicker || priceCents === null) return;

      onTradeClick({
        venue: 'polymarket',
        marketTicker: polyBook.marketTicker,
        tokenId: polyBook.tokenId,
        conditionId: polyBook.conditionId,
        tickSize: polyBook.tickSize,
        negRisk: polyBook.negRisk,
        side: 'yes',
        priceCents,
        contracts: 100,
        isTakerMode: false,
        teamLabel: `${teamLabel} ${spreadLabel}`,
        gameKey: game.eventTicker,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
      return;
    }

    // Primary click: navigate to market detail
    if (onNavigateToMarket) {
      if (venue === 'kalshi' && mainSpread?.info.book.marketTicker) {
        onNavigateToMarket(mainSpread.info.book.marketTicker);
      } else if (venue === 'polymarket') {
        const polyBook = isFavorite ? polyMain?.markets.favorite : polyMain?.markets.underdog;
        if (polyBook?.marketTicker) {
          onNavigateToMarket(polyBook.marketTicker);
        }
      }
    }
  };

  // Alternating row colors (per game, not per row)
  const zebraStyle =
    gameIndex % 2 === 1
      ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
      : undefined;

  const hasClickHandler = onTradeClick || onNavigateToMarket;
  const awayCellClickable = awayOdds.priceCents !== null && hasClickHandler;
  const homeCellClickable = homeOdds.priceCents !== null && hasClickHandler;
  const awayPolyClickable = awayPolyOdds.priceCents !== null && hasClickHandler;
  const homePolyClickable = homePolyOdds.priceCents !== null && hasClickHandler;

  return (
    <Fragment>
      {/* Away team row */}
      <tr
        className="hover:bg-accent/50 transition-colors"
        style={zebraStyle}
      >
        {/* Date */}
        <Td className="text-muted-foreground font-mono text-xs">
          <DateDisplay value={game.date} />
        </Td>

        {/* Time - yellow */}
        <Td className="font-mono text-xs text-yellow-400">{formatStartTimePt(game.startTimePt)}</Td>

        {/* Team - away in cyan */}
        <Td className="font-medium">
          <span className="text-cyan-400">{awayDisplay}</span>
        </Td>

        {/* Spread - yellow */}
        <Td className="text-right font-mono text-yellow-400">
          {floorStrike > 0 ? formatSpread(awaySpread) : '—'}
        </Td>

        {/* Kalshi odds - clickable */}
        <Td
          className={cn(
            'text-right font-mono',
            awayCellClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            awayCellClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('away', 'kalshi', e)
              : undefined
          }
          title={awayCellClickable ? `Click to view market (Shift+click to trade)` : undefined}
        >
          <Odds
            cents={awayOdds.priceCents}
            format="dual"
            nullText={kNullText}
          />
        </Td>

        {/* Kalshi Liquidity */}
        <Td>
          <LiquidityBar
            value={awayLiquidity}
            maxValue={maxLiquidity}
          />
        </Td>

        {/* Polymarket odds - clickable */}
        <Td
          className={cn(
            'text-right font-mono',
            awayPolyClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            awayPolyClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('away', 'polymarket', e)
              : undefined
          }
          title={awayPolyClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={awayPolyOdds.priceCents}
            format="dual"
            nullText={pNullText}
          />
        </Td>

        {/* Polymarket Liquidity */}
        <Td>
          <LiquidityBar
            value={awayPolyLiquidity}
            maxValue={maxLiquidity}
          />
        </Td>

        {/* Volume columns */}
        {volumeWindows.map((w) => (
          <Td
            key={`vol-away-${w}`}
            className="text-muted-foreground text-right font-mono text-xs"
          >
            —
          </Td>
        ))}
      </tr>

      {/* Home team row */}
      <tr
        className="hover:bg-accent/50 transition-colors"
        style={zebraStyle}
      >
        {/* Empty date cell */}
        <Td />

        {/* Empty time cell */}
        <Td />

        {/* Team - home in white */}
        <Td className="font-medium">
          <span>{homeDisplay}</span>
        </Td>

        {/* Spread - yellow */}
        <Td className="text-right font-mono text-yellow-400">
          {floorStrike > 0 ? formatSpread(homeSpread) : '—'}
        </Td>

        {/* Kalshi odds - clickable */}
        <Td
          className={cn(
            'text-right font-mono',
            homeCellClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            homeCellClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('home', 'kalshi', e)
              : undefined
          }
          title={homeCellClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={homeOdds.priceCents}
            format="dual"
            nullText={kNullText}
          />
        </Td>

        {/* Kalshi Liquidity */}
        <Td>
          <LiquidityBar
            value={homeLiquidity}
            maxValue={maxLiquidity}
          />
        </Td>

        {/* Polymarket odds - clickable */}
        <Td
          className={cn(
            'text-right font-mono',
            homePolyClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            homePolyClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('home', 'polymarket', e)
              : undefined
          }
          title={homePolyClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={homePolyOdds.priceCents}
            format="dual"
            nullText={pNullText}
          />
        </Td>

        {/* Polymarket Liquidity */}
        <Td>
          <LiquidityBar
            value={homePolyLiquidity}
            maxValue={maxLiquidity}
          />
        </Td>

        {/* Volume columns */}
        {volumeWindows.map((w) => (
          <Td
            key={`vol-home-${w}`}
            className="text-muted-foreground text-right font-mono text-xs"
          >
            —
          </Td>
        ))}
      </tr>
    </Fragment>
  );
}

export default TwoRowSpreads;
