/**
 * TwoRowTotals - Table showing totals with TWO rows per game
 *
 * Columns: Date | Time | Matchup | Line | Odds | Liq | Volume
 *
 * Format:
 *   Away @ Home Over   6.5   -113 (53c)
 *   Away @ Home Under  6.5   +133 (43c)
 */

import { cn } from '@/lib/utils';
import { LiquidityBar } from '@/components/atoms/LiquidityBar';
import { Odds } from '@/components/atoms/Odds';
import { DateDisplay } from '@/components/atoms/DateDisplay';
import { Th, Td as BaseTd } from '@/components/atoms/TableCells';
import { SortableTh } from '@/components/atoms/SortableTh';
import { Fragment, useMemo, useState } from 'react';
import { ChevronDown, ChevronUp } from 'lucide-react';
import { liquidityDollars } from '@/lib/nbaConsolidated/pricing';
import { type VolumeWindow, formatWindowLabel } from '@/lib/nbaConsolidated/volumeSettings';
import { getSeriesTicker, type Sport } from '@/lib/sportsDiscovery';
import {
  formatTotalLineValue,
  getBestBid,
  type TotalGameData,
  type MarketBook,
  type SportsLoadingState,
} from '@/lib/sportsStream';
import type { TradeClickRequest } from '@/components/nba-value-dashboard/ConsolidatedMoneylines';
import { LoadingPhaseStatus } from '@/components/nba-value-dashboard/LoadingPhaseStatus';
import { useTableSort, type ValueExtractor } from '@/hooks/useTableSort';
import { formatStartTimePt } from '@/lib/formatters';

/** Table cell with consistent padding */
function Td({
  children,
  className,
  ...rest
}: React.TdHTMLAttributes<HTMLTableCellElement> & { children?: React.ReactNode }) {
  return (
    <BaseTd
      {...rest}
      className={cn('border-0 py-2.5', className)}
    >
      {children}
    </BaseTd>
  );
}

interface TwoRowTotalsProps {
  games: TotalGameData[];
  sport: Sport;
  /** Show "no book" / "n/a" hints for empty data (dev explorer mode) */
  showEmptyHints?: boolean;
  isLoading?: boolean;
  loadingState?: SportsLoadingState;
  maxLiquidity?: number;
  volumeWindows?: VolumeWindow[];
  onTradeClick?: (req: TradeClickRequest) => void;
  /** Navigate to MarketDetailView (primary click). Shift+click still opens trade modal. */
  onNavigateToMarket?: (marketTicker: string) => void;
  className?: string;
}

/** Find main strike (closest to 50c) */
function findMainStrike(game: TotalGameData): {
  strike: number;
  book: MarketBook;
} | null {
  let mainStrike: { strike: number; book: MarketBook } | null = null;
  let minDiff = Infinity;

  for (const [strike, book] of game.markets) {
    const yesBid = getBestBid(book, 'yes');
    if (yesBid.priceCents === null) continue;

    const diff = Math.abs(yesBid.priceCents - 50);
    if (diff < minDiff) {
      minDiff = diff;
      mainStrike = { strike, book };
    }
  }

  return mainStrike;
}

export function TwoRowTotals({
  games,
  sport,
  showEmptyHints = false,
  isLoading = false,
  loadingState,
  maxLiquidity = 100000,
  volumeWindows = [],
  onTradeClick,
  onNavigateToMarket,
  className,
}: TwoRowTotalsProps) {
  const [showAllRows, setShowAllRows] = useState(true);
  const maxGames = 10;

  const sportLabel = sport.toUpperCase().replace('-', ' ');
  const seriesTicker = getSeriesTicker(sport, 'total');

  const extractors = useMemo<Record<string, ValueExtractor<TotalGameData>>>(
    () => ({
      line: (g) => {
        const main = findMainStrike(g);
        return main?.strike ?? null;
      },
      kalshi: (g) => {
        const main = findMainStrike(g);
        return main ? getBestBid(main.book, 'yes').priceCents : null;
      },
      poly: (g) => {
        const polyMain = g.polymarketMain ?? null;
        return polyMain ? getBestBid(polyMain.markets.over ?? null, 'yes').priceCents : null;
      },
      kliq: (g) => {
        const main = findMainStrike(g);
        if (!main) return null;
        const bid = getBestBid(main.book, 'yes');
        return liquidityDollars(bid.size, bid.priceCents) ?? null;
      },
      pliq: (g) => {
        const polyMain = g.polymarketMain ?? null;
        if (!polyMain) return null;
        const bid = getBestBid(polyMain.markets.over ?? null, 'yes');
        const dollars = liquidityDollars(bid.size, bid.priceCents);
        return dollars && dollars > 0 ? dollars : (polyMain.liquidityUsd ?? 0) / 2 || null;
      },
    }),
    []
  );

  const { sorted, sortKey, sortDir, handleSort } = useTableSort(games, extractors, {
    defaultSortKey: 'time',
    defaultSortDir: 'asc',
  });

  const visibleGames = useMemo(() => {
    if (showAllRows) return sorted;
    return sorted.slice(0, maxGames);
  }, [sorted, showAllRows]);

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header with series ticker */}
      <div className="border-border border-b px-4 py-4">
        <div className="flex items-center justify-between">
          <h2 className="text-foreground text-lg font-semibold">{sportLabel} Totals</h2>
          <div className="flex items-center gap-3">
            {seriesTicker && (
              <span className="text-muted-foreground font-mono text-xs">{seriesTicker}</span>
            )}
            <LoadingPhaseStatus
              isLoading={isLoading}
              loadingState={loadingState}
            />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full border-collapse border-spacing-0">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <SortableTh
                className="w-24"
                sortKey="date"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Date
              </SortableTh>
              <SortableTh
                className="w-20"
                sortKey="time"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Time PT
              </SortableTh>
              <SortableTh
                className="w-48"
                sortKey="team"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Matchup
              </SortableTh>
              <SortableTh
                className="w-16 text-right"
                sortKey="line"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Line
              </SortableTh>
              <SortableTh
                className="w-20 text-right"
                sortKey="kalshi"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Kalshi
              </SortableTh>
              <SortableTh
                className="w-24"
                sortKey="kliq"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                K Liq
              </SortableTh>
              <SortableTh
                className="w-20 text-right"
                sortKey="poly"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Poly
              </SortableTh>
              <SortableTh
                className="w-24"
                sortKey="pliq"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                P Liq
              </SortableTh>
              {volumeWindows.map((w) => (
                <Th
                  key={`vol-${w}`}
                  className="w-14 text-right"
                >
                  {formatWindowLabel(w)}
                </Th>
              ))}
            </tr>
          </thead>
          <tbody>
            {isLoading && games.length === 0
              ? Array.from({ length: 8 }).map((_, i) => (
                  <tr
                    key={`loading-${i}`}
                    className="animate-pulse"
                  >
                    <Td>
                      <div className="bg-muted/40 h-3 w-16 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-14 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-32 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/30 h-5 w-full rounded" />
                    </Td>
                    {volumeWindows.map((w) => (
                      <Td
                        key={`loading-vol-${i}-${w}`}
                        className="text-right"
                      >
                        <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                      </Td>
                    ))}
                  </tr>
                ))
              : visibleGames.map((game, gameIndex) => (
                  <GameRows
                    key={game.eventTicker}
                    game={game}
                    maxLiquidity={maxLiquidity}
                    volumeWindows={volumeWindows}
                    gameIndex={gameIndex}
                    showEmptyHints={showEmptyHints}
                    onTradeClick={onTradeClick}
                    onNavigateToMarket={onNavigateToMarket}
                  />
                ))}
          </tbody>
        </table>
      </div>

      {/* Show more/less */}
      {games.length > maxGames && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${games.length}`
                : `Showing ${Math.min(maxGames, games.length)} of ${games.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}

      {/* Empty state */}
      {games.length === 0 && !isLoading && (
        <div className="text-muted-foreground p-8 text-center">No games available</div>
      )}
    </div>
  );
}

interface GameRowsProps {
  game: TotalGameData;
  maxLiquidity: number;
  volumeWindows: VolumeWindow[];
  gameIndex: number;
  showEmptyHints?: boolean;
  onTradeClick?: (req: TradeClickRequest) => void;
  onNavigateToMarket?: (marketTicker: string) => void;
}

function GameRows({
  game,
  maxLiquidity,
  volumeWindows,
  gameIndex,
  showEmptyHints = false,
  onTradeClick,
  onNavigateToMarket,
}: GameRowsProps) {
  const awayDisplay = game.awayName ?? game.awayCode;
  const homeDisplay = game.homeName ?? game.homeCode;

  const mainStrike = useMemo(() => findMainStrike(game), [game]);
  const polyMain = game.polymarketMain ?? null;

  // Null text for empty venue data
  const kNullText = showEmptyHints && !mainStrike ? 'no book' : '—';
  const pNullText = showEmptyHints && !polyMain ? 'no book' : '—';
  const displayLine = mainStrike?.strike ?? polyMain?.line ?? null;

  // Over = YES, Under = NO
  const overBid = getBestBid(mainStrike?.book ?? null, 'yes');
  const underBid = getBestBid(mainStrike?.book ?? null, 'no');

  const overDollars = liquidityDollars(overBid.size, overBid.priceCents) ?? 0;
  const underDollars = liquidityDollars(underBid.size, underBid.priceCents) ?? 0;
  const polyOverBid = getBestBid(polyMain?.markets.over ?? null, 'yes');
  const polyUnderBid = getBestBid(polyMain?.markets.under ?? null, 'yes');
  const polyOverDollars = liquidityDollars(polyOverBid.size, polyOverBid.priceCents) ?? 0;
  const polyUnderDollars = liquidityDollars(polyUnderBid.size, polyUnderBid.priceCents) ?? 0;
  const polyHalfLiq = (polyMain?.liquidityUsd ?? 0) / 2;

  const lineDisplay = formatTotalLineValue(displayLine);

  // Handle trade click
  // Primary click: navigate to market detail
  // Shift+click: open trade modal (legacy behavior)
  const handleTrade = (
    selection: 'over' | 'under',
    venue: 'kalshi' | 'polymarket',
    e: React.MouseEvent
  ) => {
    // Shift+click: open trade modal (legacy behavior)
    if (e.shiftKey && onTradeClick) {
      const matchup = `${awayDisplay} vs ${homeDisplay}`;
      const formattedLine = formatTotalLineValue(displayLine);

      if (venue === 'kalshi') {
        if (!mainStrike) return;
        const marketTicker = mainStrike.book.marketTicker;
        if (!marketTicker) return;
        const tradeSide = selection === 'over' ? 'yes' : 'no';
        const priceCents = selection === 'over' ? overBid.priceCents : underBid.priceCents;
        if (priceCents === null) return;
        onTradeClick({
          venue: 'kalshi',
          marketTicker,
          side: tradeSide,
          priceCents,
          contracts: 100,
          isTakerMode: false,
          teamLabel: `${matchup} ${selection === 'over' ? 'Over' : 'Under'} ${formattedLine}`,
          gameKey: game.eventTicker,
          gameDate: game.date,
          startTimePt: game.startTimePt,
        });
        return;
      }

      const polyBook = selection === 'over' ? polyMain?.markets.over : polyMain?.markets.under;
      const priceCents = selection === 'over' ? polyOverBid.priceCents : polyUnderBid.priceCents;
      if (!polyBook?.tokenId || !polyBook.marketTicker || priceCents === null) return;
      onTradeClick({
        venue: 'polymarket',
        marketTicker: polyBook.marketTicker,
        tokenId: polyBook.tokenId,
        conditionId: polyBook.conditionId,
        tickSize: polyBook.tickSize,
        negRisk: polyBook.negRisk,
        side: 'yes',
        priceCents,
        contracts: 100,
        isTakerMode: false,
        teamLabel: `${matchup} ${selection === 'over' ? 'Over' : 'Under'} ${formattedLine}`,
        gameKey: game.eventTicker,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
      return;
    }

    // Primary click: navigate to market detail
    if (onNavigateToMarket) {
      if (venue === 'kalshi' && mainStrike?.book.marketTicker) {
        onNavigateToMarket(mainStrike.book.marketTicker);
      } else if (venue === 'polymarket') {
        const polyBook = selection === 'over' ? polyMain?.markets.over : polyMain?.markets.under;
        if (polyBook?.marketTicker) {
          onNavigateToMarket(polyBook.marketTicker);
        }
      }
    }
  };

  // Alternating row colors
  const zebraStyle =
    gameIndex % 2 === 1
      ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
      : undefined;

  const hasClickHandler = onTradeClick || onNavigateToMarket;
  const overClickable = overBid.priceCents !== null && hasClickHandler;
  const underClickable = underBid.priceCents !== null && hasClickHandler;
  const overPolyClickable = polyOverBid.priceCents !== null && hasClickHandler;
  const underPolyClickable = polyUnderBid.priceCents !== null && hasClickHandler;

  return (
    <Fragment>
      {/* Over row */}
      <tr
        className="hover:bg-accent/50 transition-colors"
        style={zebraStyle}
      >
        <Td className="text-muted-foreground font-mono text-xs">
          <DateDisplay value={game.date} />
        </Td>

        <Td className="font-mono text-xs text-yellow-400">{formatStartTimePt(game.startTimePt)}</Td>

        <Td className="font-medium">
          <span className="text-cyan-400">{awayDisplay}</span>
          <span className="text-muted-foreground px-1">@</span>
          <span>{homeDisplay}</span>
          <span className="pl-2 text-green-400">Over</span>
        </Td>

        <Td className="text-right font-mono text-yellow-400">{lineDisplay}</Td>

        <Td
          className={cn('text-right font-mono', overClickable && 'cursor-pointer hover:underline')}
          onClick={
            overClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('over', 'kalshi', e)
              : undefined
          }
          title={overClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={overBid.priceCents}
            format="dual"
            nullText={kNullText}
          />
        </Td>

        <Td>
          <LiquidityBar
            value={overDollars}
            maxValue={maxLiquidity}
          />
        </Td>

        <Td
          className={cn(
            'text-right font-mono',
            overPolyClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            overPolyClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('over', 'polymarket', e)
              : undefined
          }
          title={overPolyClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={polyOverBid.priceCents}
            format="dual"
            nullText={pNullText}
          />
        </Td>

        <Td>
          <LiquidityBar
            value={polyOverDollars > 0 ? polyOverDollars : polyHalfLiq}
            maxValue={maxLiquidity}
          />
        </Td>

        {volumeWindows.map((w) => (
          <Td
            key={`vol-over-${w}`}
            className="text-muted-foreground text-right font-mono text-xs"
          >
            —
          </Td>
        ))}
      </tr>

      {/* Under row */}
      <tr
        className="hover:bg-accent/50 transition-colors"
        style={zebraStyle}
      >
        <Td />
        <Td />

        <Td className="font-medium">
          <span className="text-cyan-400">{awayDisplay}</span>
          <span className="text-muted-foreground px-1">@</span>
          <span>{homeDisplay}</span>
          <span className="pl-2 text-red-400">Under</span>
        </Td>

        <Td className="text-right font-mono text-yellow-400">{lineDisplay}</Td>

        <Td
          className={cn('text-right font-mono', underClickable && 'cursor-pointer hover:underline')}
          onClick={
            underClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('under', 'kalshi', e)
              : undefined
          }
          title={underClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={underBid.priceCents}
            format="dual"
            nullText={kNullText}
          />
        </Td>

        <Td>
          <LiquidityBar
            value={underDollars}
            maxValue={maxLiquidity}
          />
        </Td>

        <Td
          className={cn(
            'text-right font-mono',
            underPolyClickable && 'cursor-pointer hover:underline'
          )}
          onClick={
            underPolyClickable
              ? (e: React.MouseEvent<HTMLTableCellElement>) => handleTrade('under', 'polymarket', e)
              : undefined
          }
          title={underPolyClickable ? 'Click to view market (Shift+click to trade)' : undefined}
        >
          <Odds
            cents={polyUnderBid.priceCents}
            format="dual"
            nullText={pNullText}
          />
        </Td>

        <Td>
          <LiquidityBar
            value={polyUnderDollars > 0 ? polyUnderDollars : polyHalfLiq}
            maxValue={maxLiquidity}
          />
        </Td>

        {volumeWindows.map((w) => (
          <Td
            key={`vol-under-${w}`}
            className="text-muted-foreground text-right font-mono text-xs"
          >
            —
          </Td>
        ))}
      </tr>
    </Fragment>
  );
}

export default TwoRowTotals;
