import { useState, useEffect, useCallback } from 'react';
import { initPOV, togglePOV, getPOV } from '../lib/pricingPOV';
import { initOddsFormat, cycleOddsFormat, getOddsFormat } from '../lib/oddsConversion';
import { initDataMode, toggleDataMode, getDataMode, type DataMode } from '../lib/dataMode';
import type { PricingPOV, OddsFormat } from '../types';

export interface DisplayPreferences {
  pov: PricingPOV;
  oddsFormat: OddsFormat;
  dataMode: DataMode;
  chartPeriod: '1m' | '5m' | '15m' | '1h' | '1d';
  handlePOVToggle: () => void;
  handleOddsFormatToggle: () => void;
  handleDataModeToggle: () => void;
  handleChartPeriodChange: (period: '1m' | '5m' | '15m' | '1h' | '1d') => void;
}

export function useDisplayPreferences(): DisplayPreferences {
  const [pov, setPov] = useState<PricingPOV>(getPOV());
  const [oddsFormat, setOddsFormat] = useState<OddsFormat>(getOddsFormat());
  const [dataMode, setDataModeState] = useState<DataMode>(getDataMode());
  const [chartPeriod, setChartPeriod] = useState<'1m' | '5m' | '15m' | '1h' | '1d'>('5m');

  // Initialize on mount
  useEffect(() => {
    initPOV();
    initOddsFormat();
    initDataMode();
  }, []);

  const handlePOVToggle = useCallback(() => {
    togglePOV();
    setPov(getPOV());
  }, []);

  const handleOddsFormatToggle = useCallback(() => {
    cycleOddsFormat();
    setOddsFormat(getOddsFormat());
  }, []);

  const handleDataModeToggle = useCallback(() => {
    toggleDataMode();
    setDataModeState(getDataMode());
  }, []);

  const handleChartPeriodChange = useCallback((period: '1m' | '5m' | '15m' | '1h' | '1d') => {
    setChartPeriod(period);
  }, []);

  return {
    pov,
    oddsFormat,
    dataMode,
    chartPeriod,
    handlePOVToggle,
    handleOddsFormatToggle,
    handleDataModeToggle,
    handleChartPeriodChange,
  };
}
