/**
 * Flash effect hooks for live data updates.
 *
 * Provides visual feedback when values change via CSS animations.
 */

import { useEffect, useRef, useState, useCallback } from 'react';

// ============================================================================
// Constants
// ============================================================================

/** Default flash duration in milliseconds - matches CSS animation */
const DEFAULT_DURATION = 600;

/** CSS class names for flash animations */
export const FLASH_CLASS = {
  neutral: 'animate-flash-neutral',
  up: 'animate-flash-up',
  down: 'animate-flash-down',
} as const;

export type FlashVariant = keyof typeof FLASH_CLASS;

// ============================================================================
// Core timeout management hook
// ============================================================================

/**
 * Manages flash state for a set of keys with automatic timeout cleanup.
 * This is the core primitive used by the other hooks.
 */
function useFlashSet<T extends string | number>(duration: number) {
  const [flashing, setFlashing] = useState<Set<T>>(new Set());
  const timeoutsRef = useRef<Map<T, NodeJS.Timeout>>(new Map());

  const addFlash = useCallback(
    (keys: T[]) => {
      if (keys.length === 0) return;

      setFlashing((prev) => {
        const next = new Set(prev);
        keys.forEach((key) => next.add(key));
        return next;
      });

      keys.forEach((key) => {
        // Clear existing timeout if re-flashing same key
        const existing = timeoutsRef.current.get(key);
        if (existing) clearTimeout(existing);

        const timeout = setTimeout(() => {
          setFlashing((prev) => {
            const next = new Set(prev);
            next.delete(key);
            return next;
          });
          timeoutsRef.current.delete(key);
        }, duration);

        timeoutsRef.current.set(key, timeout);
      });
    },
    [duration]
  );

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      timeoutsRef.current.forEach((timeout) => clearTimeout(timeout));
      timeoutsRef.current.clear();
    };
  }, []);

  return { flashing, addFlash };
}

// ============================================================================
// Single value flash hook
// ============================================================================

interface UseFlashOnChangeOptions {
  /** Duration in ms (default: 600) */
  duration?: number;
  /** Track direction for up/down coloring */
  trackDirection?: boolean;
}

/**
 * Returns a CSS class name that triggers a flash animation when value changes.
 *
 * @example
 * const flashClass = useFlashOnChange(price, { trackDirection: true });
 * <div className={cn('p-2 rounded', flashClass)}>{price}¢</div>
 */
export function useFlashOnChange<T>(value: T, options: UseFlashOnChangeOptions = {}): string {
  const { duration = DEFAULT_DURATION, trackDirection = false } = options;

  const prevValueRef = useRef<T>(value);
  const isFirstRender = useRef(true);
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  const [className, setClassName] = useState('');

  useEffect(() => {
    if (isFirstRender.current) {
      isFirstRender.current = false;
      prevValueRef.current = value;
      return;
    }

    if (value === prevValueRef.current) return;

    // Determine variant
    let variant: FlashVariant = 'neutral';
    if (trackDirection && typeof value === 'number' && typeof prevValueRef.current === 'number') {
      variant =
        value > prevValueRef.current ? 'up' : value < prevValueRef.current ? 'down' : 'neutral';
    }

    prevValueRef.current = value;

    if (timeoutRef.current) clearTimeout(timeoutRef.current);

    setClassName(FLASH_CLASS[variant]);

    timeoutRef.current = setTimeout(() => {
      setClassName('');
    }, duration);

    return () => {
      if (timeoutRef.current) clearTimeout(timeoutRef.current);
    };
  }, [value, duration, trackDirection]);

  return className;
}

// ============================================================================
// List flash hooks
// ============================================================================

interface UseFlashListOptions {
  /** Duration in ms (default: 600) */
  duration?: number;
  /** Max items to track to prevent memory leaks (default: 100) */
  maxTracked?: number;
}

/**
 * Tracks new items in a list and returns a Set of IDs that should flash.
 *
 * @example
 * const flashing = useFlashNewItems(trades.map(t => t.id));
 * <div className={flashing.has(trade.id) ? FLASH_CLASS.up : ''}>{trade.price}</div>
 */
export function useFlashNewItems<T extends string | number>(
  itemIds: T[],
  options: UseFlashListOptions = {}
): Set<T> {
  const { duration = DEFAULT_DURATION, maxTracked = 100 } = options;

  const seenRef = useRef<Set<T>>(new Set());
  const isFirstRender = useRef(true);
  const { flashing, addFlash } = useFlashSet<T>(duration);

  useEffect(() => {
    if (isFirstRender.current) {
      isFirstRender.current = false;
      itemIds.forEach((id) => seenRef.current.add(id));
      return;
    }

    const newIds = itemIds.filter((id) => !seenRef.current.has(id));
    newIds.forEach((id) => seenRef.current.add(id));

    addFlash(newIds);

    // Prevent memory leak
    if (seenRef.current.size > maxTracked * 2) {
      seenRef.current = new Set(Array.from(seenRef.current).slice(-maxTracked));
    }
  }, [itemIds, addFlash, maxTracked]);

  return flashing;
}

/**
 * Tracks orderbook level changes and returns a Set of prices that should flash.
 *
 * @example
 * const flashing = useFlashOrderbookChanges(bids);
 * <Level isFlashing={flashing.has(level.price)} />
 */
export function useFlashOrderbookChanges(
  levels: Array<{ price: number; quantity: number }>,
  options: Pick<UseFlashListOptions, 'duration'> = {}
): Set<number> {
  const { duration = DEFAULT_DURATION } = options;

  const prevRef = useRef<Map<number, number>>(new Map());
  const isFirstRender = useRef(true);
  const { flashing, addFlash } = useFlashSet<number>(duration);

  useEffect(() => {
    if (isFirstRender.current) {
      isFirstRender.current = false;
      levels.forEach((l) => prevRef.current.set(l.price, l.quantity));
      return;
    }

    const changed = levels
      .filter((l) => prevRef.current.get(l.price) !== l.quantity)
      .map((l) => l.price);

    // Update tracking
    const currentPrices = new Set(levels.map((l) => l.price));
    prevRef.current.forEach((_, price) => {
      if (!currentPrices.has(price)) prevRef.current.delete(price);
    });
    levels.forEach((l) => prevRef.current.set(l.price, l.quantity));

    addFlash(changed);
  }, [levels, addFlash]);

  return flashing;
}
