import { useState, useEffect, useRef, useCallback } from 'react';
import { createKalshiClient } from '../lib/kalshiApi';
import type { KalshiOrder } from '../lib/kalshiApi';
import { initAuth } from '../lib/kalshiAuth';
import { OrderMonitor } from '../lib/orderMonitor';
import { createUserStream, type UserStream } from '../lib/userStream';
import { PositionTracker } from '../lib/positionTracker';
import type { NBAMarketRow, FillNotification, KalshiCredentials } from '../types';

export interface OrderManagementParams {
  isConnected: boolean;
  apiClientRef: React.MutableRefObject<Awaited<ReturnType<typeof createKalshiClient>> | null>;
  credentials: KalshiCredentials | null;
  positionTrackerRef: React.MutableRefObject<PositionTracker>;
  marketsDataRef: React.MutableRefObject<Map<string, NBAMarketRow>>;
  refreshPositions: () => void;
  loadPositions: () => Promise<void>;
  setErrorMessage: React.Dispatch<React.SetStateAction<string | null>>;
  portfolioPollingEnabled: boolean;
  setPortfolioPollingEnabled: React.Dispatch<React.SetStateAction<boolean>>;
}

export interface OrderManagementReturn {
  orders: KalshiOrder[];
  fillNotifications: FillNotification[];
  showOrderTicket: boolean;
  orderTicketMarket: string | undefined;
  orderTicketPrice: number | undefined;
  showKillAllConfirm: boolean;
  setShowKillAllConfirm: React.Dispatch<React.SetStateAction<boolean>>;
  portfolioRealtimeEnabled: boolean;
  refreshOrders: () => Promise<void>;
  handleFill: (fill: FillNotification) => void;
  handleOrderPlaced: () => Promise<void>;
  handlePlaceKalshiDashboardOrder: (args: {
    marketTicker: string;
    side: 'yes' | 'no';
    priceCents: number;
    contracts: number;
    isTakerMode: boolean;
    expirationTs?: number;
  }) => Promise<void>;
  handleCancelOrder: (orderId: string) => Promise<void>;
  handleDecreaseOrder: (orderId: string, nextQuantity: number) => Promise<void>;
  handleKillAll: () => Promise<void>;
  openOrderTicket: (marketTicker?: string, price?: number) => void;
  closeOrderTicket: () => void;
  cleanupOrders: () => void;
}

export function useOrderManagement({
  isConnected,
  apiClientRef,
  credentials,
  positionTrackerRef,
  marketsDataRef,
  refreshPositions,
  loadPositions,
  setErrorMessage,
  portfolioPollingEnabled,
  setPortfolioPollingEnabled,
}: OrderManagementParams): OrderManagementReturn {
  const [orders, setOrders] = useState<KalshiOrder[]>([]);
  const [fillNotifications, setFillNotifications] = useState<FillNotification[]>([]);
  const [showOrderTicket, setShowOrderTicket] = useState(false);
  const [orderTicketMarket, setOrderTicketMarket] = useState<string | undefined>();
  const [orderTicketPrice, setOrderTicketPrice] = useState<number | undefined>();
  const [showKillAllConfirm, setShowKillAllConfirm] = useState(false);
  const [portfolioRealtimeEnabled, setPortfolioRealtimeEnabled] = useState(false);

  const orderMonitorRef = useRef<OrderMonitor | null>(null);
  const userStreamRef = useRef<UserStream | null>(null);

  const refreshOrders = useCallback(async () => {
    if (!apiClientRef.current) return;
    if (!portfolioPollingEnabled) return;
    try {
      const ordersData = await apiClientRef.current.getOpenOrders();
      setOrders(ordersData);
    } catch (error) {
      const msg = error instanceof Error ? error.message : String(error);
      if (
        msg.includes('API error: 401') ||
        msg.includes('"code":"authentication_error"') ||
        msg.includes('"code": "authentication_error"')
      ) {
        setPortfolioPollingEnabled(false);
        setErrorMessage(
          'Connected to market feed, but portfolio endpoints are unauthorized (401 authentication_error). ' +
            'This usually means you are using the wrong environment (prod vs demo) or your API key is not provisioned for trading/portfolio.'
        );
        return;
      }
      console.error('Failed to refresh orders:', error);
    }
  }, [apiClientRef, portfolioPollingEnabled, setErrorMessage, setPortfolioPollingEnabled]);

  const handleFill = useCallback(
    (fill: FillNotification) => {
      setFillNotifications((prev) => [...prev, fill]);
      setTimeout(() => {
        setFillNotifications((prev) => prev.filter((f) => f.orderId !== fill.orderId));
      }, 5000);

      const market = marketsDataRef.current.get(fill.ticker);
      const currentPrice = market?.awayProb || market?.homeProb || null;
      positionTrackerRef.current.addFill(fill, currentPrice);

      refreshPositions();
      refreshOrders();
    },
    [marketsDataRef, positionTrackerRef, refreshPositions, refreshOrders]
  );

  // Init user stream + order monitor when connected
  useEffect(() => {
    if (!isConnected || !credentials || !apiClientRef.current) return;

    // Already have user stream
    if (userStreamRef.current) return;

    let cancelled = false;
    const apiClient = apiClientRef.current;

    // Load initial data
    refreshOrders();
    loadPositions();

    (async () => {
      try {
        const { privateKey: cryptoKey } = await initAuth(
          credentials.accessKeyId,
          credentials.privateKeyPem
        );

        if (cancelled) return;

        // Start user stream for fills/orders
        try {
          const userStream = createUserStream(true);
          userStreamRef.current = userStream;
          userStream.onFill((fill) => handleFill(fill));
          userStream.onOrderEvent(() => {
            refreshOrders();
          });
          userStream.onError((err) => {
            console.warn('User stream error (will fall back to polling):', err);
            setPortfolioRealtimeEnabled(false);

            if (!orderMonitorRef.current) {
              const fallback = new OrderMonitor();
              fallback.start(
                apiClient,
                {
                  onFill: (fill) => handleFill(fill),
                  onStatusChange: (orderId, oldStatus, newStatus) => {
                    console.log(`Order ${orderId} status: ${oldStatus} → ${newStatus}`);
                    refreshOrders();
                  },
                },
                { pollIntervalMs: 15000 }
              );
              orderMonitorRef.current = fallback;
            }
          });

          await userStream.connect(credentials.accessKeyId, cryptoKey, credentials.environment);
          userStream.subscribe();
          if (!cancelled) {
            setPortfolioRealtimeEnabled(true);
          }
        } catch (err) {
          console.warn('User stream unavailable, falling back to polling:', err);
          if (!cancelled) {
            setPortfolioRealtimeEnabled(false);
          }

          const fallback = new OrderMonitor();
          fallback.start(
            apiClient,
            {
              onFill: (fill) => handleFill(fill),
              onStatusChange: (orderId, oldStatus, newStatus) => {
                console.log(`Order ${orderId} status: ${oldStatus} → ${newStatus}`);
                refreshOrders();
              },
            },
            { pollIntervalMs: 5000 }
          );
          orderMonitorRef.current = fallback;
        }
      } catch (err) {
        console.error('Order management auth init failed:', err);
      }
    })();

    return () => {
      cancelled = true;
    };
  }, [
    isConnected,
    credentials,
    apiClientRef,
    handleFill,
    refreshOrders,
    loadPositions,
    setPortfolioPollingEnabled,
  ]);

  // Polling intervals
  useEffect(() => {
    if (!isConnected) return;
    if (!portfolioPollingEnabled) return;

    const ordersInterval = setInterval(() => {
      refreshOrders();
    }, 5000);

    const positionsIntervalMs = portfolioRealtimeEnabled ? 30000 : 5000;
    const positionsInterval = setInterval(() => {
      refreshPositions();
    }, positionsIntervalMs);

    return () => {
      clearInterval(ordersInterval);
      clearInterval(positionsInterval);
    };
  }, [
    isConnected,
    portfolioPollingEnabled,
    portfolioRealtimeEnabled,
    refreshOrders,
    refreshPositions,
  ]);

  const handleOrderPlaced = useCallback(async () => {
    await refreshOrders();
    if (orderMonitorRef.current) {
      await orderMonitorRef.current.checkImmediate();
    }
  }, [refreshOrders]);

  const handlePlaceKalshiDashboardOrder = useCallback(
    async (args: {
      marketTicker: string;
      side: 'yes' | 'no';
      priceCents: number;
      contracts: number;
      isTakerMode: boolean;
      expirationTs?: number;
    }) => {
      const api = apiClientRef.current;
      if (!api) throw new Error('Not connected');

      await api.placeOrder({
        ticker: args.marketTicker,
        side: args.side,
        action: 'buy',
        type: 'limit',
        count: args.contracts,
        price: args.priceCents,
        post_only: !args.isTakerMode,
        time_in_force: 'good_till_canceled',
        ...(args.expirationTs !== undefined ? { expiration_ts: args.expirationTs } : {}),
      });

      await refreshOrders();
      refreshPositions();
    },
    [apiClientRef, refreshOrders, refreshPositions]
  );

  const handleKillAll = useCallback(async () => {
    if (!apiClientRef.current) return;
    if (!confirm('Are you sure you want to cancel ALL open orders?')) return;

    try {
      await apiClientRef.current.cancelAllOrders();
      await refreshOrders();
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Failed to cancel orders';
      setErrorMessage(message);
    }
  }, [apiClientRef, refreshOrders, setErrorMessage]);

  const handleCancelOrder = useCallback(
    async (orderId: string) => {
      if (!apiClientRef.current) return;
      try {
        await apiClientRef.current.cancelOrder(orderId);
        await refreshOrders();
      } catch (error) {
        const message = error instanceof Error ? error.message : 'Failed to cancel order';
        setErrorMessage(message);
      }
    },
    [apiClientRef, refreshOrders, setErrorMessage]
  );

  const handleDecreaseOrder = useCallback(
    async (orderId: string, nextQuantity: number) => {
      if (!apiClientRef.current) return;
      try {
        await apiClientRef.current.decreaseOrder(orderId, { reduce_to: nextQuantity });
        await refreshOrders();
      } catch (error) {
        const message =
          error instanceof Error ? error.message : 'Failed to decrease order quantity';
        setErrorMessage(message);
        throw error;
      }
    },
    [apiClientRef, refreshOrders, setErrorMessage]
  );

  const openOrderTicket = useCallback((marketTicker?: string, price?: number) => {
    setOrderTicketMarket(marketTicker);
    setOrderTicketPrice(price);
    setShowOrderTicket(true);
  }, []);

  const closeOrderTicket = useCallback(() => {
    setShowOrderTicket(false);
    setOrderTicketMarket(undefined);
    setOrderTicketPrice(undefined);
  }, []);

  const cleanupOrders = useCallback(() => {
    if (userStreamRef.current) {
      userStreamRef.current.disconnect();
      userStreamRef.current = null;
    }
    if (orderMonitorRef.current) {
      orderMonitorRef.current.stop();
      orderMonitorRef.current = null;
    }
    setOrders([]);
    setFillNotifications([]);
    setPortfolioRealtimeEnabled(false);
  }, []);

  return {
    orders,
    fillNotifications,
    showOrderTicket,
    orderTicketMarket,
    orderTicketPrice,
    showKillAllConfirm,
    setShowKillAllConfirm,
    portfolioRealtimeEnabled,
    refreshOrders,
    handleFill,
    handleOrderPlaced,
    handlePlaceKalshiDashboardOrder,
    handleCancelOrder,
    handleDecreaseOrder,
    handleKillAll,
    openOrderTicket,
    closeOrderTicket,
    cleanupOrders,
  };
}
