/**
 * Chart Manager
 *
 * Manages TradingView Lightweight Charts for candlestick visualization.
 * Supports portfolio-level and market-level charts.
 */

import {
  createChart,
  CandlestickSeries,
  IChartApi,
  ISeriesApi,
  CandlestickData,
  Time,
} from 'lightweight-charts';
import type { Candlestick } from '../types';

export interface ChartConfig {
  container: HTMLElement;
  height?: number;
  darkMode?: boolean;
}

export class ChartManager {
  private chart: IChartApi | null = null;
  private candlestickSeries: ISeriesApi<'Candlestick'> | null = null;
  private container: HTMLElement;
  private height: number;
  private darkMode: boolean;

  constructor(config: ChartConfig) {
    this.container = config.container;
    this.height = config.height || 400;
    this.darkMode = config.darkMode ?? true;
  }

  /**
   * Initialize chart
   */
  init(): void {
    if (this.chart) {
      return; // Already initialized
    }

    this.chart = createChart(this.container, {
      width: this.container.clientWidth,
      height: this.height,
      layout: {
        background: { color: this.darkMode ? '#1a1a1a' : '#ffffff' },
        textColor: this.darkMode ? '#d1d4dc' : '#191919',
      },
      grid: {
        vertLines: { color: this.darkMode ? '#2B2B43' : '#e0e0e0' },
        horzLines: { color: this.darkMode ? '#2B2B43' : '#e0e0e0' },
      },
      crosshair: {
        mode: 1, // Normal
      },
      rightPriceScale: {
        borderColor: this.darkMode ? '#485154' : '#d1d4dc',
      },
      timeScale: {
        borderColor: this.darkMode ? '#485154' : '#d1d4dc',
        timeVisible: true,
        secondsVisible: false,
      },
    });

    // Add candlestick series
    this.candlestickSeries = this.chart.addSeries(CandlestickSeries, {
      upColor: '#26a69a',
      downColor: '#ef5350',
      borderVisible: false,
      wickUpColor: '#26a69a',
      wickDownColor: '#ef5350',
    });

    // Handle resize
    window.addEventListener('resize', () => {
      if (this.chart) {
        this.chart.applyOptions({ width: this.container.clientWidth });
      }
    });
  }

  /**
   * Update chart with candlestick data
   */
  updateData(candles: Candlestick[]): void {
    if (!this.candlestickSeries) {
      this.init();
    }

    // Convert our Candlestick format to TradingView format
    const chartData: CandlestickData[] = candles.map((candle) => ({
      time: (candle.timestamp / 1000) as Time, // Convert ms to seconds
      open: candle.open,
      high: candle.high,
      low: candle.low,
      close: candle.close,
    }));

    this.candlestickSeries!.setData(chartData);
  }

  /**
   * Add new candle (for real-time updates)
   */
  addCandle(candle: Candlestick): void {
    if (!this.candlestickSeries) {
      return;
    }

    const chartData: CandlestickData = {
      time: (candle.timestamp / 1000) as Time,
      open: candle.open,
      high: candle.high,
      low: candle.low,
      close: candle.close,
    };

    this.candlestickSeries.update(chartData);
  }

  /**
   * Update last candle (for real-time price updates)
   */
  updateLastCandle(candle: Candlestick): void {
    if (!this.candlestickSeries) {
      return;
    }

    const chartData: CandlestickData = {
      time: (candle.timestamp / 1000) as Time,
      open: candle.open,
      high: candle.high,
      low: candle.low,
      close: candle.close,
    };

    this.candlestickSeries.update(chartData);
  }

  /**
   * Set chart title
   */
  setTitle(_title: string): void {
    // TradingView charts don't have built-in titles, but we can add a custom element
    // For now, this is a placeholder
  }

  /**
   * Destroy chart
   */
  destroy(): void {
    if (this.chart) {
      this.chart.remove();
      this.chart = null;
      this.candlestickSeries = null;
    }
  }

  /**
   * Get chart instance (for advanced operations)
   */
  getChart(): IChartApi | null {
    return this.chart;
  }
}

/**
 * Create portfolio-level chart
 */
export function createPortfolioChart(
  container: HTMLElement,
  darkMode: boolean = true
): ChartManager {
  return new ChartManager({
    container,
    // Prefer the actual container height if it is constrained by CSS.
    height: container.clientHeight || 500,
    darkMode,
  });
}

/**
 * Create market-level chart
 */
export function createMarketChart(container: HTMLElement, darkMode: boolean = true): ChartManager {
  return new ChartManager({
    container,
    height: 400,
    darkMode,
  });
}
