/**
 * Date formatting utilities
 */

/**
 * Returns a human-readable relative time string (e.g., "5m ago", "2h ago")
 */
export function getTimeAgo(date: Date): string {
  const seconds = Math.floor((Date.now() - date.getTime()) / 1000);

  if (seconds < 60) return 'Just now';
  if (seconds < 3600) return `${Math.floor(seconds / 60)}m ago`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)}h ago`;
  return `${Math.floor(seconds / 86400)}d ago`;
}

/**
 * Returns a human-readable relative time with more detail for larger intervals
 */
export function getTimeAgoDetailed(date: Date): string {
  const seconds = Math.floor((Date.now() - date.getTime()) / 1000);

  if (seconds < 60) return 'Just now';
  if (seconds < 3600) return `${Math.floor(seconds / 60)} minutes ago`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)} hours ago`;
  if (seconds < 604800) return `${Math.floor(seconds / 86400)} days ago`;
  return `${Math.floor(seconds / 604800)} weeks ago`;
}

/**
 * Format a date for display in tables/lists
 */
export function formatDate(date: Date): string {
  return date.toLocaleDateString('en-US', {
    month: 'short',
    day: 'numeric',
    year: 'numeric',
  });
}

/**
 * Format a date with time
 */
export function formatDateTime(date: Date): string {
  return date.toLocaleString('en-US', {
    month: 'short',
    day: 'numeric',
    hour: 'numeric',
    minute: '2-digit',
  });
}

/** Get today's date as YYYY-MM-DD in local time */
export function getTodayStr(): string {
  const d = new Date();
  return `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}-${String(d.getDate()).padStart(2, '0')}`;
}

/**
 * Find the nearest available date when the selected date has no games.
 *
 * Returns `null` when no snap is needed (selected date already available,
 * or no dates to snap to).
 *
 * Preference order:
 *   1. Today or nearest future date
 *   2. Most recent past date (if all dates are past)
 */
export function findNearestAvailableDate(
  selectedDate: string,
  availableDates: string[],
  today?: string
): string | null {
  if (availableDates.length === 0) return null;
  if (availableDates.includes(selectedDate)) return null; // Already valid

  const todayStr = today ?? getTodayStr();
  const futureOrToday = availableDates.filter((d) => d >= todayStr);

  if (futureOrToday.length > 0) {
    return futureOrToday[0]!;
  }

  // All dates are past — pick the most recent
  return availableDates[availableDates.length - 1]!;
}
