/**
 * Number and currency formatting utilities
 */

/**
 * Convert cents (0-99) to American odds string
 * 50+ cents = favorite (negative), under 50 = underdog (positive)
 */
export function centsToAmerican(cents: number): string {
  if (cents <= 0 || cents >= 100) return '--';
  if (cents >= 50) {
    const odds = Math.round((-100 * cents) / (100 - cents));
    return odds.toLocaleString();
  }
  const odds = Math.round((100 * (100 - cents)) / cents);
  return `+${odds.toLocaleString()}`;
}

/**
 * Format a number as USD currency
 */
export function formatMoney(
  value: number,
  options?: {
    showSign?: boolean;
    minimumFractionDigits?: number;
    maximumFractionDigits?: number;
  }
): string {
  const { showSign = false, minimumFractionDigits = 2, maximumFractionDigits = 2 } = options ?? {};

  const absValue = Math.abs(value);
  const formatted = new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: 'USD',
    minimumFractionDigits,
    maximumFractionDigits,
  }).format(absValue);

  const sign = showSign && value > 0 ? '+' : '';
  return value < 0 ? `-${formatted}` : `${sign}${formatted}`;
}

/**
 * Format cents as a price display (e.g., 45¢)
 */
export function formatCents(
  cents: number,
  options?: {
    showUnit?: boolean;
  }
): string {
  const { showUnit = true } = options ?? {};
  return showUnit ? `${cents}¢` : String(cents);
}

/**
 * Format volume with K/M/B suffixes
 */
export function formatVolume(volume: number): string {
  if (volume >= 1_000_000_000) {
    return `${(volume / 1_000_000_000).toFixed(1)}B`;
  }
  if (volume >= 1_000_000) {
    return `${(volume / 1_000_000).toFixed(1)}M`;
  }
  if (volume >= 1_000) {
    return `${(volume / 1_000).toFixed(1)}K`;
  }
  return volume.toLocaleString();
}

/**
 * Format a percentage value
 */
export function formatPercent(
  value: number,
  options?: {
    showSign?: boolean;
    decimals?: number;
  }
): string {
  const { showSign = false, decimals = 1 } = options ?? {};
  const sign = showSign && value > 0 ? '+' : '';
  return `${sign}${value.toFixed(decimals)}%`;
}

/**
 * Format a number with commas
 */
export function formatNumber(value: number, decimals: number = 0): string {
  return value.toLocaleString('en-US', {
    minimumFractionDigits: decimals,
    maximumFractionDigits: decimals,
  });
}

/**
 * Format P&L value with sign, commas, and 2 decimal places
 * Example: 1234.56 -> "+1,234.56", -1234.56 -> "-1,234.56"
 */
export function formatPnl(value: number): string {
  const sign = value >= 0 ? '+' : '';
  return `${sign}${value.toLocaleString('en-US', {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  })}`;
}

/**
 * Format currency with compact suffixes (e.g., $1.2k, $5.3M)
 * Use for liquidity displays and large amounts where precision isn't critical.
 */
export function formatCompactCurrency(amount: number): string {
  if (amount >= 1_000_000) {
    return `$${(amount / 1_000_000).toFixed(1)}M`;
  }
  if (amount >= 1_000) {
    return `$${(amount / 1_000).toFixed(1)}K`;
  }
  return `$${amount.toFixed(0)}`;
}

/**
 * Format a probability (0-1) as percentage
 */
export function formatProbability(prob: number | null): string {
  if (prob === null || prob === undefined) return '—';
  return `${(prob * 100).toFixed(1)}%`;
}

/**
 * Format a 24-hour "HH:MM" start time as 12-hour "h:mm AM/PM" for display.
 * The underlying value stays 24-hour for correct sort ordering.
 */
export function formatStartTimePt(time24: string | null | undefined): string {
  if (!time24) return '—';
  const [hh, mm] = time24.split(':');
  const h = parseInt(hh ?? '0', 10);
  const ampm = h >= 12 ? 'PM' : 'AM';
  const h12 = h === 0 ? 12 : h > 12 ? h - 12 : h;
  return `${h12}:${mm} ${ampm}`;
}

/**
 * Format ISO date string as short date (e.g., "Jan 15")
 */
export function formatDate(dateStr: string): string {
  try {
    const date = new Date(dateStr);
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
  } catch {
    return dateStr;
  }
}

/**
 * Format ISO date string as time (e.g., "3:30 PM")
 */
export function formatTime(dateStr: string): string {
  try {
    const date = new Date(dateStr);
    return date.toLocaleTimeString('en-US', {
      hour: 'numeric',
      minute: '2-digit',
      hour12: true,
    });
  } catch {
    return dateStr;
  }
}

/**
 * Format a Date as relative time (e.g., "Just now", "5s ago", "2m ago")
 */
export function formatLastUpdated(date: Date | null): string {
  if (!date) return '';
  const now = new Date();
  const diffMs = now.getTime() - date.getTime();
  const diffSec = Math.floor(diffMs / 1000);
  if (diffSec < 5) return 'Just now';
  if (diffSec < 60) return `${diffSec}s ago`;
  const diffMin = Math.floor(diffSec / 60);
  return `${diffMin}m ago`;
}
