/**
 * HotkeyProvider
 *
 * Provides hotkey configuration context to the app.
 * Loads settings from localStorage on mount and persists changes.
 */

import { createContext, useContext, useState, useEffect, useCallback, type ReactNode } from 'react';
import {
  type HotkeyConfig,
  type HotkeyMapping,
  type HotkeyId,
  DEFAULT_HOTKEYS,
  HOTKEY_STORAGE_KEY,
  hotkeyConfigsEqual,
} from './types';

/** Stored format in localStorage */
interface StoredHotkeys {
  version: number;
  mappings: Record<string, HotkeyConfig>;
}

const CURRENT_VERSION = 1;

interface HotkeyContextValue {
  /** All hotkey mappings */
  mappings: Map<HotkeyId, HotkeyMapping>;
  /** Get a specific hotkey config by ID */
  getHotkey: (id: HotkeyId) => HotkeyConfig | null;
  /** Update a hotkey mapping */
  updateHotkey: (id: HotkeyId, config: HotkeyConfig) => void;
  /** Reset a single hotkey to default */
  resetHotkey: (id: HotkeyId) => void;
  /** Reset all hotkeys to defaults */
  resetAll: () => void;
  /** Whether hotkeys have loaded from storage */
  isReady: boolean;
}

const HotkeyContext = createContext<HotkeyContextValue | null>(null);

/**
 * Load hotkeys from localStorage
 */
function loadHotkeys(): Map<HotkeyId, HotkeyMapping> {
  const mappings = new Map<HotkeyId, HotkeyMapping>();

  // Start with defaults
  for (const [id, def] of Object.entries(DEFAULT_HOTKEYS)) {
    mappings.set(id as HotkeyId, {
      ...def,
      currentConfig: { ...def.defaultConfig },
    });
  }

  // Load stored overrides
  try {
    const stored = localStorage.getItem(HOTKEY_STORAGE_KEY);
    if (stored) {
      const data: StoredHotkeys = JSON.parse(stored);
      if (data.version === CURRENT_VERSION && data.mappings) {
        for (const [id, config] of Object.entries(data.mappings)) {
          const mapping = mappings.get(id as HotkeyId);
          if (mapping) {
            mapping.currentConfig = config;
          }
        }
      }
    }
  } catch (err) {
    console.warn('Failed to load hotkey settings:', err);
  }

  return mappings;
}

/**
 * Save hotkeys to localStorage (only non-default configs)
 */
function saveHotkeys(mappings: Map<HotkeyId, HotkeyMapping>): void {
  const toStore: Record<string, HotkeyConfig> = {};

  for (const [id, mapping] of mappings) {
    // Only store if different from default
    if (!hotkeyConfigsEqual(mapping.currentConfig, mapping.defaultConfig)) {
      toStore[id] = mapping.currentConfig;
    }
  }

  const data: StoredHotkeys = {
    version: CURRENT_VERSION,
    mappings: toStore,
  };

  try {
    localStorage.setItem(HOTKEY_STORAGE_KEY, JSON.stringify(data));
  } catch (err) {
    console.warn('Failed to save hotkey settings:', err);
  }
}

interface HotkeyProviderProps {
  children: ReactNode;
}

export function HotkeyProvider({ children }: HotkeyProviderProps) {
  const [mappings, setMappings] = useState<Map<HotkeyId, HotkeyMapping>>(new Map());
  const [isReady, setIsReady] = useState(false);

  // Load hotkeys on mount
  useEffect(() => {
    const loaded = loadHotkeys();
    setMappings(loaded);
    setIsReady(true);
  }, []);

  const getHotkey = useCallback(
    (id: HotkeyId): HotkeyConfig | null => {
      const mapping = mappings.get(id);
      return mapping?.currentConfig ?? null;
    },
    [mappings]
  );

  const updateHotkey = useCallback((id: HotkeyId, config: HotkeyConfig) => {
    setMappings((prev) => {
      const next = new Map(prev);
      const mapping = next.get(id);
      if (mapping) {
        next.set(id, { ...mapping, currentConfig: config });
        saveHotkeys(next);
      }
      return next;
    });
  }, []);

  const resetHotkey = useCallback((id: HotkeyId) => {
    setMappings((prev) => {
      const next = new Map(prev);
      const mapping = next.get(id);
      if (mapping) {
        next.set(id, { ...mapping, currentConfig: { ...mapping.defaultConfig } });
        saveHotkeys(next);
      }
      return next;
    });
  }, []);

  const resetAll = useCallback(() => {
    setMappings((prev) => {
      const next = new Map(prev);
      for (const [id, mapping] of next) {
        next.set(id, { ...mapping, currentConfig: { ...mapping.defaultConfig } });
      }
      saveHotkeys(next);
      return next;
    });
  }, []);

  const value: HotkeyContextValue = {
    mappings,
    getHotkey,
    updateHotkey,
    resetHotkey,
    resetAll,
    isReady,
  };

  return <HotkeyContext.Provider value={value}>{children}</HotkeyContext.Provider>;
}

/**
 * Hook to access hotkey context
 */
export function useHotkeyContext(): HotkeyContextValue {
  const context = useContext(HotkeyContext);
  if (!context) {
    throw new Error('useHotkeyContext must be used within a HotkeyProvider');
  }
  return context;
}
