/**
 * Kalshi API Authentication using WebCrypto
 * Implements RSA-PSS signing for request authentication
 * Automatically converts PKCS#1 keys to PKCS#8 format
 */

import type { Environment } from '../types';
import * as forge from 'node-forge';

// Kalshi Trade API base URL (per official docs)
// Production: https://api.elections.kalshi.com
const BASE_URLS: Record<Environment, string> = {
  prod: 'https://api.elections.kalshi.com',
  // Demo environment (per Kalshi docs)
  demo: 'https://demo-api.kalshi.co',
};

/**
 * Convert PKCS#1 RSA private key to PKCS#8 format
 */
function convertPkcs1ToPkcs8(pkcs1Pem: string): string {
  try {
    console.log('Converting PKCS#1 key to PKCS#8 format...');

    // Parse PKCS#1 PEM private key using node-forge
    const privateKey = forge.pki.privateKeyFromPem(pkcs1Pem);

    // Convert the RSA private key into ASN.1 RSAPrivateKey
    const rsaPrivate = forge.pki.privateKeyToAsn1(privateKey);

    // Wrap RSAPrivateKey DER in a PKCS#8 PrivateKeyInfo
    const privateKeyInfo = forge.pki.wrapRsaPrivateKey(rsaPrivate);

    // Convert to PKCS#8 PEM
    const pkcs8Pem = forge.pki.privateKeyInfoToPem(privateKeyInfo);

    console.log('Successfully converted PKCS#1 to PKCS#8 format');
    return pkcs8Pem;
  } catch (error) {
    const errorMsg = error instanceof Error ? error.message : 'Unknown error';
    throw new Error(`Failed to convert PKCS#1 to PKCS#8: ${errorMsg}`);
  }
}

/**
 * Parse PEM private key to ArrayBuffer
 * Supports multiple PEM formats: BEGIN PRIVATE KEY, BEGIN RSA PRIVATE KEY, etc.
 * Automatically converts PKCS#1 to PKCS#8 if needed
 */
export async function parsePemPrivateKey(pem: string): Promise<ArrayBuffer> {
  // Normalize: trim, remove carriage returns, handle copy-paste issues
  let cleaned = pem
    .replace(/\r\n/g, '\n') // Normalize line endings
    .replace(/\r/g, '\n') // Handle old Mac line endings
    .trim();

  // Remove any leading/trailing whitespace from each line
  cleaned = cleaned
    .split('\n')
    .map((line) => line.trim())
    .join('\n');

  // Detect key format
  const _isPkcs8 = /-----BEGIN\s+PRIVATE\s+KEY-----/i.test(cleaned);
  const isPkcs1 = /-----BEGIN\s+RSA\s+PRIVATE\s+KEY-----/i.test(cleaned);
  const _isEC = /-----BEGIN\s+EC\s+PRIVATE\s+KEY-----/i.test(cleaned);

  // Common PEM header patterns (case-insensitive)
  const headerPatterns = [
    /-----BEGIN\s+PRIVATE\s+KEY-----/i,
    /-----BEGIN\s+RSA\s+PRIVATE\s+KEY-----/i,
    /-----BEGIN\s+EC\s+PRIVATE\s+KEY-----/i,
  ];

  const footerPatterns = [
    /-----END\s+PRIVATE\s+KEY-----/i,
    /-----END\s+RSA\s+PRIVATE\s+KEY-----/i,
    /-----END\s+EC\s+PRIVATE\s+KEY-----/i,
  ];

  // Check if any header pattern matches
  const hasHeader = headerPatterns.some((pattern) => pattern.test(cleaned));
  if (!hasHeader) {
    // Provide more helpful error message with debugging info
    const firstLine = cleaned.split('\n')[0].substring(0, 80);
    const firstChars = cleaned.substring(0, 100).replace(/\n/g, '\\n');
    throw new Error(
      `Invalid PEM format: missing BEGIN header.\n` +
        `First line: "${firstLine}"\n` +
        `First 100 chars: "${firstChars}"\n` +
        `Expected formats: "-----BEGIN PRIVATE KEY-----" or "-----BEGIN RSA PRIVATE KEY-----"`
    );
  }

  // Auto-convert PKCS#1 to PKCS#8 if needed
  if (isPkcs1) {
    console.log('Detected PKCS#1 format key. Converting to PKCS#8 automatically...');
    cleaned = convertPkcs1ToPkcs8(cleaned);
  }

  // Remove headers and footers (case-insensitive, handle any whitespace)
  for (const pattern of headerPatterns) {
    cleaned = cleaned.replace(pattern, '');
  }
  for (const pattern of footerPatterns) {
    cleaned = cleaned.replace(pattern, '');
  }

  // Remove all whitespace (newlines, spaces, etc.)
  cleaned = cleaned.replace(/\s/g, '');

  if (!cleaned) {
    throw new Error('Invalid PEM format: no key data found after removing headers');
  }

  // Base64 decode
  try {
    const binaryString = atob(cleaned);
    const bytes = new Uint8Array(binaryString.length);
    for (let i = 0; i < binaryString.length; i++) {
      bytes[i] = binaryString.charCodeAt(i);
    }
    return bytes.buffer;
  } catch (error) {
    throw new Error(
      `Failed to decode PEM base64: ${error instanceof Error ? error.message : 'Unknown error'}`
    );
  }
}

/**
 * Import RSA private key from ArrayBuffer
 * Supports both PKCS#8 (BEGIN PRIVATE KEY) and PKCS#1 (BEGIN RSA PRIVATE KEY) formats
 */
export async function importPrivateKey(keyData: ArrayBuffer): Promise<CryptoKey> {
  // Try PKCS#8 format first (BEGIN PRIVATE KEY)
  // WebCrypto API: hash should be a string, not an object
  console.log('importPrivateKey: Attempting to import key, data length:', keyData.byteLength);

  // Validate key data size (RSA keys should be reasonable size)
  if (keyData.byteLength < 100 || keyData.byteLength > 10000) {
    console.warn(
      `Key data size seems unusual: ${keyData.byteLength} bytes. Expected ~1000-2000 bytes for RSA keys.`
    );
  }

  // Check first few bytes to see if it looks like PKCS#8
  const firstBytes = new Uint8Array(keyData.slice(0, Math.min(20, keyData.byteLength)));
  console.log(
    'Key data first bytes (hex):',
    Array.from(firstBytes)
      .map((b) => b.toString(16).padStart(2, '0'))
      .join(' ')
  );

  // PKCS#8 should start with 0x30 (SEQUENCE)
  // PKCS#1 should start with 0x30 0x82 (SEQUENCE with length)
  if (firstBytes[0] !== 0x30) {
    console.warn(
      'Key data does not start with 0x30 (SEQUENCE). This might not be a valid DER-encoded key.'
    );
  }

  try {
    const key = await crypto.subtle.importKey(
      'pkcs8',
      keyData,
      {
        name: 'RSA-PSS',
        hash: 'SHA-256', // String, not object
      },
      false,
      ['sign']
    );
    console.log('importPrivateKey: Successfully imported key');
    return key;
  } catch (pkcs8Error) {
    // If PKCS#8 fails, the key might be PKCS#1 format (BEGIN RSA PRIVATE KEY)
    // WebCrypto doesn't directly support PKCS#1, so provide helpful error
    console.error('PKCS#8 import failed:', pkcs8Error);

    // Extract error message - handle DOMException and Error objects
    let errorMsg = 'Unknown error';
    if (pkcs8Error instanceof Error) {
      errorMsg = pkcs8Error.message || pkcs8Error.name || 'Unknown error';
    } else if (pkcs8Error instanceof DOMException) {
      errorMsg = pkcs8Error.message || pkcs8Error.name || 'DOMException';
    } else if (typeof pkcs8Error === 'string') {
      errorMsg = pkcs8Error;
    } else {
      errorMsg = String(pkcs8Error);
    }

    // Log full error details for debugging
    console.error('Error details:', {
      name: pkcs8Error instanceof Error ? pkcs8Error.name : 'unknown',
      message: errorMsg,
      error: pkcs8Error,
    });

    // Check if it's a format issue (DataError is common for wrong format)
    const lowerMsg = errorMsg.toLowerCase();
    const isDataError = pkcs8Error instanceof DOMException && pkcs8Error.name === 'DataError';

    if (
      isDataError ||
      lowerMsg.includes('pkcs8') ||
      lowerMsg.includes('format') ||
      lowerMsg.includes('decode') ||
      lowerMsg.includes('unrecognized') ||
      lowerMsg.includes('not supported') ||
      lowerMsg.includes('data') ||
      lowerMsg.includes('invalid')
    ) {
      let helpfulMessage = `Failed to import private key. `;

      if (isDataError) {
        helpfulMessage += `The key data format is invalid (DataError). `;
      }

      helpfulMessage += `WebCrypto requires PKCS#8 format (-----BEGIN PRIVATE KEY-----). `;
      helpfulMessage += `If your key is PKCS#1 format (-----BEGIN RSA PRIVATE KEY-----), `;
      helpfulMessage += `you need to convert it using:\n\n`;
      helpfulMessage += `openssl pkcs8 -topk8 -inform PEM -outform PEM -nocrypt -in key.pem -out key-pkcs8.pem\n\n`;
      helpfulMessage += `Or if you're on Windows without OpenSSL, you can use an online converter or install OpenSSL.\n`;
      helpfulMessage += `Original error: ${errorMsg}`;

      throw new Error(helpfulMessage);
    }

    throw new Error(`Failed to import private key: ${errorMsg}`);
  }
}

/**
 * Build canonical request string for signing
 * Format: METHOD\nPATH\nQUERY\nBODY
 *
 * TODO: Verify exact format with Kalshi documentation
 * This may need adjustment based on actual API requirements
 */
export function canonicalRequest(
  method: string,
  path: string,
  query: string = '',
  body: string = ''
): string {
  return `${method}\n${path}\n${query}\n${body}`;
}

/**
 * Sign a request using RSA-PSS
 */
export async function signRequest(privateKey: CryptoKey, payload: string): Promise<ArrayBuffer> {
  const encoder = new TextEncoder();
  const data = encoder.encode(payload);

  return await crypto.subtle.sign(
    {
      name: 'RSA-PSS', // Match the algorithm name used in importKey
      saltLength: 32, // Standard for PSS
    },
    privateKey,
    data
  );
}

/**
 * Convert signature ArrayBuffer to base64
 */
export function signatureToBase64(signature: ArrayBuffer): string {
  const bytes = new Uint8Array(signature);
  let binary = '';
  for (let i = 0; i < bytes.length; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  // Kalshi expects standard base64 (not base64url)
  return btoa(binary);
}

/**
 * Build the string that Kalshi expects to be signed.
 * Per Kalshi docs: signature payload is `${timestamp}${METHOD}${path}`.
 *
 * Notes:
 * - timestamp is milliseconds since epoch as a string
 * - method should be uppercase
 * - path should be the URL path (no scheme/host). Docs note to strip query params.
 */
export function kalshiSignPayload(timestampMs: string, method: string, path: string): string {
  const upperMethod = method.toUpperCase();

  // Per docs:
  // - strip query params
  // - ensure leading slash
  // - remove trailing slash (except root "/")
  let normalizedPath = path.split('?')[0].trim();
  if (!normalizedPath.startsWith('/')) {
    normalizedPath = `/${normalizedPath}`;
  }
  if (normalizedPath.length > 1 && normalizedPath.endsWith('/')) {
    normalizedPath = normalizedPath.slice(0, -1);
  }

  return `${timestampMs}${upperMethod}${normalizedPath}`;
}

/**
 * Build Kalshi authentication headers
 */
export async function buildAuthHeaders(
  accessKeyId: string,
  privateKey: CryptoKey,
  method: string,
  path: string,
  _query: string = '',
  _body: string = ''
): Promise<Record<string, string>> {
  const timestamp = Date.now().toString(); // ms
  const payload = kalshiSignPayload(timestamp, method, path);
  const signature = await signRequest(privateKey, payload);
  const signatureBase64 = signatureToBase64(signature);

  return {
    'KALSHI-ACCESS-KEY': accessKeyId,
    'KALSHI-ACCESS-TIMESTAMP': timestamp,
    'KALSHI-ACCESS-SIGNATURE': signatureBase64,
  };
}

/**
 * Initialize authentication: parse PEM and import key
 */
export async function initAuth(
  accessKeyId: string,
  privateKeyPem: string
): Promise<{ accessKeyId: string; privateKey: CryptoKey }> {
  const keyData = await parsePemPrivateKey(privateKeyPem);
  const privateKey = await importPrivateKey(keyData);

  return { accessKeyId, privateKey };
}

/**
 * Get base URL for environment
 */
export function getBaseUrl(environment: Environment): string {
  return BASE_URLS[environment];
}
