/**
 * Generic Kalshi ticker parsing types
 *
 * Supports multiple sports and market types with a unified interface.
 */

/** Supported sports/market categories */
export type TickerCategory =
  // NBA
  | 'nba-game' // KXNBAGAME - NBA game moneylines
  | 'nba-spread' // KXNBASPREAD - NBA spread markets
  | 'nba-total' // KXNBATOTAL - NBA totals (over/under)
  | 'nba-season' // KXNBA - NBA season props (finals, MVP, etc.)
  // NFL
  | 'nfl-game' // KXNFLGAME - NFL game moneylines
  | 'nfl-spread' // KXNFLSPREAD - NFL spread markets
  | 'nfl-total' // KXNFLTOTAL - NFL totals
  | 'nfl-2td' // KXNFL2TD - NFL 2+ touchdowns prop
  | 'nfl-first-td' // KXNFLFIRSTTD - NFL first touchdown prop
  | 'nfl-superbowl' // KXSB - Super Bowl winner
  // NHL
  | 'nhl-game' // KXNHLGAME - NHL game moneylines
  | 'nhl-spread' // KXNHLSPREAD - NHL spread markets
  | 'nhl-total' // KXNHLTOTAL - NHL totals
  // College Basketball
  | 'ncaa-game' // KXNCAA*GAME - College basketball game
  | 'ncaa-spread' // KXNCAA*SPREAD - College basketball spreads
  | 'ncaa-total' // KXNCAA*TOTAL - College basketball totals
  // Tennis
  | 'tennis-match' // KXATPMATCH, KXWTAMATCH - Tennis match moneylines
  // Other
  | 'politics' // KXATTY*, etc. - Political markets
  | 'crypto' // KXBTC*, etc. - Crypto markets
  | 'unknown'; // Fallback for unrecognized tickers

/** Parsed ticker information */
export interface ParsedTicker {
  /** Original ticker string */
  raw: string;

  /** Detected category */
  category: TickerCategory;

  /** Series ticker (e.g., KXNBAGAME, KXNFL2TD) */
  series: string;

  /** Parsed date if available (YYYY-MM-DD format) */
  date: string | null;

  /** Game/event description if available */
  game: GameInfo | null;

  /** Player information if available (for prop bets) */
  player: PlayerInfo | null;

  /** Spread info if this is a spread market */
  spread: SpreadInfo | null;

  /** Total info if this is a total (over/under) market */
  total: TotalInfo | null;

  /** Tennis match info if this is a tennis market */
  tennis: TennisMatchInfo | null;

  /** Market-specific outcome (e.g., team code, strike value) */
  outcome: string | null;

  /** Human-readable short description */
  shortLabel: string;

  /** Human-readable full description */
  fullLabel: string;
}

export interface GameInfo {
  /** Away team code (e.g., SEA, BOS) */
  awayCode: string;
  /** Home team code (e.g., NE, MIA) */
  homeCode: string;
  /** Away team full name if known */
  awayName?: string;
  /** Home team full name if known */
  homeName?: string;
}

export interface PlayerInfo {
  /** Team code (e.g., SEA) */
  teamCode: string;
  /** Player identifier from ticker (e.g., ABARNER) */
  playerId: string;
  /** Jersey number if present */
  jerseyNumber?: number;
  /** Formatted player name if we can derive it */
  displayName?: string;
}

/** Spread market info */
export interface SpreadInfo {
  /** The favored team code */
  favoredTeam: string;
  /** The spread value (positive number, e.g., 5.5 means -5.5 for favorite) */
  spreadValue: number;
  /** Whether this is for the favorite (cover) or underdog (miss) side */
  side: 'cover' | 'miss';
}

/** Total (over/under) market info */
export interface TotalInfo {
  /** The total points line */
  totalValue: number;
  /** Whether this is over or under */
  side: 'over' | 'under';
}

/** Tennis player info */
export interface TennisPlayerInfo {
  /** Player name/code from ticker */
  playerCode: string;
  /** Formatted player name */
  displayName: string;
}

/** Tennis match info */
export interface TennisMatchInfo {
  /** First player */
  player1: TennisPlayerInfo;
  /** Second player */
  player2: TennisPlayerInfo;
  /** Tournament code if present */
  tournament?: string;
}

/** Month abbreviation to number mapping */
export const MONTH_MAP: Record<string, number> = {
  JAN: 1,
  FEB: 2,
  MAR: 3,
  APR: 4,
  MAY: 5,
  JUN: 6,
  JUL: 7,
  AUG: 8,
  SEP: 9,
  OCT: 10,
  NOV: 11,
  DEC: 12,
};

/** Parse date from YYMONDD format (e.g., 26FEB08 -> 2026-02-08) */
export function parseDateFromTicker(dateStr: string): string | null {
  if (dateStr.length < 7) return null;

  const yy = Number(dateStr.slice(0, 2));
  const monStr = dateStr.slice(2, 5).toUpperCase();
  const dd = Number(dateStr.slice(5, 7));
  const mm = MONTH_MAP[monStr];

  if (!Number.isFinite(yy) || !Number.isFinite(dd) || !mm) return null;

  const year = 2000 + yy;
  const month = String(mm).padStart(2, '0');
  const day = String(dd).padStart(2, '0');

  return `${year}-${month}-${day}`;
}

/** Format date for display (MM-DD) */
export function formatDateShort(dateYyyyMmDd: string): string {
  const parts = dateYyyyMmDd.split('-');
  if (parts.length !== 3) return dateYyyyMmDd;
  return `${parts[1]}-${parts[2]}`;
}
