/**
 * Market Data Transformation Utilities
 *
 * Transforms Kalshi API market data into the format needed by UI components
 */

import type { KalshiMarket } from '@/types';
import { probabilityToAmerican } from './oddsConversion';

/**
 * Extracted team/side information from market title
 */
export interface ExtractedSides {
  /** First team mentioned (typically away in "X at Y") */
  teamA: string | null;
  /** Second team mentioned (typically home in "X at Y") */
  teamB: string | null;
  /** Which team YES corresponds to (from yes_sub_title) */
  yesTeam: 'A' | 'B' | null;
  /** Label for YES side - the team name or "YES" */
  yesLabel: string;
  /** Label for NO side - the other team name or "NO" */
  noLabel: string;
  /** Whether we successfully extracted teams */
  hasTeams: boolean;
}

/**
 * Transformed market data for UI components
 */
export interface TransformedMarket {
  ticker: string;
  title: string;
  subtitle?: string;
  yesPrice: number; // cents (0-100)
  noPrice: number; // cents (0-100)
  yesOdds: number; // American odds
  noOdds: number; // American odds
  change24h: number; // percentage change
  volume24h: number; // volume in dollars
  openInterest: number;
  liquidity: number;
  status: 'live' | 'closed' | 'halted' | 'upcoming';
  expiresAt: string; // ISO string
  expiresFormatted: string; // Human-readable
  /** Extracted team/side labels */
  sides: ExtractedSides;
  // Original data for reference
  raw: KalshiMarket;
}

/**
 * Calculate percentage change from previous price
 */
export function calculatePriceChange(current: number | null, previous: number | null): number {
  if (current === null || previous === null || previous === 0) {
    return 0;
  }
  return ((current - previous) / previous) * 100;
}

/**
 * Convert price in cents (0-100) to probability (0-1)
 */
export function priceToProbability(price: number): number {
  return price / 100;
}

/**
 * Get American odds from price in cents
 */
export function priceToAmericanOdds(price: number): number {
  const prob = priceToProbability(price);
  return probabilityToAmerican(prob);
}

/**
 * Format American odds as string (+138 or -200)
 */
export function formatAmericanOdds(odds: number): string {
  return odds > 0 ? `+${odds}` : `${odds}`;
}

/**
 * Determine market status from timestamps
 */
export function getMarketStatus(market: KalshiMarket): 'live' | 'closed' | 'halted' | 'upcoming' {
  const now = new Date();
  const openTime = market.open_time ? new Date(market.open_time) : null;
  const closeTime = market.close_time ? new Date(market.close_time) : null;

  if (closeTime && now > closeTime) {
    return 'closed';
  }
  if (openTime && now < openTime) {
    return 'upcoming';
  }
  return 'live';
}

/**
 * Format expiration time for display
 */
export function formatExpiration(dateStr: string | null | undefined): string {
  if (!dateStr) return '--';

  try {
    const date = new Date(dateStr);
    const now = new Date();
    const diffMs = date.getTime() - now.getTime();
    const diffDays = Math.ceil(diffMs / (1000 * 60 * 60 * 24));

    if (diffDays < 0) return 'Expired';
    if (diffDays === 0) return 'Today';
    if (diffDays === 1) return 'Tomorrow';
    if (diffDays < 7) return `${diffDays}d`;

    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
  } catch {
    return '--';
  }
}

/**
 * Transform a Kalshi market into UI-ready format
 */
export function transformMarket(market: KalshiMarket): TransformedMarket {
  const yesPrice = market.last_price ?? market.yes_bid ?? 50;
  const noPrice = 100 - yesPrice;

  return {
    ticker: market.market_ticker,
    title: market.title,
    subtitle: market.subtitle || market.yes_sub_title || undefined,
    yesPrice,
    noPrice,
    yesOdds: priceToAmericanOdds(yesPrice),
    noOdds: priceToAmericanOdds(noPrice),
    change24h: calculatePriceChange(market.last_price, market.previous_price),
    volume24h: market.volume || 0,
    openInterest: market.open_interest || 0,
    liquidity: market.liquidity || 0,
    status: getMarketStatus(market),
    expiresAt: market.close_time || market.expiry_time || '',
    expiresFormatted: formatExpiration(market.close_time || market.expiry_time),
    sides: extractSides(market.title, market.yes_sub_title),
    raw: market,
  };
}

/**
 * Transform an array of Kalshi markets
 */
export function transformMarkets(markets: KalshiMarket[]): TransformedMarket[] {
  return markets.map(transformMarket);
}

/**
 * Sort markets by absolute price change (biggest movers first)
 */
export function sortByChange(markets: TransformedMarket[]): TransformedMarket[] {
  return [...markets].sort((a, b) => Math.abs(b.change24h) - Math.abs(a.change24h));
}

/**
 * Sort markets by volume (highest first)
 */
export function sortByVolume(markets: TransformedMarket[]): TransformedMarket[] {
  return [...markets].sort((a, b) => b.volume24h - a.volume24h);
}

/**
 * Get top gainers (positive change, sorted by change descending)
 */
export function getGainers(markets: TransformedMarket[], limit = 5): TransformedMarket[] {
  return markets
    .filter((m) => m.change24h > 0)
    .sort((a, b) => b.change24h - a.change24h)
    .slice(0, limit);
}

/**
 * Get top losers (negative change, sorted by change ascending)
 */
export function getLosers(markets: TransformedMarket[], limit = 5): TransformedMarket[] {
  return markets
    .filter((m) => m.change24h < 0)
    .sort((a, b) => a.change24h - b.change24h)
    .slice(0, limit);
}

/**
 * Convert TransformedMarket to TopMover format
 */
export function toTopMover(market: TransformedMarket) {
  return {
    ticker: market.ticker,
    title: market.title,
    price: market.yesPrice,
    change: market.change24h,
  };
}

/**
 * Format volume for display (re-exported from formatters for convenience)
 */
export { formatCompactCurrency as formatVolume } from './formatters';

/**
 * Extract team names from market title and determine YES/NO mapping
 *
 * Handles patterns like:
 * - "Auburn at Ole Miss Winner?" with yes_sub_title "Ole Miss"
 * - "Lakers vs Celtics Winner?"
 * - "Will Miami win?"
 *
 * The yes_sub_title field tells us which team YES corresponds to.
 * This is critical because Kalshi creates separate markets for each team.
 */
export function extractSides(title: string, yesSubTitle?: string | null): ExtractedSides {
  const defaultResult: ExtractedSides = {
    teamA: null,
    teamB: null,
    yesTeam: null,
    yesLabel: 'YES',
    noLabel: 'NO',
    hasTeams: false,
  };

  if (!title) return defaultResult;

  // Helper to determine which team YES corresponds to based on yes_sub_title
  const determineYesTeam = (teamA: string, teamB: string | null): 'A' | 'B' | null => {
    if (!yesSubTitle) return null;
    const subLower = yesSubTitle.toLowerCase().trim();
    if (subLower === teamA.toLowerCase() || subLower.startsWith(teamA.toLowerCase())) {
      return 'A';
    }
    if (teamB && (subLower === teamB.toLowerCase() || subLower.startsWith(teamB.toLowerCase()))) {
      return 'B';
    }
    return null;
  };

  // Pattern 1: "Team A at Team B Winner?" or "Team A at Team B"
  const atPattern = /^(.+?)\s+at\s+(.+?)(?:\s+Winner\??|\s*\?)?$/i;
  const atMatch = title.match(atPattern);
  if (atMatch) {
    const teamA = atMatch[1].trim();
    const teamB = atMatch[2].trim();
    const yesTeam = determineYesTeam(teamA, teamB);

    return {
      teamA,
      teamB,
      yesTeam,
      yesLabel: yesTeam === 'A' ? teamA : yesTeam === 'B' ? teamB : 'YES',
      noLabel: yesTeam === 'A' ? teamB : yesTeam === 'B' ? teamA : 'NO',
      hasTeams: true,
    };
  }

  // Pattern 2: "Team A vs Team B Winner?" or "Team A vs Team B"
  const vsPattern = /^(.+?)\s+(?:vs\.?|v\.?)\s+(.+?)(?:\s+Winner\??|\s*\?)?$/i;
  const vsMatch = title.match(vsPattern);
  if (vsMatch) {
    const teamA = vsMatch[1].trim();
    const teamB = vsMatch[2].trim();
    const yesTeam = determineYesTeam(teamA, teamB);

    return {
      teamA,
      teamB,
      yesTeam,
      yesLabel: yesTeam === 'A' ? teamA : yesTeam === 'B' ? teamB : 'YES',
      noLabel: yesTeam === 'A' ? teamB : yesTeam === 'B' ? teamA : 'NO',
      hasTeams: true,
    };
  }

  // Pattern 3: "Will X win?" - single team mentioned
  const willWinPattern = /^Will\s+(.+?)\s+win\??$/i;
  const willMatch = title.match(willWinPattern);
  if (willMatch) {
    const team = willMatch[1].trim();
    return {
      teamA: team,
      teamB: null,
      yesTeam: 'A',
      yesLabel: team,
      noLabel: 'NO',
      hasTeams: true,
    };
  }

  // Pattern 4: Use yes_sub_title directly if no title pattern matched
  if (yesSubTitle) {
    return {
      teamA: null,
      teamB: null,
      yesTeam: null,
      yesLabel: yesSubTitle,
      noLabel: 'NO',
      hasTeams: false,
    };
  }

  return defaultResult;
}
