import { describe, expect, it } from 'vitest';
import type { VenueGameBooks } from './types';
import {
  applyTakerFee,
  getTeamWinData,
  liquidityDollars,
  next3NoLadders,
  next3YesLadders,
} from './pricing';

/**
 * Inline fixture data matching ob_kxnbagame-26jan14waslac.json.
 * Embedded directly so tests don't depend on external files outside the repo.
 *
 * Pricing logic: "team wins" = NO on opponent market.
 *   - Away (WAS) win → look at home (LAC) market NO levels
 *   - Home (LAC) win → look at away (WAS) market NO levels
 */
const WASLAC_FIXTURE: VenueGameBooks = {
  venue: 'kalshi',
  eventId: 'KXNBAGAME-26JAN14WASLAC',
  awayCode: 'WAS',
  homeCode: 'LAC',
  tsMs: 1736899200000,
  markets: {
    away: {
      marketTicker: 'KXNBAGAME-26JAN14WASLAC-WAS',
      yes: [
        { priceCents: 16.753, size: 100000 },
        { priceCents: 16, size: 200 },
      ],
      no: [
        { priceCents: 83.247, size: 130729 },
        { priceCents: 83, size: 500 },
        { priceCents: 82, size: 300 },
      ],
    },
    home: {
      marketTicker: 'KXNBAGAME-26JAN14WASLAC-LAC',
      yes: [
        { priceCents: 84.236, size: 100000 },
        { priceCents: 84, size: 400 },
      ],
      no: [
        { priceCents: 15.224, size: 163703 },
        { priceCents: 15, size: 400 },
        { priceCents: 14, size: 200 },
      ],
    },
  },
};

describe('nbaConsolidated pricing parity', () => {
  it('computes maker-mode team win prices from opponent NO bids', () => {
    const game = WASLAC_FIXTURE;

    // Away (WAS) win = NO on opponent (home=LAC) market → LAC.no best bid = 15.224
    const awayWin = getTeamWinData({ game, venue: 'kalshi', teamSide: 'away', isTakerMode: false });
    // Home (LAC) win = NO on opponent (away=WAS) market → WAS.no best bid = 83.247
    const homeWin = getTeamWinData({ game, venue: 'kalshi', teamSide: 'home', isTakerMode: false });

    expect(awayWin.marketTicker).toBe('KXNBAGAME-26JAN14WASLAC-LAC');
    expect(homeWin.marketTicker).toBe('KXNBAGAME-26JAN14WASLAC-WAS');

    expect(awayWin.rawBidPriceCents).toBeCloseTo(15.224, 3);
    expect(awayWin.priceCents).toBeCloseTo(15.224, 3);
    expect(awayWin.liq).toBe(163703);

    expect(homeWin.rawBidPriceCents).toBeCloseTo(83.247, 3);
    expect(homeWin.priceCents).toBeCloseTo(83.247, 3);
    expect(homeWin.liq).toBe(130729);
  });

  it('computes taker-mode team win prices from opponent YES bids (NO ask + fee)', () => {
    const game = WASLAC_FIXTURE;

    // Taker mode: opponent YES best bid → LAC.yes[0] = 84.236
    const awayWin = getTeamWinData({ game, venue: 'kalshi', teamSide: 'away', isTakerMode: true });
    expect(awayWin.rawBidPriceCents).toBeCloseTo(84.236, 3);

    const noAsk = 100 - 84.236;
    const expected = applyTakerFee(noAsk, 0.07);
    expect(awayWin.priceCents).toBeCloseTo(expected ?? 0, 3);
  });

  it('liquidityDollars = contracts * price / 100', () => {
    expect(liquidityDollars(1000, 25)).toBe(250);
    expect(liquidityDollars(null, 25)).toBeNull();
    expect(liquidityDollars(10, null)).toBeNull();
  });

  it('next3 ladder selection matches descending below best', () => {
    const levels = [
      { priceCents: 50, size: 1 },
      { priceCents: 40, size: 1 },
      { priceCents: 30, size: 1 },
      { priceCents: 20, size: 1 },
      { priceCents: 10, size: 1 },
    ];
    expect(next3NoLadders(levels, 50).map((x) => x.priceCents)).toEqual([40, 30, 20]);
    expect(next3YesLadders(levels, 40).map((x) => x.priceCents)).toEqual([30, 20, 10]);
  });
});
