/**
 * Volume Aggregator
 *
 * Fetches market trades and aggregates volume by configurable time windows.
 */

import type { KalshiApiClient, KalshiTrade } from '../kalshiApi';
import type { VolumeWindow } from './volumeSettings';

export interface VolumeByWindow {
  /** Time window in minutes */
  window: VolumeWindow;
  /** Volume in contracts */
  contracts: number;
  /** Volume in dollars (contracts * price / 100) */
  dollars: number;
}

export interface MarketVolume {
  ticker: string;
  volumes: VolumeByWindow[];
  lastUpdated: number;
}

// Cache for fetched trades to avoid re-fetching
const tradeCache: Map<string, { trades: KalshiTrade[]; fetchedAt: number }> = new Map();
const CACHE_TTL_MS = 30_000; // 30 seconds

/**
 * Fetch recent trades for a market (with caching)
 */
async function fetchTrades(api: KalshiApiClient, ticker: string): Promise<KalshiTrade[]> {
  const cached = tradeCache.get(ticker);
  const now = Date.now();

  if (cached && now - cached.fetchedAt < CACHE_TTL_MS) {
    return cached.trades;
  }

  try {
    // Fetch up to 500 recent trades (should cover 60 min window for most markets)
    const trades = await api.getMarketTrades({ ticker, limit: 500 });
    tradeCache.set(ticker, { trades, fetchedAt: now });
    return trades;
  } catch (err) {
    console.warn(`Failed to fetch trades for ${ticker}:`, err);
    return cached?.trades ?? [];
  }
}

/**
 * Aggregate trades into volume buckets by time window
 */
function aggregateVolume(trades: KalshiTrade[], windows: VolumeWindow[]): VolumeByWindow[] {
  const now = Date.now();
  const results: VolumeByWindow[] = [];

  for (const window of windows) {
    const cutoffMs = now - window * 60 * 1000;
    let contracts = 0;
    let dollars = 0;

    for (const trade of trades) {
      const tradeTime = new Date(trade.created_time).getTime();
      if (tradeTime >= cutoffMs) {
        contracts += trade.count;
        // Use yes_price for dollar calculation (price is in cents 0-100)
        dollars += (trade.count * trade.yes_price) / 100;
      }
    }

    results.push({ window, contracts, dollars });
  }

  return results;
}

/**
 * Get volume data for a single market
 */
export async function getMarketVolume(
  api: KalshiApiClient,
  ticker: string,
  windows: VolumeWindow[]
): Promise<MarketVolume> {
  const trades = await fetchTrades(api, ticker);
  const volumes = aggregateVolume(trades, windows);

  return {
    ticker,
    volumes,
    lastUpdated: Date.now(),
  };
}

/**
 * Get volume data for multiple markets in parallel
 */
export async function getMarketsVolume(
  api: KalshiApiClient,
  tickers: string[],
  windows: VolumeWindow[]
): Promise<Map<string, MarketVolume>> {
  const results = new Map<string, MarketVolume>();

  // Fetch in parallel with concurrency limit
  const BATCH_SIZE = 5;
  for (let i = 0; i < tickers.length; i += BATCH_SIZE) {
    const batch = tickers.slice(i, i + BATCH_SIZE);
    const batchResults = await Promise.all(
      batch.map((ticker) => getMarketVolume(api, ticker, windows))
    );
    for (const result of batchResults) {
      results.set(result.ticker, result);
    }
  }

  return results;
}

/**
 * Format volume for display
 */
export function formatVolume(value: number, type: 'contracts' | 'dollars'): string {
  if (type === 'dollars') {
    if (value >= 1000) {
      return `$${(value / 1000).toFixed(1)}K`;
    }
    return `$${value.toFixed(0)}`;
  }

  // Contracts
  if (value >= 1000) {
    return `${(value / 1000).toFixed(1)}K`;
  }
  return value.toFixed(0);
}

/**
 * Clear the trade cache (useful when refreshing)
 */
export function clearVolumeCache(): void {
  tradeCache.clear();
}
