/**
 * Volume Time Window Settings
 *
 * Manages user preferences for which time windows to display in the NBA dashboard.
 * Default windows: 5, 10, 30, 60 minutes
 */

export type VolumeWindow = 5 | 10 | 30 | 60;

export const ALL_VOLUME_WINDOWS: VolumeWindow[] = [5, 10, 30, 60];

const STORAGE_KEY = 'nba_volume_windows';

let enabledWindows: Set<VolumeWindow> = new Set(ALL_VOLUME_WINDOWS);

/**
 * Initialize from localStorage
 */
export function initVolumeSettings(): VolumeWindow[] {
  try {
    const saved = localStorage.getItem(STORAGE_KEY);
    if (saved) {
      const parsed = JSON.parse(saved) as number[];
      const valid = parsed.filter((w): w is VolumeWindow =>
        ALL_VOLUME_WINDOWS.includes(w as VolumeWindow)
      );
      if (valid.length > 0) {
        enabledWindows = new Set(valid);
      }
    }
  } catch {
    // Use defaults
  }
  return getEnabledWindows();
}

/**
 * Get currently enabled windows (sorted)
 */
export function getEnabledWindows(): VolumeWindow[] {
  return ALL_VOLUME_WINDOWS.filter((w) => enabledWindows.has(w));
}

/**
 * Check if a window is enabled
 */
export function isWindowEnabled(window: VolumeWindow): boolean {
  return enabledWindows.has(window);
}

/**
 * Toggle a window on/off
 */
export function toggleWindow(window: VolumeWindow): VolumeWindow[] {
  if (enabledWindows.has(window)) {
    // Don't allow disabling all windows - keep at least one
    if (enabledWindows.size > 1) {
      enabledWindows.delete(window);
    }
  } else {
    enabledWindows.add(window);
  }
  saveSettings();
  return getEnabledWindows();
}

/**
 * Set all enabled windows at once
 */
export function setEnabledWindows(windows: VolumeWindow[]): VolumeWindow[] {
  const valid = windows.filter((w) => ALL_VOLUME_WINDOWS.includes(w));
  if (valid.length > 0) {
    enabledWindows = new Set(valid);
    saveSettings();
  }
  return getEnabledWindows();
}

function saveSettings(): void {
  try {
    localStorage.setItem(STORAGE_KEY, JSON.stringify(getEnabledWindows()));
  } catch {
    // Ignore storage errors
  }
}

/**
 * Format window for display (e.g., "5m", "60m")
 */
export function formatWindowLabel(window: VolumeWindow): string {
  return `${window}m`;
}
